package com.flybits.concierge

import com.flybits.commons.library.exceptions.FlybitsException

/**
 *  Represents Configuration fields for the Flybits Concierge
 *
 */
class ConciergeConfiguration private constructor(builder: Builder) {

    companion object {

        const val KEY_TIME_TO_UPLOAD_CONTEXT = "timeToUploadContext"
        const val DEFAULT_TIME_TO_UPLOAD_CONTEXT = 10
        const val DEFAULT_GATEWAY_URL = "https://v3.flybits.com"
        const val DEFAULT_TNC_URL = "https://www.flybits.com/legal/terms-of-use"
        const val DEFAULT_PRIVACY_POLICY_URL = "https://www.flybits.com/legal/privacy-policy"
    }

    var projectID: String = builder.projectID
        private set

    var tncUrl: String = builder.tncUrl
        private set

    var tncText: String? = builder.tncText
        private set

    var privacyPolicyUrl: String = builder.privacyPolicyUrl
        private set

    var privacyPolicyText: String? = builder.privacyPolicyText
        private set

    var timeToUploadContext: Long = builder.timeToUploadContext
        private set

    var gatewayUrl: String = builder.gatewayUrl
        private set

    var optInTermsOfService: String? = builder.optInTermsOfService
        private set

    var optInTitle: String? = builder.optInTitle
        private set

    init {

        if (projectID == "") {
            throw FlybitsException("Project ID must not be empty")
        }


    }

    /**
     * Builder class for ConciergeConfiguration
     *
     * @param projectID string representing projectID
     */
    class Builder(val projectID: String) {
        var tncUrl: String = DEFAULT_TNC_URL
            private set
        var tncText: String? = null
            private set
        var timeToUploadContext: Long = DEFAULT_TIME_TO_UPLOAD_CONTEXT.toLong()
            private set
        var privacyPolicyUrl: String = DEFAULT_PRIVACY_POLICY_URL
            private set
        var privacyPolicyText: String? = null
            private set
        var gatewayUrl: String = DEFAULT_GATEWAY_URL
            private set
        var optInTitle: String? = null
            private set
        var optInTermsOfService: String? = null
            private set

        /**
         * Set terms and Services Url for the ConciergeConfiguration object
         *
         * @param tncUrl url pointing to terms and services webpage
         *
         * @Deprecated("use setTermsAndConditionsUrl, deprecated in version 3.0.0, will be removed in version 4.0.0")
         */
        @Deprecated("use setTermsAndConditionsUrl, deprecated in version 3.0.0, will be removed in version 4.0.0")
        fun setTermsAndServicesRequired(tncUrl: String): Builder {
            this.tncUrl = tncUrl
            return this
        }

        /**
         * Set terms and Services Url for the ConciergeConfiguration object
         *
         * @param tncUrl url pointing to terms and services webpage
         *
         */
        fun setTermsAndConditionsUrl(tncUrl: String): Builder {
            this.tncUrl = tncUrl
            return this
        }

        /**
         * Set terms and Services text for the ConciergeConfiguration object
         *
         * @param tncText The terms and conditions text
         *
         */
        fun setTermsAndConditionsText(tncText: String): Builder {
            this.tncText = tncText
            return this
        }

        /**
         * Set Gateway Url for the ConciergeConfiguration object
         *
         * @param gatewayUrl Gateway/Host URL to be used for communicating with Flybits servers.
         *
         * Please Note:
         * To set Region vise GatewayUrl :
         *  US = https://v3.flybits.com
         *  Canada = https://v3.canada.flybits.com
         *  Europe = https://v3-europe.flybits.com
         *  South America = https://v3-sa.flybits.com
         */
        fun setGatewayUrl(gatewayUrl: String): Builder {
            this.gatewayUrl = gatewayUrl
            return this
        }

        /**
         * Set privacyPolicyUrl for the ConciergeConfiguration object
         *
         * @param privacyPolicyUrl url pointing to privacy policy webpage
         */
        fun setPrivacyPolicyUrl(privacyPolicyUrl: String): Builder {
            this.privacyPolicyUrl = privacyPolicyUrl
            return this
        }

        /**
         * Set privacyPolicyText for the ConciergeConfiguration object
         *
         * @param privacyPolicyText the privacy policy text
         */
        fun setPrivacyPolicyText(privacyPolicyText: String): Builder {
            this.privacyPolicyText = privacyPolicyText
            return this
        }

        /**
         * Set setOptInText for the ConciergeConfiguration object
         *
         * @param setOptInText The text displayed on the opt In screen
         */
        fun setOptInTermsOfService(optInText: String): Builder {
            this.optInTermsOfService = optInText
            return this
        }

        /**
         * Set setOptInTitle for the ConciergeConfiguration object
         *
         * @param setOptInTitle The text displayed on the opt In screen
         */
        fun setOptInTitle(setOptInTitle: String): Builder {
            this.optInTitle = setOptInTitle
            return this
        }

        /**
         *  Set the time to upload context for the ConciergeConfiguration object
         *
         *  @param timeToUploadContext The time to upload Context data to the server
         */
        fun setTimeToUploadContext(timeToUploadContext: Long): Builder {
            this.timeToUploadContext = timeToUploadContext
            return this
        }


        fun build(): ConciergeConfiguration {
            return ConciergeConfiguration(this)
        }
    }

}