package com.flybits.concierge.fragments

import android.arch.lifecycle.Observer
import android.arch.lifecycle.ViewModelProviders
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.os.Bundle
import android.support.design.widget.Snackbar
import android.support.v4.app.Fragment
import android.support.v4.app.FragmentActivity
import android.support.v4.widget.SwipeRefreshLayout
import android.support.v7.widget.LinearLayoutManager
import android.support.v7.widget.RecyclerView
import android.util.Log
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.ConciergeConstants
import com.flybits.concierge.ConciergeFragment
import com.flybits.concierge.ConciergeFragment.CONCIERGE_FRAGMENT_TAG
import com.flybits.concierge.FlybitsConcierge
import com.flybits.concierge.R
import com.flybits.concierge.adapters.FeedAdapter
import com.flybits.concierge.exception.InvalidConciergeFragmentTagException
import com.flybits.concierge.models.BaseTemplate
import com.flybits.concierge.models.Category
import com.flybits.concierge.viewmodels.CategoryViewModel
import com.flybits.concierge.viewmodels.CategoryViewModelFactory
import com.flybits.concierge.viewmodels.TwitterViewModel
import com.flybits.context.ContextManager

/**
 * This fragment is responsible for displaying content that belongs to a particular
 * [Category]. Instantiate this fragment using the exposed `newInstance(Category)` method.
 */
class CategoryFragment : BasePagerFragment(), CategoryViewModel.FeedErrorDisplayer {

    companion object {
        const val CATEGORY_EXTRA = "category"

        fun newInstance(category: Category): CategoryFragment {
            val bundle = Bundle()
            bundle.putParcelable(CATEGORY_EXTRA, category)
            val fragment = CategoryFragment()
            fragment.arguments = bundle
            return fragment
        }
    }

    private var linearLayoutManager: LinearLayoutManager? = null

    private var recyclerView: RecyclerView? = null
    private var categoryViewModel: CategoryViewModel? = null
    private var feedAdapter: FeedAdapter? = null
    private var noDataView: View? = null
    private var refreshReceiver: BroadcastReceiver? = null
    private var swipeRefreshLayout: SwipeRefreshLayout? = null
    private var firstStart = true
    private var isValidConciergeTag = true
    private var conciergeFragment : Fragment? = null
    private val filter = IntentFilter(ConciergeConstants.BROADCAST_ENTITY_PUSH_RECEIVED)
    var category: Category? = null
        private set

    override fun titleRes(): Int {
        return R.string.flybits_con_fragment_title_feed
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View? {
        // Inflate the layout for this fragment
        return inflater.inflate(R.layout.flybits_con_fragment_category, container, false)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        category = arguments?.getParcelable(CATEGORY_EXTRA)
        val context = context

        Logger.d("Category: $category fragment view created!")

        linearLayoutManager = LinearLayoutManager(context)
        recyclerView = view.findViewById(R.id.flybits_con_fragment_feed_lytRecycler)
        recyclerView?.layoutManager = linearLayoutManager
        conciergeFragment =
            activity?.supportFragmentManager?.fragments?.find { it is ConciergeFragment }
        if (conciergeFragment == null) {
            val supportFragment: List<Fragment>? = activity?.supportFragmentManager?.fragments
            isValidConciergeTag = getConciergeFragmentByTag(supportFragment)
        }

        if (isValidConciergeTag) {
            feedAdapter =
                FeedAdapter(FlybitsConcierge.with(context), conciergeFragment as ConciergeFragment)
        } else {
            throw InvalidConciergeFragmentTagException()
        }
        recyclerView?.adapter = feedAdapter
        recyclerView?.setHasFixedSize(false)
        recyclerView?.addOnScrollListener(object : RecyclerView.OnScrollListener() {
            override fun onScrolled(recyclerView: RecyclerView, dx: Int, dy: Int) {
                categoryViewModel?.onScroll(getVisibleItems())
                super.onScrolled(recyclerView, dx, dy)
            }
        })

        //Logic for refresh animations after the first data load
        swipeRefreshLayout = view.findViewById(R.id.flybits_con_swipe_ref_layout)
        swipeRefreshLayout?.setOnRefreshListener {
            Logger.d(CategoryFragment::class.java.simpleName + ": onRefresh()")

            val refreshResult = object : BasicResultCallback {
                override fun onException(exception: FlybitsException) {
                    Logger.e("onRefresh : $exception")
                }

                override fun onSuccess() {
                    categoryViewModel?.refresh {
                        //hide animation once view model notifies us that its finished
                        swipeRefreshLayout?.isRefreshing = false
                    }
                }
            }
            // Run the ContextPlugins once on refresh of content
            ContextManager.refreshAll(context,refreshResult)

            if (context != null) {
                val twitterViewModel = ViewModelProviders
                    .of(context as FragmentActivity)
                    .get(TwitterViewModel::class.java)

                twitterViewModel.clearTweetLists()
            }
        }

        noDataView = view.findViewById(R.id.flybits_con_fragment_feed_lytNoData)

        if (category != null && context != null){
            categoryViewModel = ViewModelProviders.of(this, CategoryViewModelFactory(context
                , FlybitsConcierge.with(context), category!!)).get(CategoryViewModel::class.java)
        }

        //First load, show refresh animation
        swipeRefreshLayout?.isRefreshing = true
        categoryViewModel?.feedContent
            ?.observe(this, Observer { baseTemplates ->
                baseTemplates?.let {
                    Logger.d("Category: $category, Got data! data: $it")
                    feedAdapter?.submitList(it)
                    swipeRefreshLayout?.isRefreshing = false
                    categoryViewModel?.onScroll(getVisibleItems()) //New items may appear after refresh
                }
            })

        // Observe isNoData to check if there is no content available
        categoryViewModel?.isNoData?.observe(this, Observer { isNoData ->
            isNoData?.let {
                recyclerView?.visibility = if (isNoData) View.GONE else View.VISIBLE
                noDataView?.visibility = if (isNoData) View.VISIBLE else View.GONE
            }
        })

        categoryViewModel?.setFeedErrorDisplayer(this)

        refreshReceiver = object : BroadcastReceiver() {
            override fun onReceive(context: Context, intent: Intent) {
                if (feedAdapter != null) {
                    try {
                        Logger.i("Refreshing Contents")
                        categoryViewModel?.refresh()
                        feedAdapter?.notifyDataSetChanged()
                    } catch (e: Exception) {
                        Logger.exception("CategoryFragment.RefreshReceiver.onReceiver()", e)
                    }

                }
            }
        }

        activity?.registerReceiver(refreshReceiver,filter)

        parentFragment.let {
            if (it is FeedHolderFragment && it.isVisible(this)) {
                //Have to do this in case visibility changes prior to registration of listener
                categoryViewModel?.visibilityChange(true, getVisibleItems())
            }
        }

        //Refresh data to make sure local data is in sync with server
        categoryViewModel?.refresh()
    }

    override fun onResume() {
        super.onResume()
        activity?.registerReceiver(refreshReceiver,filter)
    }

    override fun onPause() {
        super.onPause()
        activity?.unregisterReceiver(refreshReceiver)
    }


    private fun getConciergeFragmentByTag(supportFragment: List<Fragment>?): Boolean {
        var conciergeFragmentByTag: Fragment?
        if (supportFragment != null) {
            outer@ for (element in supportFragment.indices) {
                if (supportFragment[element].tag == (CONCIERGE_FRAGMENT_TAG)) {
                    conciergeFragment = supportFragment[element]
                    isValidConciergeTag = true
                    break@outer
                }
                var childFragment: List<Fragment>? =
                    supportFragment[element].childFragmentManager.fragments
                for (childFragmentCounter in childFragment!!.indices) {
                    conciergeFragmentByTag =
                        childFragment[childFragmentCounter].fragmentManager?.findFragmentByTag(
                            CONCIERGE_FRAGMENT_TAG
                        )
                    if (conciergeFragmentByTag != null) {
                        conciergeFragment = conciergeFragmentByTag
                        isValidConciergeTag = true
                        break@outer
                    } else {
                        isValidConciergeTag = false
                    }
                    if (childFragmentCounter == (childFragment.size.minus(1)) && !isValidConciergeTag) {
                        childFragment =
                            childFragment[childFragmentCounter].childFragmentManager.fragments
                        getConciergeFragmentByTag(childFragment)
                    }
                }
            }
        } else {
            isValidConciergeTag = false
        }
        return isValidConciergeTag
    }

    //Refresh content if activity is brought to start state ONLY after it is called more than once
    //This refreshes the content when the app is brought into the foreground from the background
    override fun onStart() {
        super.onStart()
        if (firstStart){
            firstStart = false
        } else {
            swipeRefreshLayout?.isRefreshing = true
            categoryViewModel?.refresh {
                //hide animation once view model notifies us that its finished
                swipeRefreshLayout?.isRefreshing = false
            }
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        //Receiver may already be unregistered or may have never been registered in that case catch the exception
        categoryViewModel?.clean()
    }

    override fun onError(err: String) {
        Logger.d("CategoryFragment: onError() err: $err")
        val activity = activity
        val v = view
        /*The instanceof check is done to make sure that the user is still within
        * the concierge feature at the time the toast is displayed.*/
        if (activity != null && v != null) {
            Snackbar.make(v, err, Snackbar.LENGTH_LONG).show()
        }
    }

    /**
     * Notify the [CategoryFragment] of the new visibility to the user.
     *
     * @param visible Whether the [CategoryFragment] is visible.
     */
    fun onVisibilityChange(visible: Boolean) {
        categoryViewModel?.visibilityChange(visible, getVisibleItems())
    }

    private fun getVisibleItems(): List<BaseTemplate> {
        val firstVisibleItem = linearLayoutManager?.findFirstCompletelyVisibleItemPosition() ?: 0
        val lastVisibleItem = linearLayoutManager?.findLastCompletelyVisibleItemPosition() ?: 0
        val list = feedAdapter?.currentList
        list?.let {
            return if (it.size > 0 && (firstVisibleItem != RecyclerView.NO_POSITION && lastVisibleItem != RecyclerView.NO_POSITION)) {
                val items = list.subList(firstVisibleItem, lastVisibleItem + 1)
                items.filterNotNull()
            } else
                emptyList()
        } ?: return emptyList()
    }
}
