package com.flybits.concierge.viewpopulators

import android.os.Bundle
import androidx.core.content.ContextCompat
import android.text.TextUtils
import android.view.View
import android.widget.ImageView
import android.widget.TextView
import com.bumptech.glide.Glide
import com.bumptech.glide.load.DecodeFormat
import com.bumptech.glide.request.RequestOptions
import com.flybits.android.kernel.ContentAnalytics
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.FlybitsNavigator
import com.flybits.concierge.R
import com.flybits.concierge.Utils
import com.flybits.concierge.activities.VideoActivity
import com.flybits.concierge.activities.VideoPlaylistActivity
import com.flybits.concierge.models.Video
import com.flybits.concierge.models.Videos
import java.net.URL

/**
 * Handles logic for populating video content to UI
 */
class VideoViewPopulator(private val contentAnalytics: ContentAnalytics) {

    /**
     *  Populate video content to video related views
     *  @param data videos data to show
     */
    fun populate(data: Videos, flybitsNavigator: FlybitsNavigator, thisView: View
                 , singleContainer: View, doubleContainer: View, tripleContainer: View
                 , singlePlaceHolderViews: Array<ImageView>
                 , doublePlaceHolderViews: Array<ImageView>
                 , triplePlaceHolderViews: Array<ImageView>, callToActionView: View){
        val videos = data.videos.list
        videos.forEach { it.content = data.content }

        tripleContainer.visibility = if (videos.size > 2) View.VISIBLE else View.GONE
        doubleContainer.visibility = if (videos.size == 2) View.VISIBLE else View.GONE
        singleContainer.visibility = if (videos.size == 1) View.VISIBLE else View.GONE
        callToActionView.visibility = if (videos.size > 1) View.VISIBLE else View.GONE

        val placeHolderViews: Array<ImageView> = when (videos.size) {
            0 -> return
            1 -> {
                val titleTextView : TextView = thisView.findViewById(R.id.flybits_con_item_template_videos_txtTitle)
                val durationTextView: TextView = thisView.findViewById(R.id.flybits_con_item_template_videos_txtDuration)
                populateVideoViews(videos[0], data.content.id ,flybitsNavigator, titleTextView, durationTextView, singlePlaceHolderViews[0], thisView)
                return
            }
            2 -> doublePlaceHolderViews
            else -> triplePlaceHolderViews
        }

        for (i in placeHolderViews.indices) {
            val view = placeHolderViews[i]
            if (i < videos.size) {
                val video = videos.get(i)

                populateVideoViews(video, data.content.id, flybitsNavigator,null, null, view, view)
            } else {
                view.setImageDrawable(ContextCompat.getDrawable(view.context, android.R.drawable.bottom_bar))
                view.setOnClickListener(null)
            }
        }

        callToActionView.setOnClickListener {
            contentAnalytics.trackEngaged(data.content.id)
            val extras = Bundle()
            extras.putParcelableArrayList(VideoPlaylistActivity.EXTRA_VIDEO, videos)
            flybitsNavigator.openActivity(VideoPlaylistActivity::class.java, extras)
        }
    }

    /**
     *  Populate video content to video related views
     *  @param video video data to show
     */
    fun populateVideoViews(video: Video, parentContentId: String, fragmentSwitcher: FlybitsNavigator
                           , titleTextView: TextView?, durationTextView: TextView?
                           , placeholderImageView: ImageView, callToActionView: View) {
        titleTextView?.text = video.title.value

        if (durationTextView != null) {
            durationTextView.visibility = if (!TextUtils.isEmpty(video.duration)) View.VISIBLE else View.INVISIBLE
            durationTextView.text = video.duration
        }

        val options = RequestOptions().format(DecodeFormat.PREFER_RGB_565)

        Glide.with(placeholderImageView.context)
            .load(video.imageURL)
            .apply(options)
            .into(placeholderImageView)

        callToActionView.setOnClickListener(View.OnClickListener { v ->
            contentAnalytics.trackEngaged(parentContentId)
            // YouTube
            try {
                val videoUrl = URL(video.url)
                videoUrl.host

                var videoId: String? = null

                when (videoUrl.host) {
                    "www.youtube.com" -> {
                        val map = Utils.splitQuery(videoUrl)
                        videoId = map["v"]
                    }
                    "youtu.be" -> {
                        val path = videoUrl.path
                        if (!TextUtils.isEmpty(path) && path.length > 2) {
                            videoId = videoUrl.path
                                .substring(1)
                        }
                    }
                }

                if (!TextUtils.isEmpty(videoId)) {
                    video.watchYoutubeVideo(v.context, videoId)
                    return@OnClickListener
                }
            } catch (e: Exception) {
                Logger.exception(Video::class.java.simpleName, e)
            }

            // Server
            val extras = Bundle()
            extras.putParcelable(VideoActivity.EXTRA_VIDEO, video)
            fragmentSwitcher.openActivity(VideoActivity::class.java, extras)
        })
    }

}