package com.flybits.concierge.viewmodels

import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import com.flybits.android.push.models.newPush.ContentPush
import com.flybits.android.push.models.newPush.Push
import com.flybits.android.push.models.newPush.WeblinkPush
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.FlybitsNavigator
import com.flybits.concierge.FlybitsViewProviderGetter
import com.flybits.concierge.models.BaseTemplate
import com.flybits.concierge.models.NotificationContent
import com.flybits.concierge.repository.RepositoryResponse
import com.flybits.concierge.repository.push.NotificationContentRepository

/**
 * Responsible for retrieving and handling data/actions related to [NotificationContent]
 * using [LiveData].
 */
class NotificationsViewModel(private val notificationContentRepository: NotificationContentRepository
                             , private val flybitsViewProviderGetter: FlybitsViewProviderGetter
                             , private val flybitsNavigator: FlybitsNavigator): ViewModel() {

    data class ViewModelNotificationContent(val data: List<NotificationContent>? = null, val positionOfPush: Int? = null)

    /**
     * Returns LiveData which streams [RepositoryResponse] that holds [ViewModelNotificationContent]
     * to be displayed.
     *
     * @param findPosition [Push] who's position must be known by the View for scrolling purposes.
     *
     * @return LiveData containing [RepositoryResponse] holding [ViewModelNotificationContent].
     */
    fun getNotificationContents(findPosition: ContentPush?): LiveData<RepositoryResponse<ViewModelNotificationContent>>{
        val liveData = MutableLiveData<RepositoryResponse<ViewModelNotificationContent>>()

        notificationContentRepository.getNotificationContentRemote(object: ObjectResultCallback<List<NotificationContent>> {
            override fun onSuccess(item: List<NotificationContent>) {

                //Find position of passed push, if none is found null will be returned
                val position = if (findPosition != null){
                    val index = item.indexOfFirst {
                        findPosition.contentId == it.baseTemplate.content.id
                    }
                    //return null if none is found
                    if (index == -1)
                        null
                    else index
                } else {
                    null
                }
                liveData.value = RepositoryResponse(data = ViewModelNotificationContent(data = item, positionOfPush = position))
            }

            override fun onException(exception: FlybitsException) {
                liveData.value = RepositoryResponse(error = exception)
            }
        })

        return liveData
    }

    /**
     * Handle individual [Push] that has been passed down. If the [Push] is tied to Content
     * and a FlybitsViewProvider is registered, its push handling method will be invoked.
     */
    fun handlePassedPush(push: Any?): LiveData<RepositoryResponse<Boolean>>{
        val liveData = MutableLiveData<RepositoryResponse<Boolean>>()
        when (push) {
            is WeblinkPush -> {
                Logger.i("processing web link push")
                flybitsNavigator.openUrl(push.url)
                liveData.value = RepositoryResponse(data = true)
                return liveData
            }
            is ContentPush -> {
                Logger.i("processing content push")
                notificationContentRepository.getBaseTemplate(push.contentId, object: ObjectResultCallback<BaseTemplate> {
                    override fun onSuccess(item: BaseTemplate) {
                        val provider = flybitsViewProviderGetter.getFlybitsViewProvider(item.content?.type ?: "")
                        if (provider == null) {
                            liveData.value = RepositoryResponse(error = FlybitsException("No associated FlybitsViewProvider for content"))
                        } else {
                            Logger.d("NotificationsViewModel: content type found = ${item.content?.type}")
                            provider.onPushOpened(item, flybitsNavigator)
                            liveData.value = RepositoryResponse(data = true)
                        }
                    }

                    override fun onException(exception: FlybitsException) {
                        liveData.value = RepositoryResponse(error = exception)
                    }
                })
                return liveData

            }
            else -> liveData.value = RepositoryResponse(error = FlybitsException("Expected push, got something else"))
        }

        return liveData
    }
}