package com.flybits.concierge.fragments;


import androidx.lifecycle.ViewModelProviders;
import android.os.Bundle;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CheckBox;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.flybits.android.kernel.reservedtemplates.survey.models.Question;
import com.flybits.android.kernel.reservedtemplates.survey.models.QuestionChoice;
import com.flybits.commons.library.logging.Logger;
import com.flybits.concierge.R;
import com.flybits.concierge.enums.QuestionUIType;
import com.flybits.concierge.viewmodels.SurveyViewModel;
import com.flybits.concierge.views.SurveyBaseView;
import com.flybits.concierge.views.SurveyDate;
import com.flybits.concierge.views.SurveyDropDown;
import com.flybits.concierge.views.SurveyFile;
import com.flybits.concierge.views.SurveyLocation;
import com.flybits.concierge.views.SurveyLongAnswer;
import com.flybits.concierge.views.SurveyMultipleChoice;
import com.flybits.concierge.views.SurveyNumber;
import com.flybits.concierge.views.SurveyShortAnswer;
import com.flybits.concierge.views.SurveyTime;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Locale;

import static com.flybits.concierge.ConciergeConstants.SURVEY_QUESTION_REQUIRED;

public class QuestionFragment extends Fragment
{
    private static final String ARG_QUESTION = "flybits_con_arg_question";
    private static final String ARG_NUMBER = "flybits_con_arg_number";

    private Question question;
    private int number;
    private boolean required;
    private QuestionUIType uiType;
    private LinearLayout containerView;
    private View.OnClickListener clickListener;

    private SurveyViewModel viewModel;

    public QuestionFragment()
    {
        // Required empty public constructor
    }

    public static QuestionFragment newInstance(Question question, int number)
    {
        QuestionFragment fragment = new QuestionFragment();
        Bundle args = new Bundle();
        args.putParcelable(ARG_QUESTION, question);
        args.putInt(ARG_NUMBER, number);
        fragment.setArguments(args);
        return fragment;
    }

    @Override
    public void onCreate(Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);
        if (getArguments() != null)
        {
            question = getArguments().getParcelable(ARG_QUESTION);
            number = getArguments().getInt(ARG_NUMBER);
        }
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState)
    {
        // Inflate the layout for this fragment
        return inflater.inflate(R.layout.flybits_con_fragment_question, container, false);
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState)
    {
        super.onViewCreated(view, savedInstanceState);

        TextView questionNumberView = view.findViewById(R.id.flybits_con_fragment_question_txtQuestionNumber);
        questionNumberView.setText(String.format(Locale.getDefault(), getString(R.string.flybits_con_fragment_question_question_format), number));

        TextView questionView = view.findViewById(R.id.flybits_con_fragment_question_txtQuestion);
        questionView.setText(question.question.getValue());

        containerView = view.findViewById(R.id.flybits_con_fragment_question_lytQuestionContainer);
        clickListener = new View.OnClickListener()
        {
            @Override
            public void onClick(View v)
            {
                populateAnswer();
            }
        };

        viewModel = ViewModelProviders.of(getActivity()).get(SurveyViewModel.class);

        // use a state machine to populate the screen with the type of answer.
        String uiTypeKey = "unknown";

        try
        {
            uiTypeKey = question.getMetadata().getString(QuestionUIType.ARG_JSON_KEY_UI_TYPE);
            required = question.getMetadata().getBoolean(SURVEY_QUESTION_REQUIRED);
        }
        catch (Exception e)
        {
            Logger.exception(QuestionFragment.class.getSimpleName(), e);
        }

        uiType = QuestionUIType.fromString(uiTypeKey);

        TextView requiredView = view.findViewById(R.id.flybits_con_fragment_question_txtRequired);
        requiredView.setVisibility(required ? View.VISIBLE : View.GONE);

        configureAnswer();
    }

    private void configureAnswer()
    {
        Object answer = viewModel.getAnswers().get(question.questionNumber);

        containerView.removeAllViews();
        switch (uiType)
        {
            case SHORT:
                containerView.addView(new SurveyShortAnswer(getContext(), answer instanceof String ? answer.toString() : null));
                break;
            case LONG:
                containerView.addView(new SurveyLongAnswer(getContext(), answer instanceof String ? answer.toString() : null));
                break;
            case CHECK:
                List<Integer> selections = answer instanceof List ? (List<Integer>) answer : null;
                for (int i = 0; i < question.choices.size(); i++)
                {
                    QuestionChoice choice = question.choices.get(i);
                    CheckBox box = (CheckBox) getLayoutInflater().inflate(R.layout.flybits_con_view_survey_check_box, containerView, false);
                    box.setText(choice.value.getValue());
                    box.setId(i + 100);
                    containerView.addView(box);

                    if (selections != null && !box.isChecked())
                    {
                        box.setChecked(selections.contains(i));
                    }
                }
                break;
            case MULTIPLE:
                containerView.addView(new SurveyMultipleChoice(getContext(), question.choices, answer instanceof Integer ? (Integer) answer : -1));
                break;
            case DROPDOWN:
                containerView.addView(new SurveyDropDown(getContext(), question.choices, answer instanceof Integer ? (Integer) answer : -1));
                break;
            case NUMBER:
                containerView.addView(new SurveyNumber(getContext(), answer instanceof Long ? (Long) answer :-1));
                break;
            case DATE:
                containerView.addView(new SurveyDate(getContext(), getActivity().getFragmentManager(), answer instanceof Calendar ? (Calendar) answer : null));
                break;
            case TIME:
                containerView.addView(new SurveyTime(getContext(), getActivity().getFragmentManager(), answer instanceof Calendar ? (Calendar) answer : null));
                break;
            case LOCATION:
                containerView.addView(new SurveyLocation(getContext(), answer instanceof String ? answer.toString() : null));
                break;
            case FILE:
                containerView.addView(new SurveyFile(getContext()));
                break;
            default:
                TextView unsupported = new TextView(getContext());
                unsupported.setText(R.string.flybits_con_view_currently_not_supported);
                containerView.addView(unsupported);
                break;
        }

        for (int i = 0; i < containerView.getChildCount(); i++)
        {
            containerView.getChildAt(i).setOnClickListener(clickListener);
        }
    }

    private void populateAnswer()
    {
        List<Integer> selections = new ArrayList<>();
        for (int i = 0; i < containerView.getChildCount(); i++)
        {
            View view = containerView.getChildAt(i);

            if (view instanceof CheckBox)
            {
                CheckBox checkBox = (CheckBox) view;
                if (checkBox.isChecked())
                {
                    selections.add(i);
                }
            }
            else if (view instanceof SurveyBaseView)
            {
                Object answer = ((SurveyBaseView) view).getAnswer();

                if (answer == null)
                {
                    viewModel.getAnswers().remove(question.questionNumber);
                }
                else
                {
                    viewModel.getAnswers().put(question.questionNumber, answer);
                }

                return;
            }
        }

        if (!selections.isEmpty())
        {
            viewModel.getAnswers().put(question.questionNumber, selections);
        }
        else
        {
            viewModel.getAnswers().remove(question.questionNumber);
        }
    }
}
