package com.flybits.concierge;

import android.app.Activity;
import android.app.ActivityManager;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.graphics.Point;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentActivity;
import android.view.Display;
import android.view.View;
import android.view.WindowManager;
import android.view.inputmethod.InputMethodManager;

import com.flybits.commons.library.logging.Logger;

import org.jetbrains.annotations.Nullable;

import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;

public class Utils
{
    private static final SimpleDateFormat TIMESTAMP_FLYBITS_JSON_DATE = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ");

    private static final SimpleDateFormat DATE_TIME_FORMAT = new SimpleDateFormat("MMMM d, yyyy' - 'h:mma");
    private static final SimpleDateFormat DATE_FULL_FORMAT = new SimpleDateFormat("EEE, MMMM d, yyyy");
    private static final SimpleDateFormat TIME_FORMAT = new SimpleDateFormat("h:mma");
    private static final SimpleDateFormat WEEK_DAY_FORMAT = new SimpleDateFormat("EEEE", Locale.US);

    /**
     * Transform Calendar to ISO 8601 string.
     */
    public static String fromCalendar(final Calendar calendar)
    {
        Date date = calendar.getTime();
        String formatted = TIMESTAMP_FLYBITS_JSON_DATE.format(date);
        return formatted.substring(0, 22) + ":" + formatted.substring(22);
    }

    /**
     * Get current date and time formatted as ISO 8601 string.
     */
    public static String now()
    {
        return fromCalendar(GregorianCalendar.getInstance());
    }

    /**
     * Transform ISO 8601 string to Calendar.
     */
    public static Calendar toCalendar(final String iso8601string) throws ParseException
    {
        Calendar calendar = GregorianCalendar.getInstance(TimeZone.getDefault());
        String s = iso8601string.replace("Z", "+00:00");
        try
        {
            s = s.substring(0, 22) + s.substring(23);  // to get rid of the ":"
        }
        catch (IndexOutOfBoundsException e)
        {
            throw new ParseException("Invalid length", 0);
        }
        Date date = TIMESTAMP_FLYBITS_JSON_DATE.parse(s);
        calendar.setTime(date);
        return calendar;
    }

    public static String getDateRange(Date startDate, Date endDate)
    {
        Calendar startCal = Calendar.getInstance();
        Calendar endCal = Calendar.getInstance();
        startCal.setTime(startDate);
        endCal.setTime(endDate);

        boolean sameDay = startCal.get(Calendar.YEAR) == endCal.get(Calendar.YEAR) && startCal.get(Calendar.DAY_OF_YEAR) == endCal.get(Calendar.DAY_OF_YEAR);

        return String.format(Locale.getDefault(), "%s to %s", // TODO: this need to be localized
                DATE_TIME_FORMAT.format(startDate), sameDay ? TIME_FORMAT.format(endDate) : DATE_TIME_FORMAT.format(endDate));
    }

    public static String getTimeRange(Date startDate, Date endDate)
    {
        return String.format(Locale.getDefault(), "%s - %s", TIME_FORMAT.format(startDate), TIME_FORMAT.format(endDate));
    }

    public static String getDateString(Date date)
    {
        return DATE_FULL_FORMAT.format(date);
    }

    public static String getTimeString(Date date)
    {
        return TIME_FORMAT.format(date);
    }

    public static String getWeekDayString(Date date)
    {
        return WEEK_DAY_FORMAT.format(date);
    }

    /**
     * Returns the {@link ConciergeFragment} if one exists.
     *
     * @param activity Activity hosting the {@link ConciergeFragment}
     * @return {@link ConciergeFragment} found, or null.
     */
    public static ConciergeFragment getConciergeFragment(FragmentActivity activity)
    {
        List<Fragment> fragments = activity.getSupportFragmentManager().getFragments();
        for (Fragment fragment : fragments)
        {
            if (fragment instanceof ConciergeFragment)
            {
                return (ConciergeFragment) fragment;
            }
        }

        return null;
    }

    public static Map<String, String> splitQuery(URL url) throws UnsupportedEncodingException
    {
        Map<String, String> query_pairs = new LinkedHashMap<>();
        String query = url.getQuery();
        String[] pairs = query.split("&");
        for (String pair : pairs)
        {
            int idx = pair.indexOf("=");
            query_pairs.put(URLDecoder.decode(pair.substring(0, idx), StandardCharsets.UTF_8.name()), URLDecoder.decode(pair.substring(idx + 1), StandardCharsets.UTF_8.name()));
        }

        return query_pairs;
    }

    public static int screenWidth(Context context)
    {
        WindowManager wm = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        Display display = wm.getDefaultDisplay();
        Point size = new Point();
        display.getSize(size);

        return size.x;
    }

    public static float screenWidthDp(Context context)
    {
        float density  = context.getResources().getDisplayMetrics().density;
        return screenWidth(context) / density;
    }

    public static void dismissKeyboard(Activity activity)
    {
        View v = activity.getWindow().getCurrentFocus();
        if (v != null)
        {
            InputMethodManager imm = (InputMethodManager) activity.getSystemService(Context.INPUT_METHOD_SERVICE);
            if (imm != null && imm.isAcceptingText())
            {
                imm.hideSoftInputFromWindow(v.getWindowToken(), 0);
            }
        }
    }

    /**
     * Retrieve the launcher activity associated with the application.
     *
     * @param context for the application.
     * @return Launcher activity associated with the application.
     */
    @Nullable
    public static Class<Activity> launcherActivity(Context context)
    {
        try
        {
            final PackageManager packageManager = context.getPackageManager();

            // package manager is provider of all the application information
            Intent mainIntent = new Intent(Intent.ACTION_MAIN, null);
            mainIntent.addCategory(Intent.CATEGORY_LAUNCHER);

            List<ResolveInfo> appList = packageManager.queryIntentActivities(mainIntent, 0);

            for (ResolveInfo info : appList)
            {
                if (context.getPackageName().equals(info.activityInfo.packageName))
                {
                    return (Class<Activity>) Class.forName(info.activityInfo.name);
                }
            }
        }
        catch (ClassNotFoundException e)
        {
            Logger.exception("Utils.launcherActivity", e);
        }

        return null;
    }

    /**
     *
     * @param context Context associated with the application.
     * @return Whether the application is active or not.
     */
    public static boolean isAppActive(Context context) {
        ActivityManager mgr = (ActivityManager)context.getSystemService(Context.ACTIVITY_SERVICE);
        if (mgr != null) {
            for (ActivityManager.RunningAppProcessInfo app : mgr.getRunningAppProcesses()) {
                if (app.importance == ActivityManager.RunningAppProcessInfo.IMPORTANCE_FOREGROUND) {
                    return app.processName.equals(context.getPackageName());
                }
            }
        }
        return false;
    }
}
