package com.flybits.concierge;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.res.XmlResourceParser;
import android.net.ConnectivityManager;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.XmlRes;

import androidx.work.*;
import com.flybits.android.kernel.KernelScope;
import com.flybits.android.push.PushManager;
import com.flybits.android.push.PushScope;
import com.flybits.android.push.analytics.PushAnalytics;
import com.flybits.android.push.models.Push;
import com.flybits.commons.library.SharedElementsFactory;
import com.flybits.commons.library.api.FlybitsManager;
import com.flybits.commons.library.api.FlybitsScope;
import com.flybits.commons.library.api.idps.IDP;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.User;
import com.flybits.concierge.activities.ConciergeActivity;
import com.flybits.concierge.activities.ConciergePopupActivity;
import com.flybits.concierge.activities.NotificationsActivity;
import com.flybits.concierge.enums.ShowMode;
import com.flybits.concierge.exception.ConciergeUnauthenticatedException;
import com.flybits.concierge.exception.ConciergeUninitializedException;
import com.flybits.concierge.repository.push.NotificationContentRepository;
import com.flybits.concierge.services.PreloadingWorker;
import com.flybits.concierge.viewproviders.*;
import com.flybits.context.ContextManager;
import com.flybits.context.ContextScope;
import com.flybits.context.plugins.FlybitsContextPlugin;
import com.flybits.internal.db.CommonsDatabase;
import org.jetbrains.annotations.NotNull;

import java.lang.ref.WeakReference;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

/**
 * A {@code FlybitsConcierge} is a singleton responsible for dealing with Concierge
 * SDK configuration, authentication, and display.
 *
 */
public class FlybitsConcierge implements FlybitsViewProviderGetter {
    private static FlybitsConcierge INSTANCE;

    private WeakReference<Context> lastProvidedContext;
    private ConciergeConfiguration currentConfig;
    private IDP idp;    //IDP set during most recent authenticate call
    protected FlybitsManager flybitsManager;
    private NotificationContentRepository notificationContentRepository;
    private CommonsDatabase commonsDatabase;
    private PushAnalytics pushAnalytics;
    private final Set<OptedStateChangeListener> optedStateChangeListeners;
    private final Set<AuthenticationStatusListener> authenticationStatusListeners;
    private final Map<String, FlybitsViewProvider> flybitsViewProviders;

    private boolean authenticating = false;     //If authentication is currently happening
    private boolean authenticationRequested = false;    //Tracks whether authentication has ever been requested so we know whether to register network receiver
    private boolean networkReceiverRegistered = false;
    private boolean initialized = false;

    private BroadcastReceiver networkReceiver = new BroadcastReceiver()
    {
        @Override
        public void onReceive(Context context, Intent intent)
        {
            try {
                ConnectivityManager manager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);

                //Retry authentication if its been previously requested and we're not authenticated yet and phone came online
                if (manager != null
                        && manager.getActiveNetworkInfo().isConnected()
                        && isAutoRetryAuthenticationOnConnectedEnabled()
                        && !isAuthenticated()
                        && authenticationRequested) {
                    retryAuthentication();
                }
            }
            catch(Exception e) {
                Logger.exception("networkReceiver.onReceive",e);
            }

        }
    };

    private BasicResultCallback conciergeConnectionResultCallback = new BasicResultCallback() {
        @Override
        public void onSuccess() {
            authenticating = false;

            broadcastAuthenticationState(AuthenticationStatusListener.AUTHENTICATED);

            unregisterNetworkReceiver();

            boolean success = enablePushMessagingUsingLocalToken();
            Logger.setTag(FlybitsConcierge.class.getSimpleName()).d("Push messaging using local token success: "+success);
        }

        @Override
        public void onException(FlybitsException exception) {
            authenticating = false;

            //Hacky for now but only solution without making specific exceptions
            Context context = getContext();
            if (context != null && SharedElementsFactory.INSTANCE.get(context).getSavedJWTToken().isEmpty()) {
                broadcastAuthenticationError(exception);
            } else if (context != null && exception.getMessage().contains("Connecting") && isAuthenticated()) {
                broadcastAuthenticationState(AuthenticationStatusListener.AUTHENTICATED);
            }

            registerNetworkReceiver();
        }
    };

    FlybitsConcierge(Context context, PushAnalytics pushAnalytics, CommonsDatabase commonsDatabase) {
        context = context.getApplicationContext();
        optedStateChangeListeners = Collections.synchronizedSet(new HashSet<>());
        authenticationStatusListeners = Collections.synchronizedSet(new HashSet<>());
        flybitsViewProviders = Collections.synchronizedMap(new HashMap<>());
        lastProvidedContext = new WeakReference<>(context);
        this.commonsDatabase = commonsDatabase;
        this.pushAnalytics = pushAnalytics;
        this.notificationContentRepository = NotificationContentRepository.Companion.create(context, this);
    }

    /**
     * Used to get reference to {@code FlybitsConcierge} singleton instance.
     *
     * @param context The context of your application.
     * @return {@code FlybitsConcierge} singleton instance.
     */
    public static FlybitsConcierge with(Context context) {
        if (context == null) {
            throw new NullPointerException("Context cannot be null");
        }

        synchronized (FlybitsConcierge.class) {
            if (INSTANCE == null) {
                PushAnalytics pushAnalytics = new PushAnalytics(context);
                CommonsDatabase commonsDatabase = CommonsDatabase.getDatabase(context);
                INSTANCE = new FlybitsConcierge(context.getApplicationContext(), pushAnalytics, commonsDatabase);
                return INSTANCE;
            }
        }
        return INSTANCE;
    }

    Context getContext() throws NullPointerException {
        if (lastProvidedContext != null && lastProvidedContext.get() != null) {
            return lastProvidedContext.get();
        } else {
            throw new NullPointerException("Context is null");
        }
    }

    private FlybitsManager createFlybitsManager() {

        FlybitsManager.Builder builder = new FlybitsManager.Builder(getContext())
                .setAccount(null)
                .setProjectId(getConfiguration().getProjectID())
                .setGatewayURL(currentConfig.getGatewayUrl());

        addScopes();

        if (BuildConfig.DEBUG) {
            builder.setDebug();
        }

        return builder.build();
    }

    private void addScopes() {
        List<FlybitsScope> listScopes = new ArrayList<>(Arrays.asList(PushScope.SCOPE, KernelScope.SCOPE, new ContextScope(currentConfig.getTimeToUploadContext(), TimeUnit.MINUTES)));
        for (FlybitsScope scopes : listScopes) {
            FlybitsManager.addScope(scopes);
        }
    }

    /**
     * Initializes the {@code FlybitsConcierge} instance using the conciergeConfiguration object, allowing for further method invocations.
     *
     * @param conciergeConfiguration ConciergeConfiguration object created using the ConciergeConfiguration Builder
     */
    public void initialize(ConciergeConfiguration conciergeConfiguration) {
        if (initialized) {
            return;
        }

        initialized = true;

        currentConfig = conciergeConfiguration;

        flybitsManager = createFlybitsManager();

        //Registration needs to happen here if you want push to work
        registerViewProviders();
    }

    /**
     * Initializes the {@code FlybitsConcierge} instance, allowing for further method invocations.
     *
     * @param cfgResource Resource file containing required and optional configuration fields.
     */
    public void initialize(@XmlRes int cfgResource) {
        if (initialized) {
            return;
        }

        XmlResourceParser xmlResourceParser = new ResourceProvider(getContext()).getXmlParser(cfgResource);

        ConciergeConfiguration config = ConciergeConfigurationXMLParser.INSTANCE.generateConfigurationFromXML(xmlResourceParser);

        initialize(config);

    }

    /*
        Registering all necessary viewProviders
     */
    private void registerViewProviders() {

        Context context = getContext();

        registerFlybitsViewProvider(new EventsViewProvider());
        registerFlybitsViewProvider(new ImageViewProvider(context));
        registerFlybitsViewProvider(new LinksViewProvider(context));
        registerFlybitsViewProvider(new OnboardingViewProvider());
        registerFlybitsViewProvider(new ScheduleViewProvider(context));
        registerFlybitsViewProvider(new SurveyViewProvider());
        registerFlybitsViewProvider(new TextViewProvider());
        registerFlybitsViewProvider(new TwitterViewProvider());
        registerFlybitsViewProvider(new VideosViewProvider(context));
        registerFlybitsViewProvider(new ArticlesViewProvider(context));

    }

    /**
     * Enable debug mode that will allow for more verbose logs to be printed in the console.
     * Do not use this in a production environment.
     */
    public void enableDebugMode(){
        FlybitsManager.setDebug();
    }

    /**
     * @return Whether the {@code FlybitsConcierge} is initialized.
     */
    public boolean isInitialized() {
        return initialized;
    }

    /**
     * Register {@code AuthenticationStateListener} that will be notified about any changes to
     * the authentication status.
     *
     * This method is purposely seperate from the authenticate method because it is needed in order
     * to get updates in the ConciergeFragment even if authenticate isn't called in there due to
     * AutoAuthenticate being set to false.
     *
     * @param authenticationStatusListener {@code AuthenticationStateListener} that will be notified about changes to the authentication status.
     *
     * @throws ConciergeUninitializedException if {@link FlybitsConcierge#initialize(int)} has not been called.
     */
    public void registerAuthenticationStateListener(@NonNull AuthenticationStatusListener authenticationStatusListener) throws ConciergeUninitializedException {

        if (!isInitialized()){
            throw new ConciergeUninitializedException();
        }

        synchronized (authenticationStatusListeners) {
            authenticationStatusListeners.add(authenticationStatusListener);
        }
    }

    /**
     * Unregister {@code AuthenticationStateListener} so that it is no longer notified about changes
     * to the authentication status.
     *
     * @param authenticationStatusListener {@code AuthenticationStateListener} being unregistered.
     *
     * @return true if unregistered successfully, and false otherwise.
     *
     * @throws ConciergeUninitializedException if {@link FlybitsConcierge#initialize(int)} has not been called.
     */
    public boolean unregisterAuthenticationStateListener(@NonNull AuthenticationStatusListener authenticationStatusListener) throws ConciergeUninitializedException {

        if (!isInitialized()){
            throw new ConciergeUninitializedException();
        }

        synchronized (authenticationStatusListeners) {
            return authenticationStatusListeners.remove(authenticationStatusListener);
        }
    }

    private void broadcastAuthenticationState(String state) {
        synchronized (authenticationStatusListeners) {
            //Copy over to array to avoid ConcurrentModificationException
            Set<AuthenticationStatusListener> listeners = new HashSet<>(authenticationStatusListeners);

            for (AuthenticationStatusListener authenticationStatusListener : listeners) {
                switch(state) {
                    case AuthenticationStatusListener.AUTHENTICATED:
                        authenticationStatusListener.onAuthenticated();
                        break;
                    case AuthenticationStatusListener.AUTHENTICATION_STARTED:
                        authenticationStatusListener.onAuthenticationStarted();
                        break;
                    default:
                }
            }
        }
    }

    private void broadcastAuthenticationError(FlybitsException err) {
        synchronized (authenticationStatusListeners) {
            //Copy over to array to avoid ConcurrentModificationException
            Set<AuthenticationStatusListener> listeners = new HashSet<>(authenticationStatusListeners);

            for (AuthenticationStatusListener authenticationStatusListener : listeners) {
                authenticationStatusListener.onAuthenticationError(err);
            }
        }
    }

    /**
     * Opt in to the concierge. User data collection, and communication with servers
     * is stopped while the user is opted out, and resumed when the user opts back in.
     *
     * @param basicResultCallback Callback that will be used for passing the result of the
     *                            opt in request.
     *
     * @throws ConciergeUninitializedException if {@link FlybitsConcierge#initialize(int)} has not been called.
     *
     */
    public void optIn(@NonNull BasicResultCallback basicResultCallback) throws ConciergeUninitializedException {
        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        } else if(isAuthenticating()|| !isAuthenticated()) {
            basicResultCallback.onException(new ConciergeUnauthenticatedException());
        } else {
            flybitsManager.optIn(new BasicResultCallback() {
                @Override
                public void onSuccess() {
                    broadcastOptedInStateChange(true);
                    basicResultCallback.onSuccess();
                }

                @Override
                public void onException(@NotNull FlybitsException e) {
                    basicResultCallback.onException(e);
                }
            });
        }
    }

    /**
     * Retry the authentication with a previously provided {@link com.flybits.commons.library.api.idps.FlybitsIDP}
     * Do not call this if authenticate() has never been called.
     *
     * @return Whether the retry succeeded.
     *
     * @throws ConciergeUninitializedException if the concierge has not been initialized.
     */
    synchronized boolean retryAuthentication() throws ConciergeUninitializedException {
        /*Do not allow for retrying authentication after user logged out or if authentication was never
            requested. That way if the view is shown and the sdk is logged out it won't re-authenticate immediately.*/
        if(!initialized || idp == null || authenticating || isAuthenticated() || !authenticationRequested) {
            return false;
        }

        return authenticate(idp);
    }

    /**
     * Attempt to authenticate the user. Feedback will be provided after the request is made
     * through the registered {@code AuthenticationStateListener}.
     *
     * This method will opt the user back in right away, and is the only way to do so.
     *
     * If the authentication request
     * fails it will be retried automatically upon change in connection status or "retry" button
     * click unless the autoRetryAuthenticationOnConnected flag is set to false.
     *
     * @param idp {@code IDP} for the user being authenticated.
     *
     * @return True if the authentication request was made, and false otherwise.
     *
     * @throws ConciergeUninitializedException if {@link FlybitsConcierge#initialize(int)} has not been called.
     */
    public synchronized boolean authenticate(@NonNull IDP idp) throws ConciergeUninitializedException {
        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        }

        this.idp = idp; // store the IDP in the event the "user" is opted out and would like to opt in

        if (isAuthenticated() || authenticating) {
            return false;
        }

        authenticating = true;

        broadcastAuthenticationState(AuthenticationStatusListener.AUTHENTICATION_STARTED);

        authenticationRequested = true;

        flybitsManager.connect(idp, conciergeConnectionResultCallback);

        return true;
    }

    /**
     * Ends the authenticated user session. Cannot be performed offline.
     *
     * @param callback {@code BasicResultCallback} that will be notified about the status of the logout request.
     *
     * @throws ConciergeUninitializedException if the concierge has not been initialized.
     */
    public void logOut(final BasicResultCallback callback) throws ConciergeUninitializedException {
        if (!initialized) {
            throw new ConciergeUninitializedException();
        }else if (!isAuthenticated() || authenticating || flybitsManager == null){
            callback.onException(new ConciergeUnauthenticatedException());
        } else {
            flybitsManager.disconnect(new BasicResultCallback() {
                @Override
                public void onSuccess() {
                    authenticationRequested = false;
                    unregisterNetworkReceiver();
                    if (callback != null){
                        callback.onSuccess();
                    }
                }

                @Override
                public void onException(@NonNull FlybitsException exception) {
                    if (callback != null){
                        callback.onException(exception);
                    }
                }
            });
        }
    }

    /**
     * Unauthenticates the FlybitsConcierge without calling logOut(). This should only
     * be used if the JWT token expires and there's no way to log out but the application
     * still needs to be aware that the session expires.
     *
     * @param e The cause of the unauthenticate method call.
     *
     * @throws ConciergeUninitializedException if the concierge has not been initialized.
     *
     * @return Whether unauthenticated successfully.
     *
     * @throws ConciergeUninitializedException if initialize wasn't called.
     */
    public boolean unauthenticateWithoutLogout(FlybitsException e) throws ConciergeUninitializedException {
        if (!initialized) {
            throw new ConciergeUninitializedException();
        }else if (authenticating) {
            return false;
        } else {
            SharedElementsFactory.INSTANCE.get(getContext()).setJWTToken(""); //So that isAuthenticated() returns false
            authenticationRequested = false;
            unregisterNetworkReceiver();
            broadcastAuthenticationError(e);
            return true;
        }
    }

    /**
     * Set the {@code OptedStateChangeListener} that will be notified if the user opts out.
     *
     * @param listener {@code OptedStateChangeListener} that will be notified when the user opts out.
     *
     * @throws ConciergeUninitializedException if initialize wasn't called.
     *
     */
    public void registerOptedStateChangeListener(@NonNull OptedStateChangeListener listener) throws ConciergeUninitializedException {

        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        }

        synchronized (optedStateChangeListeners) {
            optedStateChangeListeners.add(listener);
        }
    }

    /**
     * Unregister {@code OptedStateChangeListener}.
     *
     * @param listener {@link OptedStateChangeListener} to be registered.
     *
     * @throws ConciergeUninitializedException if {@link FlybitsConcierge} is uninitialized.
     *
     */
    public void unregisterOptedStateChangeListener(@NonNull OptedStateChangeListener listener) throws ConciergeUninitializedException {
        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        }

        synchronized (optedStateChangeListeners) {
            optedStateChangeListeners.remove(listener);
        }
    }

    private void broadcastOptedInStateChange(boolean optedIn) {
        synchronized (optedStateChangeListeners) {
            Set<OptedStateChangeListener> copy = new HashSet<>(optedStateChangeListeners);
            for (OptedStateChangeListener listener : copy) {
                listener.onOptedStateChange(optedIn);
            }
        }
    }

    /**
     * Find out whether the user is currently opted in.
     *
     * @param basicResultCallback Callback where the result or any errors will be passed.
     *
     * @throws ConciergeUninitializedException if concierge isn't initialized.
     */
    public void isOptedIn(ObjectResultCallback<Boolean> basicResultCallback) throws ConciergeUninitializedException {
        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        } else if (!isAuthenticated()) {
            basicResultCallback.onException(new ConciergeUnauthenticatedException());
        } else {
            User.getSelf(getContext(), new ObjectResultCallback<User>() {
                @Override
                public void onSuccess(User user) {
                    basicResultCallback.onSuccess(user.isOptedIn());
                }

                @Override
                public void onException(@NotNull FlybitsException e) {
                    basicResultCallback.onException(e);
                }
            });
        }
    }

    /**
     * Find out whether the user is currently opted in using the local state of the SDK.
     * Callbacks will be invoked on main thread, logic will execute on worker thread.
     *
     * @param basicResultCallback Callback where the result or any errors will be passed.
     *
     */
    public void isOptedInLocal(ObjectResultCallback<Boolean> basicResultCallback) {
        Executors.newSingleThreadExecutor().execute(() ->
                isOptedInLocal(basicResultCallback, new Handler(Looper.getMainLooper())));

    }

    /**
     * Find out whether the user is currently opted in using the local state of the SDK.
     * Runs on the UI thread.
     *
     * @param basicResultCallback Callback where the result or any errors will be passed.
     * @param handler The handler that the callbacks will be invoked in.
     *
     * @throws ConciergeUninitializedException if concierge isn't initialized.
     *
     */
    public void isOptedInLocal(ObjectResultCallback<Boolean> basicResultCallback, Handler handler) throws ConciergeUninitializedException{
        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        } else if (!isAuthenticated()) {
            handler.post(() -> basicResultCallback.onException(new ConciergeUnauthenticatedException()));
        } else {
            Context context = getContext();
            if (context != null) {
                User user = commonsDatabase.userDao().getActiveUser();
                handler.post(() -> {
                    if (user == null) {
                        basicResultCallback.onException(new FlybitsException("User not present, make sure you're authenticated."));
                    } else {
                        basicResultCallback.onSuccess(user.isOptedIn());
                    }
                });
            } else {
                handler.post(() -> basicResultCallback.onException(new FlybitsException("Context null")));
            }

        }
    }

    /**
     * Opts the user out of the {@code FlybitsConcierge}. The user needs to be authenticated
     * to be opted out. After opting out the user will remain authenticated but data retrieval
     * and processing will be stopped until {@code FlybitsConcierge#optIn()} is called.
     *
     * @param callback {@code BasicResultCallback} that will be notified about whether the opt-out request succeeded
     * or what kind of failure occurred.
     *
     * @throws ConciergeUninitializedException if {@link FlybitsConcierge#initialize(int)} has not been called.
     */
    public void optOut(@NonNull final BasicResultCallback callback) throws ConciergeUninitializedException {
        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        } else if (!isAuthenticated() || isAuthenticating() || flybitsManager == null) {
            callback.onException(new ConciergeUnauthenticatedException());
        } else {
            flybitsManager.optOut(new BasicResultCallback() {
                @Override
                public void onSuccess() {
                    broadcastOptedInStateChange(false);

                    Context context = getContext();
                    if (context != null) {
                        InternalPreferences.saveSurveyDone(context,false);
                        InternalPreferences.saveOnBoardingDone(context, false);
                    }
                    callback.onSuccess();
                }

                @Override
                public void onException(@NotNull FlybitsException e) {
                    callback.onException(e);
                }
            });
        }
    }

    private boolean unregisterNetworkReceiver() {
        if (networkReceiverRegistered) {
            try {
                getContext().unregisterReceiver(networkReceiver);
                networkReceiverRegistered = false;
                return true;
            } catch(Exception e) {
                Logger.exception("FlybitsConcierge.unregisterNetworkReceiver()",e);
                return false;
            }
        } else {
            return false;
        }
    }

    private boolean registerNetworkReceiver() {
        //Register receiver responsible for auto authenticating in the future once connection is established
        if (!networkReceiverRegistered && isAutoRetryAuthenticationOnConnectedEnabled()) {
            IntentFilter intentFilter = new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION);
            try {
                getContext().registerReceiver(networkReceiver,intentFilter);
                networkReceiverRegistered = true;
                return true;
            } catch(Exception e) {
                Logger.exception("FlybitsConcierge.registerNetworkReceiver()",e);
                return false;
            }
        } else {
            return false;
        }
    }

    /**
     * Show the Concierge to the user.
     *
     * @param mode Mode of display, either OVERLAY or NEW_ACTIVITY(recommended).
     *
     * @throws IllegalArgumentException if mode is invalid.
     *
     * @throws ConciergeUninitializedException if initialize hasn't been called.
     *
     * @deprecated Please use {@link FlybitsConcierge#show(DisplayConfiguration)}.
     */
    @Deprecated
    public void show(ShowMode mode) throws IllegalArgumentException, ConciergeUninitializedException {
        DisplayConfiguration displayConfiguration = new DisplayConfiguration(
                ConciergeFragment.MenuType.MENU_TYPE_APP_BAR,
                mode,
                true
        );
        show(displayConfiguration);
    }

    /**
     * Show the Concierge to the user.
     *
     * @param displayConfiguration {@link DisplayConfiguration} to show the Concierge with.
     *
     * @throws IllegalArgumentException if mode is invalid.
     *
     * @throws ConciergeUninitializedException if initialize hasn't been called.
     */
    public void show(DisplayConfiguration displayConfiguration) throws IllegalArgumentException, ConciergeUninitializedException {

        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        }

        Context context = getContext();
        Intent activityIntent = null;

        switch (displayConfiguration.getShowMode()) {
            case NEW_ACTIVITY: {
                activityIntent = new Intent(context, ConciergeActivity.class);
                activityIntent.putExtra(ConciergeActivity.ARG_DISPLAY_CONFIGURATION, displayConfiguration);
                break;
            }
            case OVERLAY: {
                activityIntent = new Intent(context, ConciergePopupActivity.class);
                activityIntent.putExtra(ConciergePopupActivity.ARG_DISPLAY_CONFIGURATION, displayConfiguration);
                break;
            }
            case EXTERNAL:
                //this does nothing.
            default:
                throw new IllegalArgumentException("Invalid ShowMode, use NEW_ACTIVITY or OVERLAY.");
        }

        activityIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        context.startActivity(activityIntent);
    }

    /**
     * Show the Concierge Notifications to the user.
     *
     * When displaying a push in {@link ShowMode}.EXTERNAL, if the push has no content or URL within it
     * this method will return doing nothing since there's nothing to show. Other {@link ShowMode}s
     * will launch the {@link NotificationsActivity}.
     *
     * @param mode Mode of display, either OVERLAY or NEW_ACTIVITY(recommended).
     * @param push Push notification to be displayed within the scope of the concierge.
     *
     * @throws ConciergeUninitializedException if mode is invalid or {@link FlybitsConcierge#initialize(int)}
     * has not been called.
     *
     * @throws IllegalArgumentException if mode is invalid.
     *
     * @deprecated Please use {@link FlybitsConcierge#showPush(DisplayConfiguration, Push)}.
     */
    @Deprecated
    public void showPush(ShowMode mode, Push push) throws ConciergeUninitializedException, IllegalArgumentException {
        DisplayConfiguration displayConfiguration = new DisplayConfiguration(
                ConciergeFragment.MenuType.MENU_TYPE_APP_BAR,
                mode,
                true
        );
        showPush(displayConfiguration, push);
    }

    /**
     * Show the Concierge Notifications to the user.
     *
     * When displaying a push in {@link ShowMode}.EXTERNAL, if the push has no content or URL within it
     * this method will return doing nothing since there's nothing to show. Other {@link ShowMode}s
     * will launch the {@link NotificationsActivity}.
     *
     * @param displayConfiguration {@link DisplayConfiguration} to show the Concierge with.
     * @param push Push notification to be displayed within the scope of the concierge.
     *
     * @throws ConciergeUninitializedException if mode is invalid or {@link FlybitsConcierge#initialize(int)}
     * has not been called.
     *
     * @throws IllegalArgumentException if mode is invalid.
     *
     */
    public void showPush(DisplayConfiguration displayConfiguration, Push push) throws ConciergeUninitializedException, IllegalArgumentException {
        Intent intent = getShowPushIntent(displayConfiguration, push);
        if (intent != null) {
            getContext().startActivity(intent);
        }
        pushAnalytics.trackEngaged(push, System.currentTimeMillis());
    }

    /**
     * Get intent that will be used to launch an activity responsible for displaying
     * a push notification. Use this method over {@link FlybitsConcierge#showPush(ShowMode, Push)}
     * if you want more control over how the activity is started.
     *
     * @param mode Mode of display, either OVERLAY or NEW_ACTIVITY(recommended).
     * @param push Push notification to be displayed within the scope of the concierge.
     *
     * @return Intent that can be used to launch an activity responsible for displaying a push
     * notification. Null if no intent could be produced because there's no content to show.
     *
     * @throws ConciergeUninitializedException if mode is invalid or {@link FlybitsConcierge#initialize(int)}
     * has not been called.
     *
     * @throws IllegalArgumentException if mode is invalid.
     *
     * @throws ConciergeUninitializedException if initialize hasn't been called.
     *
     * @deprecated Please use {@link FlybitsConcierge#getShowPushIntent(DisplayConfiguration, Push)}.
     */
    @Deprecated
    @Nullable
    public Intent getShowPushIntent(ShowMode mode, Push push) throws ConciergeUninitializedException, IllegalArgumentException {
        DisplayConfiguration displayConfiguration = new DisplayConfiguration(
                ConciergeFragment.MenuType.MENU_TYPE_APP_BAR,
                mode,
                true
        );
        return getShowPushIntent(displayConfiguration, push);
    }

    /**
     * Get intent that will be used to launch an activity responsible for displaying
     * a push notification. Use this method over {@link FlybitsConcierge#showPush(ShowMode, Push)}
     * if you want more control over how the activity is started.
     *
     * @param displayConfiguration {@link DisplayConfiguration} to show the Concierge with.
     * @param push Push notification to be displayed within the scope of the concierge.
     *
     * @return Intent that can be used to launch an activity responsible for displaying a push
     * notification. Null if no intent could be produced because there's no content to show.
     *
     * @throws ConciergeUninitializedException if mode is invalid or {@link FlybitsConcierge#initialize(int)}
     * has not been called.
     *
     * @throws IllegalArgumentException if mode is invalid.
     *
     * @throws ConciergeUninitializedException if initialize hasn't been called.
     */
    @Nullable
    public Intent getShowPushIntent(DisplayConfiguration displayConfiguration, Push push) throws ConciergeUninitializedException, IllegalArgumentException {

        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        }

        Context context = getContext();
        Intent intent;

        boolean isViewablePush = notificationContentRepository.getContentId(push) != null
                || notificationContentRepository.getUrl(push) != null;

        switch (displayConfiguration.getShowMode()) {
            case NEW_ACTIVITY: {
                intent = new Intent(context, ConciergeActivity.class);
                intent.putExtra(ConciergeActivity.ARG_DISPLAY_CONFIGURATION, displayConfiguration);
                intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

                if (isViewablePush){
                    intent.putExtra(ConciergeConstants.PUSH_EXTRA, push);
                }

                break;
            }
            case OVERLAY: {
                intent = new Intent(context, ConciergePopupActivity.class);
                intent.putExtra(ConciergePopupActivity.ARG_DISPLAY_CONFIGURATION, displayConfiguration);
                intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

                if (isViewablePush) {
                    intent.putExtra(ConciergeConstants.PUSH_EXTRA, push);
                }

                break;
            }
            /*We go straight to NotificationsActivity so that when the user presses back
             * they end up in the client app which is likely showing the concierge fragment*/
            case EXTERNAL:
                if (isViewablePush) {
                    intent = new Intent(context, NotificationsActivity.class);
                    intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    intent.putExtra(ConciergeConstants.PUSH_EXTRA, push);
                } else {
                    intent = null;
                }
                break;
            default:
                throw new IllegalArgumentException("Invalid ShowMode, use NEW_ACTIVITY, OVERLAY, or EXTERNAL.");

        }

        return intent;
    }

    /**
     * If set to false the {@code FlybitsConcierge} will not retry authentication on change in connection if authentication failed previously.
     * It is set to true by default, this is the recommended setting.
     *
     * @param enabled Whether to retry authentication on internet connection state change if authentication failed previously.
     *
     * @throws ConciergeUninitializedException if initialize has not been called.
     */
    public void setAutoRetryAuthenticationOnConnected(boolean enabled) throws ConciergeUninitializedException {
        if (initialized) {
            InternalPreferences.setAutoRetryAuthOnConnected(getContext(), enabled);
        } else {
            throw new ConciergeUninitializedException();
        }
    }

    /**
     *
     * @return Whether the {@code FlybitsConcierge} will retry authentication on change in connection
     * if authentication failed previously. Will return true by default if flag was never altered.
     *
     * @throws ConciergeUninitializedException if initialize has not been called.
     */
    public boolean isAutoRetryAuthenticationOnConnectedEnabled() throws ConciergeUninitializedException {
        if (initialized) {
            return InternalPreferences.isAutoRetryAuthOnConnectedEnabled(getContext());
        } else {
            throw new ConciergeUninitializedException();
        }
    }

    /**
     *
     * @return {@code ConciergeConfiguration} that was set during initialization.
     */
    public ConciergeConfiguration getConfiguration()
    {
        return currentConfig;
    }

    /**
     * Enable push messaging using provided token.
     *
     * @param token Token to be used in enabling the push messaging.
     *
     * @throws ConciergeUninitializedException if initialize has not been called.
     */
    public void enablePushMessaging(String token) throws ConciergeUninitializedException {
        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        } else if (!isAuthenticated()) {
            InternalPreferences.savePushToken(getContext(), token);
            Logger.setTag(FlybitsConcierge.class.getSimpleName()).d("enablePushMessaging() not authenticated, saving token locally");
        } else {
            Handler handler = new Handler(Looper.getMainLooper());
            PushManager.enablePush(getContext(), token, new HashMap<String, String>(), null, handler);
            Logger.setTag(FlybitsConcierge.class.getSimpleName()).d("enablePushMessaging() push enabled");
        }
    }

    private boolean enablePushMessagingUsingLocalToken(){
        Handler handler = new Handler(Looper.getMainLooper());
        String token = InternalPreferences.pushToken(getContext());
        if (token != null) {
            PushManager.enablePush(getContext(), token, new HashMap<String, String>(), null, handler);
            return true;
        } else {
            return false;
        }
    }

    /**
     *
     * @return Whether the {@code FlybitsConcierge} is currently authenticated.
     *
     * @throws ConciergeUninitializedException if {@link FlybitsConcierge#initialize(int)}  has not been called.
     */
    public boolean isAuthenticated() throws ConciergeUninitializedException {
        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        } else {
            return !SharedElementsFactory.INSTANCE.get(getContext()).getSavedJWTToken().isEmpty();
        }
    }

    /**
     * @return Whether the {@code FlybitsConcierge} is currently in the process of attempting to authenticate.
     *
     * @throws ConciergeUninitializedException if {@link FlybitsConcierge#initialize(int)}  has not been called.
     */
    public boolean isAuthenticating() throws ConciergeUninitializedException {
        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        } else {
            return authenticating;
        }
    }

    /**
     * Schedules a {@link androidx.work.Worker} to retrieve content for the authorized user.
     * Worker will wait until a connection is established prior to executing.
     *
     * Preloading content will often result in faster loading times in the Concierge but this
     * highly depends on how many differences exist between the state of the model locally and on the Flybits server.
     *
     * @return false if user hasn't been authenticated, true otherwise.
     *
     * @throws ConciergeUninitializedException if {@link FlybitsConcierge#initialize(int)}  has not been called.
     */
    public boolean preloadContent() throws ConciergeUninitializedException {
        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        } else if (!isAuthenticated()) {
            return false;
        }

        WorkManager workManager = WorkManager.getInstance();

        Constraints workConstraints = new Constraints.Builder().setRequiredNetworkType(NetworkType.CONNECTED)
                .build();
        final WorkRequest workRequest = new OneTimeWorkRequest.Builder(PreloadingWorker.class).setConstraints(workConstraints)
                .build();

        //Worker responsible for preloading
        workManager.enqueue(workRequest);

        return true;
    }

    /**
     * Register {@link FlybitsViewProvider} that will be used to provide the functionality
     * and visuals for different types of content templates retrieved from the Flybits server.
     *
     * Ideally this method should be called prior to the FlybitsConcierge show() method being called
     * or the {@link ConciergeFragment} being displayed. Doing otherwise may result in certain content types
     * not appearing in the feed.
     *
     * For example if you want to display a completely separate content type in the SDK
     * that follows a structure different from anything that already exists in the Flybits Experience Studio.
     *
     * @param flybitsViewProvider FlybitsViewProvider to be registered
     *
     * @throws ConciergeUninitializedException if {@link FlybitsConcierge#initialize(int)}  has not been called.
     */
    public void registerFlybitsViewProvider(FlybitsViewProvider flybitsViewProvider) throws ConciergeUninitializedException {
        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        } else {
            flybitsViewProviders.put(flybitsViewProvider.getContentType(), flybitsViewProvider);
        }
    }

    /**
     * Retrieve list of registered {@link FlybitsViewProvider}
     *
     * @return list of registered {@link FlybitsViewProvider}
     *
     * @throws ConciergeUninitializedException if {@link FlybitsConcierge#initialize(int)}  has not been called.
     */
    @Override
    public Collection<FlybitsViewProvider> getFlybitsViewProviders() throws ConciergeUninitializedException {
        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        } else {
            return flybitsViewProviders.values();
        }
    }

    /**
     * Retrieve list of supported content types
     *
     * @return list of content types for which a {@link FlybitsViewProvider} is registered
     *
     * @throws ConciergeUninitializedException if {@link FlybitsConcierge#initialize(int)}  has not been called.
     */
    @Override
    public Set<String> getViewProviderSupportedContentTypes() {
        return flybitsViewProviders.keySet();
    }

    /**
     * Retrieve flybits view provider with the provided content type.
     *
     * @param contentType Content type to search for.
     *
     * @return {@link FlybitsViewProvider} with the provided content type. Null if none found.
     *
     * @throws ConciergeUninitializedException if {@link FlybitsConcierge#initialize(int)}  has not been called.
     */
    @Nullable
    @Override
    public FlybitsViewProvider getFlybitsViewProvider(String contentType) throws ConciergeUninitializedException {
        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        } else {
            return flybitsViewProviders.get(contentType);
        }
    }

    /**
     * Start {@link FlybitsContextPlugin}.
     *
     * Make sure that all the necessary permissions are defined in your AndroidManifest
     * and have been requested from the user prior to calling this method.
     *
     * Data will be sent to the server periodically with the timeToUploadContext value defined
     * in the resource xml file. If none is provided it will use 10 minutes by default.
     *
     * For access to plugins available by default that include tracking
     * location, battery, fitness, weather and more, see {@link FlybitsContextPlugin.Builder}
     * and {@link com.flybits.context.ReservedContextPlugin}.
     *
     * @param contextPlugin {@link FlybitsContextPlugin} to be registered.
     *
     * @throws ConciergeUninitializedException if {@link FlybitsConcierge#initialize(int)}  has not been called.
     */
    public void startContextPlugin(FlybitsContextPlugin contextPlugin) throws ConciergeUninitializedException {
        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        }
        ContextManager.start(getContext(), contextPlugin);
    }

    /**
     * Stop {@link FlybitsContextPlugin} from running.
     *
     * It is recommended to stop all context plugins prior to logging out.
     *
     * @param contextPlugin {@link FlybitsContextPlugin} to be stopped.
     *
     * @throws ConciergeUninitializedException if {@link FlybitsConcierge#initialize(int)}  has not been called.
     */
    public void stopContextPlugin(FlybitsContextPlugin contextPlugin) throws ConciergeUninitializedException {
        if (!isInitialized()) {
            throw new ConciergeUninitializedException();
        }
        ContextManager.stop(getContext(), contextPlugin);
    }
}