package com.flybits.concierge.services

import android.app.Notification
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.support.v4.app.NotificationCompat
import com.flybits.android.push.PushManager
import com.flybits.android.push.analytics.PushAnalytics
import com.flybits.android.push.models.Push
import com.flybits.android.push.models.PushAction
import com.flybits.commons.library.deserializations.IDeserializer
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.ConciergeConstants
import com.flybits.concierge.FlybitsConcierge
import com.flybits.concierge.InternalPreferences
import com.flybits.concierge.Utils
import com.google.firebase.messaging.FirebaseMessagingService
import com.google.firebase.messaging.RemoteMessage

/**
 * Extend this class to automate the handling of Flybits push notifications.
 *
 * If push notifications are already being handled in a service in such a way that subclassing isn't possible
 * then see the methods found in the companion object for this class. Also, make sure to implement
 * [ConciergeIntentCreator] to be able to use `handleRemoteMessage`.
 */
abstract class ConciergeMessagingService : FirebaseMessagingService(), ConciergeIntentCreator {

    override fun onMessageReceived(remoteMessage: RemoteMessage?) {
        super.onMessageReceived(remoteMessage)

        Logger.setTag(ConciergeMessagingService::class.java.simpleName).d("onMessageReceived(), data: ${remoteMessage?.data}")

        if (remoteMessage != null) {
            try {
                val push = PushManager.parsePushNotification<IDeserializer<*>>(this, remoteMessage.data)
                push?.let {
                    val pendingIntent = this.onCreatePendingIntent(onCreateIntent(it), it)
                    showNotification(onCreateNotification(it, pendingIntent), it.id, PushAnalytics(applicationContext)
                            , getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager?)
                } ?: onNonFlybitsPushReceived(remoteMessage)
            } catch (e: FlybitsException) {
                Logger.exception("ConciergeMessagingService.onMessageReceived()", e)
            }
        }
    }

    /**
     * This method will be invoked whenever a non Flybits push notification is received.
     *
     * @param remoteMessage The [RemoteMessage] associated with the non Flybits push notification.
     */
    abstract fun onNonFlybitsPushReceived(remoteMessage: RemoteMessage)

    /**
     * @return id of the icon resource that will be displayed with notifications.
     */
    abstract fun getNotificationIconRes(): Int

    /**
     * @param push The [Push] provided to extract information in order to construct the notification.
     * @return id of already existing notification channel.
     *
     * Please Note : The Notification Channel creation is NOT required for devices with API less than 26,
     * notifications would be displayed even if non existing channel id is returned.
     * and
     * The Notification Channel creation is required for devices with API more than or equal to 26,
     * notifications would NOT be displayed if non existing channel id is returned.
     *
     */
    abstract fun getNotificationChannelId(push: Push): String

    override fun onCreatePendingIntent(intent: Intent, push: Push): PendingIntent {
        return createPendingIntent(applicationContext, push, intent)
    }

    override fun onCreateIntent(push: Push): Intent {
        return createIntent(applicationContext, push)
    }

    override fun onNewToken(recentToken: String?) {
        super.onNewToken(recentToken)

        Logger.setTag(ConciergeMessagingService::class.java.simpleName).d("onNewToken() token: $recentToken")

        if (recentToken != null) {
            FlybitsConcierge.with(applicationContext).enablePushMessaging(recentToken)
        }
    }

    /**
     * @param push The [Push] provided to construct the notification.
     * @param pendingIntent The Pending intent provided to add information on interaction with notification.
     * @return notification.
     */
    override fun onCreateNotification(push: Push, pendingIntent: PendingIntent): Notification {
        return createNotification(applicationContext, getNotificationIconRes(), getNotificationChannelId(push), push, pendingIntent)
    }

    /**
     * Methods in this object are purposely made globally available to accommodate for
     * applications that are handling notifications in such a way where the [ConciergeMessagingService]
     * cannot be subclassed.
     */
    companion object {

        /**
         * This method will create the notification with its properties.
         *
         * @param context The Context provided for system to create the notification.
         * @param notificationIcon The Notification icon provided to display the icon in notification.
         * @param notificationChannel The notification channel provided to display notifications for API >= 26(required).
         * @param push The [Push] provided to extract information in order to construct the notification.
         * @param pendingIntent The Pending intent provided to add information on interaction with notification.
         */
        fun createNotification(context: Context, notificationIcon: Int, notificationChannel: String, push: Push,
                               pendingIntent: PendingIntent): Notification {
            return NotificationCompat.Builder(context, notificationChannel).setAutoCancel(true)
                    .setContentTitle(push.title)
                    .setContentText(push.message)
                    .setSmallIcon(notificationIcon)
                    .setContentIntent(pendingIntent)
                    .setStyle(NotificationCompat.BigTextStyle().bigText(push.message))
                    .build()
        }

        /**
         * This method will show the notification in the notification trey.
         *
         * @param notification The notification provides the structure of notification to be displayed .
         * @param id The [Push] id provided for notification manager to notify the system about the notification.
         * @param analytics The [PushAnalytics] will track the push analytics.
         * @param notificationManager The notification manager used for notifying the system to show the notification.
         *
         * @return true if notification manager is not null
         */
        fun showNotification(notification: Notification, id: String, analytics: PushAnalytics
                             , notificationManager: NotificationManager?): Boolean {
            notificationManager?.notify(id.hashCode(), notification) ?: return false
            analytics.trackViewed(id)
            return true
        }

        /**
         * This method will handle all of the logic related to displaying, tracking, and interacting
         * with the push notification contained in `remoteMessage`.
         *
         * @param remoteMessage The [RemoteMessage] associated with the non Flybits push notification
         * @param context The Context provided for system to create the notification.
         * @param pushAnalytics The [PushAnalytics] that will track the push analytics.
         * @param defaultPush The optional push provided to extract information in order to construct the notification.
         * @param conciergeIntentCreator the conciergeIntentCreator object provided.
         * Will be instantiated by default if not provided.
         */
        @JvmStatic
        @Throws(FlybitsException::class)
        fun handleRemoteMessage(context: Context, remoteMessage: RemoteMessage
                                , conciergeIntentCreator: ConciergeIntentCreator
                                , defaultPush: Push? = PushManager.parsePushNotification<IDeserializer<*>>(context, remoteMessage.data)
                                , pushAnalytics: PushAnalytics = PushAnalytics(context)): Boolean {

            defaultPush?.let {
                when (it.action) {

                    //Silent push handling
                    PushAction.ADDED -> broadcastIntent(context, ConciergeConstants.BROADCAST_CONTENT_ADD)
                    PushAction.REMOVED -> broadcastIntent(context, ConciergeConstants.BROADCAST_CONTENT_REMOVE)
                    PushAction.UPDATED -> broadcastIntent(context, ConciergeConstants.BROADCAST_CONTENT_UPDATE)
                    PushAction.RULE_UPDATED -> broadcastIntent(context, ConciergeConstants.BROADCAST_RULE_UPDATE)
                    PushAction.STATUS_CHANGED -> broadcastIntent(context, ConciergeConstants.BROADCAST_RULE_STATE)

                    PushAction.CUSTOM -> {
                        val pushIntent = conciergeIntentCreator.onCreateIntent(it)
                        val pendingIntent = conciergeIntentCreator.onCreatePendingIntent(pushIntent, it)
                        val notification = conciergeIntentCreator.onCreateNotification(it, pendingIntent)
                        return showNotification(notification, it.id, pushAnalytics
                                , context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager?)
                    }
                    // unknown + other unimplemented values
                    else -> {
                        throw FlybitsException("Received unhandled flybits push notification action ${it.action}")
                    }
                }

                return true //push was handled properly

            } ?: return false //push is null meaning its not flybits

        }

        /**
         * Create [PendingIntent] for the associated [Push].
         *
         * @param context Context of the application.
         * @param push [Push] that will be used in constructing the PendingIntent.

         * @return The created [PendingIntent].
         */
        @Throws(FlybitsException::class)
        fun createPendingIntent(context: Context, push: Push, intent: Intent): PendingIntent {
            return PendingIntent.getActivity(context
                    , push.id.hashCode(), intent, PendingIntent.FLAG_UPDATE_CURRENT)
        }

        /**
         * Create [Intent] which will contain the launcher activity and the passed [Push].
         *
         * @param context Context of the application.
         * @param push [Push] that will be inserted into the Intent.
         *
         * @return The created [Intent].
         */
        @Throws(FlybitsException::class)
        fun createIntent(context: Context, push: Push): Intent {
            val activityClass = Utils.launcherActivity(context)
                    ?: throw FlybitsException("No launcher activity found!")
            return Intent(context, activityClass).apply {
                flags = Intent.FLAG_ACTIVITY_CLEAR_TOP or
                        Intent.FLAG_ACTIVITY_SINGLE_TOP or
                        Intent.FLAG_ACTIVITY_NEW_TASK
                putExtra(ConciergeConstants.PUSH_EXTRA, push)
            }
        }

        /**
         * Save the push token locally.
         */
        @JvmStatic
        fun savePushToken(recentToken: String, context: Context) {
            InternalPreferences.savePushToken(context, recentToken)
        }

        private fun broadcastIntent(context: Context, action: String) {
            val intent = Intent()
            intent.action = action
            context.sendBroadcast(intent)
        }
    }
}
