package com.flybits.concierge

import android.content.res.XmlResourceParser
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.concierge.ConciergeConfiguration.Companion.DEFAULT_TIME_TO_UPLOAD_CONTEXT
import com.flybits.concierge.ConciergeConfiguration.Companion.KEY_TIME_TO_UPLOAD_CONTEXT
import org.xmlpull.v1.XmlPullParserException
import java.io.IOException

/**
 *  Parser that extracts configuration fields from the xml file and builds a conciergeConfiguration object with
 *  the fields set
 */
internal object ConciergeConfigurationXMLParser {

    private const val PROJECT_ID = "projectID"
    private const val GATEWAY_URL = "gatewayUrl"
    private const val PRIVACY_URL = "privacyPolicyUrl"
    private const val TNC_URL = "tncUrl"
    private const val OPT_IN_TEXT = "optInTermsOfService"

    fun generateConfigurationFromXML(xmlResourceParser: XmlResourceParser): ConciergeConfiguration {

        var timeToUploadContext: Long
        val conciergeTag = "concierge"

        lateinit var conciergeConfiguration: ConciergeConfiguration

        try {

            var eventType = xmlResourceParser.eventType
            lateinit var conciergeConfigurationBuilder: ConciergeConfiguration.Builder

            while (eventType != XmlResourceParser.END_DOCUMENT) {

                when (eventType) {
                    XmlResourceParser.START_TAG -> {
                        when (xmlResourceParser.name) {

                            conciergeTag -> {
                                // extract projectID, tncUrl, privacyPolicyUrl, gatewayUrl and timeToUploadContext

                                timeToUploadContext = xmlResourceParser.getAttributeIntValue(
                                    null,
                                    KEY_TIME_TO_UPLOAD_CONTEXT,
                                    DEFAULT_TIME_TO_UPLOAD_CONTEXT
                                ).toLong()

                                xmlResourceParser.getAttributeValue(null, PROJECT_ID)
                                    ?.let { projectID ->
                                        conciergeConfigurationBuilder =
                                            ConciergeConfiguration.Builder(projectID)
                                                .setTimeToUploadContext(timeToUploadContext)
                                    } ?: throw FlybitsException("Project ID must be specified")

                                xmlResourceParser.getAttributeValue(null, TNC_URL)?.let { tncUrl ->
                                    conciergeConfigurationBuilder.setTermsAndServicesRequired(tncUrl)
                                }

                                xmlResourceParser.getAttributeValue(null, PRIVACY_URL)
                                    ?.let { privacyPolicyUrl ->
                                        conciergeConfigurationBuilder.setPrivacyPolicyUrl(
                                            privacyPolicyUrl
                                        )
                                    }

                                xmlResourceParser.getAttributeValue(null, GATEWAY_URL)
                                    ?.let { gatewayUrl ->
                                        conciergeConfigurationBuilder.setGatewayUrl(
                                            gatewayUrl
                                        )
                                    }

                                xmlResourceParser.getAttributeValue(null, OPT_IN_TEXT)
                                    ?.let { optInTermsOfService ->
                                        conciergeConfigurationBuilder.setOptInTermsOfService(
                                            optInTermsOfService
                                        )
                                    }

                                conciergeConfiguration = conciergeConfigurationBuilder.build()

                            }
                        }

                    }

                    XmlResourceParser.END_TAG -> {
                        when (xmlResourceParser.name) {
                            conciergeTag -> {
                                return conciergeConfiguration
                            }
                        }
                    }

                }

                eventType = xmlResourceParser.next()
            }

            throw FlybitsException("Incorrect Concierge Configuration XML Format : No Concierge tag found")

        } catch (xmlPullParserException: XmlPullParserException) {
            throw IllegalStateException("Concierge: Cfg syntax error.")
        } catch (ioException: IOException) {
            throw IOException("Concierge: Error reading XML resource.")
        }
    }

}