package com.flybits.concierge;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.design.widget.Snackbar;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentActivity;
import android.support.v4.app.FragmentManager;
import android.support.v4.app.FragmentTransaction;
import android.view.*;
import android.widget.LinearLayout;
import android.widget.TextView;
import com.afollestad.materialdialogs.MaterialDialog;
import com.flybits.android.kernel.ContentAnalytics;
import com.flybits.android.kernel.models.Content;
import com.flybits.android.kernel.utilities.ContentParameters;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.internal.Pagination;
import com.flybits.concierge.activities.NotificationsActivity;
import com.flybits.concierge.activities.SettingsActivity;
import com.flybits.concierge.analytics.VisibilityObservable;
import com.flybits.concierge.analytics.VisibilityStateChangeListener;
import com.flybits.concierge.fragments.*;
import com.flybits.concierge.models.Onboarding;
import com.flybits.concierge.models.Survey;
import com.flybits.concierge.repository.ModelConverter;
import com.flybits.concierge.repository.content.ContentGetter;
import org.jetbrains.annotations.NotNull;

import java.util.*;

/**
 * This fragment is responsible for displaying all of the Concierge.
 *
 * For instantiating this fragment please see the newInstance method.
 */
public class ConciergeFragment extends Fragment implements AuthenticationStatusListener
        , OptedStateChangeListener, FlybitsNavigator, VisibilityObservable {

    @Override
    public void openSnackbar(@NotNull String content, int length) {
        View view = getView();
        if (view != null) {
            Snackbar.make(view, content, Snackbar.LENGTH_SHORT).show();
        }
    }

    @Override
    public void dismiss() {
        getChildFragmentManager().popBackStack();
    }

    /**
     * Enum responsible for specifying where and how the options menu will appear.
     */
    public enum MenuType {
        /**
         * Menu will appear in the app bar of the hosting activity.
         */
        MENU_TYPE_APP_BAR(0)
        ,
        /**
         * Menu will appear in the
         */
        MENU_TYPE_TAB(1);

        private int value;

        MenuType(int value){
            this.value = value;
        }

        public static MenuType fromInt(int type){
            if (type == 0){
                return MENU_TYPE_APP_BAR;
            } else if (type == 1) {
                return MENU_TYPE_TAB;
            } else {
                throw new IllegalArgumentException("Type is invalid use TYPE_TAB or TYPE_APP_BAR");
            }
        }

        public int getValue(){
            return value;
        }
    }

    public static final String INSTANCE_CURRENT_FRAGMENT = "instance_current_fragment";
    public static final int MESSAGE_TNC = 1;
    public static final int MESSAGE_ONBOARDING = 2;
    public static final String ARG_MENU_TYPE = "flybits_con_menu_type";
    public static final String ARG_SHOW_OPT_OUT = "flybits_con_show_opt_out";



    private LinearLayout lytLoader;
    private TextView txtLoaderText;
    private LinearLayout errorViewContainer;
    private View mainViewContainer;

    private Context currentContext;

    private Fragment currentFragment;

    private IConciergeFragmentCallbacks currentCallback;

    private boolean initializing = false;
    private boolean actionBarItemsVisible = true;
    private boolean menuVisible = false;
    private boolean isVisible = false;
    private boolean firstResume = true;
    private MenuType menuType = null;
    private boolean showOptOutOption = true;

    private Set<VisibilityStateChangeListener> visibilityStateChangeListeners;

    private FlybitsConcierge flybitsConcierge;

    //Messages back from child fragments
    public void childMessage(int code, Object data)
    {
        switch (code)
        {
            case MESSAGE_TNC:
                boolean result = (Boolean) data;
                if (result)
                {
                    initializeState(false);
                }
                else if (currentCallback != null)
                {
                    currentCallback.onTNCDecline();
                }
                break;
            case MESSAGE_ONBOARDING:
                initializeState(false);
                break;
        }
    }

    /**
     * Create instance of {@link ConciergeFragment}.
     *
     * @param menuType The {@link MenuType} appearance that will be used.
     * @return new instance of {@link ConciergeFragment}.
     *
     * @deprecated Please use {@link ConciergeFragment#newInstance(DisplayConfiguration)} instead.
     */
    @Deprecated
    public static ConciergeFragment newInstance(MenuType menuType)
    {
        ConciergeFragment fragment = new ConciergeFragment();
        Bundle bundle = new Bundle();
        bundle.putInt(ARG_MENU_TYPE, menuType.value);
        fragment.setArguments(bundle);
        return fragment;
    }

    /**
     * Create instance of {@link ConciergeFragment}.
     *
     * @param configuration The {@link DisplayConfiguration} for the instance being created.
     *
     * @return new instance of {@link ConciergeFragment}.
     */
    public static ConciergeFragment newInstance(DisplayConfiguration configuration)
    {
        ConciergeFragment fragment = new ConciergeFragment();
        Bundle bundle = new Bundle();
        bundle.putInt(ARG_MENU_TYPE, configuration.getMenuType().getValue());
        bundle.putBoolean(ARG_SHOW_OPT_OUT, configuration.getShowOptOutOption());
        fragment.setArguments(bundle);
        return fragment;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);

        if (savedInstanceState != null)
        {
            currentFragment = getChildFragmentManager().getFragment(savedInstanceState, INSTANCE_CURRENT_FRAGMENT);
        }
    }

    @Override
    public void onSaveInstanceState(@NonNull Bundle outState)
    {
        super.onSaveInstanceState(outState);

        if (currentFragment != null)
        {
            getChildFragmentManager().putFragment(outState, INSTANCE_CURRENT_FRAGMENT, currentFragment);
        }
        outState.putBoolean(ConciergeConstants.STATE_ERROR_PRESENT, errorViewContainer.getVisibility() == View.VISIBLE);

    }

    @Override
    public void onPause() {
        super.onPause();
        Logger.setTag("Concierge-DEBUG").d("onPause()");
        if (flybitsConcierge != null){
            flybitsConcierge.unregisterAuthenticationStateListener(this);
            flybitsConcierge.unregisterOptedStateChangeListener(this);
        }
        if (menuVisible) {
            Logger.setTag("Concierge-DEBUG").d("menuVisible");

            onVisibilityStateChange(false);
        }
    }

    @Override
    public void onResume() {
        super.onResume();
        Logger.setTag("Concierge-DEBUG").d("onResume()");
        if (flybitsConcierge != null) {
            flybitsConcierge.registerAuthenticationStateListener(this);
            flybitsConcierge.registerOptedStateChangeListener(this);

            // Call initialize the state
            initializeState(true);
        }
        if (menuVisible && !firstResume) {
            Logger.setTag("Concierge-DEBUG").d("menuVisiblie and Not First Time");
            onVisibilityStateChange(true);
        }
        firstResume = false;
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, Bundle savedInstanceState)
    {
        return inflater.inflate(R.layout.flybits_con_fragment_concierge, container, false);
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState)
    {
        super.onViewCreated(view, savedInstanceState);
        Logger.setTag("Concierge-DEBUG").d("onViewCreated");

        visibilityStateChangeListeners = Collections.synchronizedSet(new HashSet<VisibilityStateChangeListener>());

        /*Code below makes it so that the options menu appears in the app bar if that setting is
        * being used. It is hosted in the fragment since we do not know what the hosting activity
        * is. */
        setHasOptionsMenu(true);
        Bundle args = getArguments();
        if (args != null){
            menuType = MenuType.fromInt(args.getInt(ARG_MENU_TYPE, MenuType.MENU_TYPE_APP_BAR.value));
            showOptOutOption = args.getBoolean(ARG_SHOW_OPT_OUT, true);
        }

        lytLoader = view.findViewById(R.id.concierge_fragment_lytLoader);
        txtLoaderText = view.findViewById(R.id.concierge_fragment_txtLoadingText);
        errorViewContainer = view.findViewById(R.id.concierge_fragment_error_holder);
        mainViewContainer = view.findViewById(contentLayout());

        view.findViewById(R.id.concierge_fragment_retry_button).setOnClickListener(view1 -> initializeState(false));

        flybitsConcierge = FlybitsConcierge.with(currentContext);

        initializeState(false);
    }

    @Override
    public void onAttach(Context context)
    {
        super.onAttach(context);
        currentContext = context;
    }

    @Override
    public void onDetach()
    {
        super.onDetach();
    }

    /**
     * Attempts to resolve all requirements to see the main feed.
     *
     * @param skipOnboarding Skips onboarding (due to it not existing)
     */
    private void initializeState(final boolean skipOnboarding) {

        Logger.setTag("Concierge-DEBUG").d("ConciergeFragment line 294 - InitializeState");

        if (initializing) return;
        initializing = true;

        // reset the views to be in their default visibility
        errorViewContainer.setVisibility(View.GONE);
        mainViewContainer.setVisibility(View.INVISIBLE);

        //Check if authorized first
        if (flybitsConcierge.isAuthenticated()) {
            Logger.setTag("Concierge-DEBUG").d("flybitsConcierge.isAuthenticated()");
            mainViewContainer.setVisibility(View.VISIBLE);
            flybitsConcierge.isOptedInLocal(new ObjectResultCallback<Boolean>() {
                @Override
                public void onSuccess(Boolean optedIn) {
                    Logger.setTag("Concierge-DEBUG").d("Opted In Request Success");
                    flybitsConcierge.unregisterAuthenticationStateListener(ConciergeFragment.this);

                    if (!optedIn) {
                        Logger.setTag("Concierge-DEBUG").d("1");
                        // if TNCs haven't been shown yet, and the link is initialized in the config file
                        hideLoader();
                        currentFragment = OptInFragment.Companion.newInstance();
                        FragmentManager manager = getChildFragmentManager();
                        manager.beginTransaction()
                                .replace(contentLayout(), currentFragment)
                                .addToBackStack(null)
                                .commit();
                        setActionBarItemsVisibility(false);
                    } else if (!skipOnboarding && !InternalPreferences.isOnBoardingDone(currentContext)) {
                        Logger.setTag("Concierge-DEBUG").d("2");
                        // if you should show onboarding and onboarding hasn't been shown yet
                        hideLoader();
                        findContentAndTakeover(ConciergeConstants.ONBOARDING_CONTENT_TYPE);
                        InternalPreferences.saveOnBoardingDone(currentContext, true);
                        setActionBarItemsVisibility(false);
                    } else if (!InternalPreferences.isSurveyDone(currentContext)) {
                        Logger.setTag("Concierge-DEBUG").d("3");
                        // if you should show a survey
                        hideLoader();
                        findContentAndTakeover(ConciergeConstants.SURVEY_CONTENT_TYPE);
                        InternalPreferences.saveSurveyDone(currentContext, true);
                        setActionBarItemsVisibility(false);
                    } else if (!(currentFragment instanceof FeedHolderFragment)) {
                        Logger.setTag("Concierge-DEBUG").d("4");
                        // Otherwise, show CategoryFragment only if current fragment is not instance of FeedHolderFragment
                        hideLoader();
                        Bundle arguments = getArguments();
                        boolean showMoreTab = false;
                        if (arguments != null){
                            MenuType menuType = MenuType.fromInt(arguments.getInt(ARG_MENU_TYPE, MenuType.MENU_TYPE_APP_BAR.value));
                            showMoreTab =  menuType != MenuType.MENU_TYPE_APP_BAR;
                        }
                        Logger.setTag("Concierge-DEBUG").d("ConciergeFragment line 342 Creating FeedHolderFragment");
                        currentFragment = FeedHolderFragment.Companion.newInstance(showMoreTab, showOptOutOption);
                        FragmentManager manager = getChildFragmentManager();
                        manager.beginTransaction()
                                .replace(contentLayout(), currentFragment, currentFragment.getClass().getSimpleName())
                                .commit();
                        setActionBarItemsVisibility(true);
                    }
                    initializing = false;
                }

                @Override
                public void onException(@NotNull FlybitsException e) {
                    initializing = false;

                }
            });
        } else {
            //Retry authentication since we are not currently authenticated, wait for callback(AuthenticationStatusListener)'s methods to be invoked
            boolean retrySuccess = flybitsConcierge.retryAuthentication();
            if (retrySuccess){
                showLoader("");
            }
            errorViewContainer.setVisibility(View.VISIBLE);
            setActionBarItemsVisibility(false);
            initializing = false;
        }
    }

    @Override
    public void onCreateOptionsMenu(Menu menu, MenuInflater inflater) {
        Logger.setTag("Concierge-DEBUG").d("onCreateOptionsMenu()");
        onVisibilityStateChange(true);
        menuVisible = true;

        if (menuType == MenuType.MENU_TYPE_APP_BAR) { //This will happen because we need this callback for visibility now
            inflater.inflate(R.menu.menu, menu);
            menu.findItem(R.id.notifications).setVisible(actionBarItemsVisible);
            menu.findItem(R.id.settings).setVisible(actionBarItemsVisible);
        } else {
            super.onCreateOptionsMenu(menu, inflater);
        }
    }

    @Override
    public void onDestroyOptionsMenu() {
        super.onDestroyOptionsMenu();
        Logger.setTag("Concierge-DEBUG").d("onDestroyOptionsMenu()");
        onVisibilityStateChange(false);
        menuVisible = false;
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        int id = item.getItemId();
        if (id == R.id.settings) {
            Intent startSettingsIntent = new Intent(getContext(), SettingsActivity.class);
            startSettingsIntent.putExtra(SettingsActivity.SHOW_OPT_OUT_CELL, showOptOutOption);
            startActivityForResult(startSettingsIntent, SettingsActivity.REQUEST_CODE);
            return true;
        } else if (id == R.id.notifications) {
            Intent startNotificationsIntent = new Intent(getContext(), NotificationsActivity.class);
            startActivity(startNotificationsIntent);
            return true;
        }

        return super.onOptionsItemSelected(item);
    }

    private void setActionBarItemsVisibility(boolean visible){
        actionBarItemsVisible = visible;
        Activity activity = getActivity();
        if (activity != null){
            activity.invalidateOptionsMenu();
        }

    }

    /**
     * Replaces the current content to the TNC page
     */
    private void showTNCs()
    {
        currentFragment = TermsAndServicesFragment.newInstance(false);
        FragmentManager manager = getChildFragmentManager();
        manager.beginTransaction()
                .replace(contentLayout(), currentFragment)
                .commit();
    }

    /**
     * Replaces the current content to some Flybits Content Fragment (Onboarding, Surveys)
     */
    private void findContentAndTakeover(final String contentTemplateType)
    {
        ContentParameters params = new ContentParameters.Builder().setTemplateType(contentTemplateType)
                .build();

        Content.get(currentContext, params, new PagedResultCallback<Content>()
        {
            @Override
            public void onSuccess(ArrayList<Content> items, Pagination pagination)
            {
                if (getContext() == null) return;
                if (items.size() != 0)
                {
                    showContent(items.get(0));
                }
                else
                {
                    initializeState(true);
                }
            }

            @Override
            public void onException(FlybitsException exception)
            {
                initializeState(true);
                InternalPreferences.saveSurveyDone(getContext(), true);
            }

            @Override
            public void onLoadedAllItems()
            {

            }
        });
    }

    /**
     * Replaces the current content with a content.
     */
    private void showContent(Content contentToShow)
    {
        ModelConverter modelConverter = new ModelConverter(new ContentGetter(getContext()), flybitsConcierge);
        hideLoader();
        FragmentManager manager = getChildFragmentManager();
        switch (contentToShow.getType())
        {
            case ConciergeConstants.ONBOARDING_CONTENT_TYPE:
            {
                Onboarding onboarding = (Onboarding)modelConverter.contentToBaseTemplate(contentToShow);
                if (onboarding == null) {
                    initializeState(true);
                    return;
                }

                OnboardingFragment onboardingFragment = OnboardingFragment.newInstance(onboarding);

                onboardingFragment.setOnboardingListener(new OnboardingListener()
                {
                    @Override
                    public void onSkip()
                    {
                        initializeState(false);
                    }

                    @Override
                    public void onFinished()
                    {
                        initializeState(false);
                    }
                });

                currentFragment = onboardingFragment;
                manager.beginTransaction()
                        .replace(contentLayout(), currentFragment, currentFragment.getClass().getSimpleName())
                        .commit();
                break;
            }
            case ConciergeConstants.SURVEY_CONTENT_TYPE:
            {
                Survey survey = (Survey) modelConverter.contentToBaseTemplate(contentToShow);
                if (survey == null || survey.questions.getList().isEmpty()) {
                    new MaterialDialog.Builder(getActivity())
                            .title(R.string.flybits_con_error)
                            .content(R.string.flybits_con_dialog_survey_error)
                            .positiveText(android.R.string.ok)
                            .show();
                    InternalPreferences.saveSurveyDone(getContext(), true);
                    initializeState(false);
                    return;
                }

                SurveyFragment surveyFragment = SurveyFragment.newInstance(survey);

                surveyFragment.setSurveyListener(() -> {
                    initializeState(false);
                    InternalPreferences.saveSurveyDone(currentContext, true);
                });

                currentFragment = surveyFragment;
                manager.beginTransaction()
                        .replace(contentLayout(), currentFragment, currentFragment.getClass().getSimpleName())
                        .commit();
                break;
            }
            default:
                initializeState(false);
                break;
        }
    }

    @Override
    public void openFragment(@NotNull Fragment fragment, boolean addToBackStack)
    {
        FragmentActivity fragmentActivity = getActivity();
        if (fragmentActivity != null){
            FragmentTransaction transaction = getChildFragmentManager()
                    .beginTransaction();
            if (addToBackStack){
                transaction.addToBackStack(null);
            }
            transaction.replace(contentLayout(), fragment)
                    .commit();
        }

    }

    @Override
    public void openActivity(Class activity, Bundle extras) {
        Intent intent = new Intent(currentContext, activity);
        intent.putExtras(extras);
        startActivity(intent);
    }

    /**
     * Removes the current content and shows the loader
     *
     * @param title Message to display
     */
    private void showLoader(String title)
    {
        if (currentFragment != null)
        {
            getChildFragmentManager().beginTransaction()
                    .remove(currentFragment)
                    .commit();
        }
        txtLoaderText.setText(title);
        lytLoader.setVisibility(View.VISIBLE);
    }

    /**
     * Hides the loader
     */
    private void hideLoader()
    {
        txtLoaderText.setText("");
        lytLoader.setVisibility(View.GONE);
    }

    private int contentLayout() {
        return R.id.concierge_fragment_lytContent;
    }

    public void setCallback(IConciergeFragmentCallbacks callback)
    {
        currentCallback = callback;
    }

    @Override
    public void onAuthenticated() {
        initializeState(false);
    }

    @Override
    public void onAuthenticationStarted() {

    }

    @Override
    public void onAuthenticationError(FlybitsException e) {
        //Don't display error view if opt out view is already visible
        errorViewContainer.setVisibility(View.VISIBLE);
        hideLoader();
    }

    @Override
    public void onOptedStateChange(boolean optedIn) {
        if (getActivity() != null){
            initializeState(optedIn);
        }
    }

    @Override
    public boolean openUrl(@NotNull String url) {
        Activity activity = getActivity();
        if (activity == null) {
            return false;
        }

        Intent intent = new Intent(Intent.ACTION_VIEW);
        intent.setData(Uri.parse(url));
        //verify that you're able to open url, if URL is malformed then exception is thrown
        if (intent.resolveActivity(activity.getPackageManager()) != null) {
            startActivity(intent);
            return true;
        } else {
            return false;
        }
    }

    @Override
    public void openDialog(@NotNull String content, @NotNull String title) {
        new MaterialDialog.Builder(getActivity())
                .title(title)
                .content(content)
                .positiveText(android.R.string.ok)
                .dismissListener(dialogInterface -> getActivity().finish())
        .show();
    }

    public interface IConciergeFragmentCallbacks
    {
        void onTNCDecline();
    }

    @Override
    public void registerVisibilityStateChangeListener(VisibilityStateChangeListener v) {
        Logger.setTag("Concierge-DEBUG").d("ConciergeFragment line 672: registerVisibility " + v.toString());
        synchronized (visibilityStateChangeListeners) {
            visibilityStateChangeListeners.add(v);
        }
    }

    @Override
    public void unregisterVisibilityStateChangeListener(VisibilityStateChangeListener v) {
        Logger.setTag("Concierge-DEBUG").d("ConciergeFragment line 680: unregisterVisibility " + v.toString());
        synchronized (visibilityStateChangeListeners) {
            visibilityStateChangeListeners.remove(v);
        }
    }

    /**
     * @return Whether the [ConciergeFragment] is visible to the user.
     */
    public boolean isVisibleToUser() {
        return isVisible;
    }

    private void onVisibilityStateChange(boolean visibility) {
        Logger.setTag("Concierge-DEBUG").d("ConciergeFragment line 681: onVisibilityStateChange " + visibility + ", " + isVisible);
        if (isVisible != visibility) { //Only  broadcast if visibility state change occurred
            isVisible = visibility;
            Set<VisibilityStateChangeListener> listenersSet = new HashSet<>(visibilityStateChangeListeners);
            for (VisibilityStateChangeListener listener : listenersSet) {
                Logger.setTag("Concierge-DEBUG").d("ConciergeFragment line 686: onVisibilityStateChange " + listener.toString());
                listener.onVisibilityStateChange(visibility);
            }
        }
    }
}
