package com.flybits.concierge.models;

import android.arch.lifecycle.ViewModelProviders;
import android.content.Context;
import android.database.DataSetObserver;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.support.annotation.NonNull;
import android.support.v4.app.FragmentActivity;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.LinearLayout;

import com.flybits.android.kernel.models.PagedArray;
import com.flybits.commons.library.logging.Logger;
import com.flybits.concierge.R;
import com.flybits.concierge.viewmodels.TwitterViewModel;
import com.pierfrancescosoffritti.androidyoutubeplayer.utils.Utils;
import com.twitter.sdk.android.core.models.Tweet;
import com.twitter.sdk.android.tweetui.BaseTweetView;
import com.twitter.sdk.android.tweetui.CompactTweetView;
import com.twitter.sdk.android.tweetui.SearchTimeline;
import com.twitter.sdk.android.tweetui.TweetTimelineListAdapter;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public class Twitters extends BaseTemplate
{
    private static final int MAX_TWEETS = 5;

    public Twitters()
    {
    }

    public PagedArray<Twitter> twitter;

    private TwitterViewModel twitterViewModel;
    private List<String> twitterNames = new ArrayList<>();
    private String twitterNameKey;
    private LinearLayout viewContainer;
    private boolean isLoading;
    private TweetTimelineListAdapter adapter;
    private FlybitsTwitterObserver observer = new FlybitsTwitterObserver();
    private ExecutorService executorService = Executors.newSingleThreadExecutor();

    protected Twitters(Parcel in)
    {
        super(in);
        twitter = in.readParcelable(PagedArray.class.getClassLoader());
    }

    public static final Creator<Twitters> CREATOR = new Creator<Twitters>()
    {
        @Override
        public Twitters createFromParcel(Parcel in)
        {
            return new Twitters(in);
        }

        @Override
        public Twitters[] newArray(int size)
        {
            return new Twitters[size];
        }
    };

    @Override
    public int describeContents()
    {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags)
    {
        super.writeToParcel(dest,flags);
        dest.writeParcelable(twitter, flags);
    }

    public void generateTweets(@NonNull final LinearLayout container)
    {
        container.removeAllViews();

        twitterViewModel = ViewModelProviders.of((FragmentActivity) container.getContext()).get(TwitterViewModel.class);

        twitterNames.clear();
        for (Twitter names : twitter.getList())
        {
            twitterNames.add(names.username);
        }

        twitterNameKey = TextUtils.join("+", twitterNames);

        List<Tweet> tweetList = twitterViewModel.getTweetList(twitterNameKey);
        List<View> tweetViewList = tweetsToViews(container.getContext(), tweetList);

        if (tweetViewList.size() == 0)
        {
            if (isLoading)
            {
                return;
            }

            // only request tweets if the device is online
            if (!Utils.isOnline(container.getContext()))
            {
                isLoading = false;

                presentErrorView(container);

                return;
            }

            // present a loading cell
            View loadingView = LayoutInflater.from(container.getContext()).inflate(R.layout.flybits_con_item_template_twitter_loading, container, false);
            container.addView(loadingView);

            String query = "from:";
            if (twitterNames.size() > 1)
            {
                query = query + TextUtils.join(" OR from:", twitterNames);
            }
            else if (twitterNames.size() == 1)
            {
                query = query + twitterNames.get(0);
            }

            try
            {
                final SearchTimeline searchTimeline = new SearchTimeline.Builder().query(query)
                        .maxItemsPerRequest(MAX_TWEETS)
                        .build();

                // the timeline adapter fetches the data
                executorService.submit(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        isLoading = true;

                        adapter = new TweetTimelineListAdapter(container.getContext(), searchTimeline);
                        adapter.registerDataSetObserver(observer);
                    }
                });
            }
            catch (Exception e)
            {
                Logger.exception(Twitters.class.getSimpleName(), e);

                presentErrorView(container);

                return;
            }

            viewContainer = container;
        }
        else
        {
            for (View v : tweetViewList)
            {
                // when recycling, remove
                if (v.getParent() != null)
                {
                    ((LinearLayout) v.getParent()).removeAllViews();
                }

                container.addView(v);
            }
        }
    }

    private void presentErrorView(LinearLayout container)
    {
        // clear views, create error view, add view.
        container.removeAllViews();
        View failedView = LayoutInflater.from(container.getContext()).inflate(R.layout.flybits_con_item_template_twitter_failed, container, false);
        container.addView(failedView);
    }

    public class FlybitsTwitterObserver extends DataSetObserver
    {
        @Override
        public void onChanged()
        {
            super.onChanged();

            if (twitterViewModel == null)
            {
                isLoading = false;

                return;
            }

            final List<Tweet> tweetList = twitterViewModel.getTweetList(twitterNameKey);

            adapter.unregisterDataSetObserver(observer);

            for (int i = 0; i < adapter.getCount(); i++)
            {
                tweetList.add(adapter.getItem(i));
            }

            new Handler(Looper.getMainLooper()).post(new Runnable()
            {
                @Override
                public void run()
                {
                    viewContainer.removeAllViews();

                    List<View> tweetViewList = tweetsToViews(viewContainer.getContext(), tweetList);

                    for (View v : tweetViewList)
                    {
                        viewContainer.addView(v);
                    }

                    viewContainer = null;

                    isLoading = false;
                }
            });
        }

        @Override
        public void onInvalidated()
        {
            super.onInvalidated();
            isLoading = false;
        }
    }

    /**
     * Converts tweets to twitter views.
     *
     * @param context context needed to create views
     * @param tweets the tweet content
     *
     * @return A list containing the views
     */
    private List<View> tweetsToViews(Context context, List<Tweet> tweets)
    {
        List<View> views = new ArrayList<>();

        for (Tweet tweet : tweets)
        {
            BaseTweetView tv = new CompactTweetView(context, tweet, com.twitter.sdk.android.tweetui.R.style.tw__TweetLightStyle);
            views.add(tv);
        }
        
        return views;
    }
}
