package com.flybits.concierge

import com.flybits.commons.library.exceptions.FlybitsException

/**
 *  Represents Configuration fields for the Flybits Concierge
 *
 */
class ConciergeConfiguration private constructor(builder: Builder) {

    companion object {

        const val KEY_TIME_TO_UPLOAD_CONTEXT = "timeToUploadContext"
        const val DEFAULT_TIME_TO_UPLOAD_CONTEXT = 10
        const val DEFAULT_GATEWAY_URL = "https://v3.flybits.com"
    }

    var projectID: String = builder.projectID
        private set

    var tncUrl: String = builder.tncUrl
        private set

    var privacyPolicyUrl: String = builder.privacyPolicyUrl
        private set

    var timeToUploadContext: Long = builder.timeToUploadContext
        private set

    var gatewayUrl: String = builder.gatewayUrl
        private set

    init {

        if (projectID == "") {
            throw FlybitsException("Project ID must not be empty")
        }


    }

    /**
     * Builder class for ConciergeConfiguration
     *
     * @param projectID string representing projectID
     */
    class Builder(val projectID: String) {
        var tncUrl: String = ""
        var timeToUploadContext: Long = DEFAULT_TIME_TO_UPLOAD_CONTEXT.toLong()
        var privacyPolicyUrl: String = ""
        var gatewayUrl: String = DEFAULT_GATEWAY_URL

        /**
         * Set terms and Services Url for the ConciergeConfiguration object
         *
         * @param tncUrl url pointing to terms and services webpage
         */
        fun setTermsAndServicesRequired(tncUrl: String): Builder {
            this.tncUrl = tncUrl
            return this
        }


        /**
         * Set Gateway Url for the ConciergeConfiguration object
         *
         * @param gatewayUrl Gateway/Host URL to be used for communicating with Flybits servers.
         *
         * Please Note:
         * To set Region vise GatewayUrl :
         *  US = https://v3.flybits.com
         *  Canada = https://v3.canada.flybits.com
         *  Europe = https://v3-europe.flybits.com
         *  South America = https://v3-sa.flybits.com
         */
        fun setGatewayUrl(gatewayUrl: String): Builder {
            this.gatewayUrl = gatewayUrl
            return this
        }

        /**
         * Set privacyPolicyUrl for the ConciergeConfiguration object
         *
         * @param privacyPolicyUrl url pointing to privacy policy webpage
         */
        fun setPrivacyPolicyUrl(privacyPolicyUrl: String): Builder {
            this.privacyPolicyUrl = privacyPolicyUrl
            return this
        }

        /**
         *  Set the time to upload context for the ConciergeConfiguration object
         *
         *  @param timeToUploadContext The time to upload Context data to the server
         */
        fun setTimeToUploadContext(timeToUploadContext: Long): Builder {
            this.timeToUploadContext = timeToUploadContext
            return this
        }


        fun build(): ConciergeConfiguration {
            return ConciergeConfiguration(this)
        }
    }

}