package com.flybits.concierge.services

import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import com.flybits.android.push.FlybitsNotificationManager
import com.flybits.android.push.PushManager
import com.flybits.android.push.analytics.PushAnalytics
import com.flybits.android.push.models.Push
import com.flybits.android.push.models.PushAction
import com.flybits.commons.library.deserializations.IDeserializer
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.ConciergeConstants
import com.flybits.concierge.FlybitsConcierge
import com.flybits.concierge.InternalPreferences
import com.flybits.concierge.Utils
import com.google.firebase.messaging.FirebaseMessagingService
import com.google.firebase.messaging.RemoteMessage

/**
 * Extend this class to automate the handling of Flybits push notifications.
 *
 * If push notifications are already being handled in a service in such a way that subclassing isn't possible
 * then see the methods found in the companion object for this class. Also, make sure to implement
 * [ConciergeIntentCreator] to be able to use `handleRemoteMessage`.
 */
abstract class ConciergeMessagingService : FirebaseMessagingService(), ConciergeIntentCreator {

    override fun onMessageReceived(remoteMessage: RemoteMessage?) {
        super.onMessageReceived(remoteMessage)

        Logger.setTag(ConciergeMessagingService::class.java.simpleName).d("onMessageReceived()")

        if (remoteMessage != null){
            try{
                val success = handleRemoteMessage(remoteMessage, this, this, getNotificationIconRes())
                if (!success){
                    onNonFlybitsPushReceived(remoteMessage)
                }
            }catch (e: FlybitsException){
                Logger.exception("ConciergeMessagingService.onMessageReceived()", e)
            }
        }
    }

    /**
     * This method will be invoked whenever a non Flybits push notification is received.
     *
     * @param remoteMessage The [RemoteMessage] associated with the non Flybits push notification.
     */
    abstract fun onNonFlybitsPushReceived(remoteMessage: RemoteMessage)

    /**
     * @return id of the icon resource that will be displayed with notifications.
     */
    abstract fun getNotificationIconRes(): Int

    override fun onCreatePendingIntent(intent: Intent, push: Push): PendingIntent{
        return createPendingIntent(applicationContext, push, intent)
    }

    override fun onCreateIntent(push: Push): Intent {
        return createIntent(applicationContext, push)
    }

    override fun onNewToken(recentToken: String?) {
        super.onNewToken(recentToken)

        Logger.setTag(ConciergeMessagingService::class.java.simpleName).d("onNewToken() token: $recentToken")

        if (recentToken != null){
            FlybitsConcierge.with(applicationContext).enablePushMessaging(recentToken)
        }
    }

    /**
     * Methods in this object are purposely made globally available to accommodate for
     * applications that are handling notifications in such a way where the [ConciergeMessagingService]
     * cannot be subclassed.
     */
    companion object {

        /**
         * This method will handle all of the logic related to displaying, tracking, and interacting
         * with the push notification contained in `remoteMessage`.
         *
         * @param remoteMessage The [RemoteMessage] received.
         * @param context The context tied to your application.
         * @param pushAnalytics The [PushAnalytics] that will track the push analytics.
         * Will be instantiated by default if not provided.
         */
        @JvmStatic
        @Throws(FlybitsException::class)
        fun handleRemoteMessage(remoteMessage: RemoteMessage
                                , context: Context
                                , conciergeIntentCreator: ConciergeIntentCreator
                                , notificationIconRes: Int
                                , pushAnalytics: PushAnalytics = PushAnalytics(context)): Boolean {

            val push = PushManager.parsePushNotification<IDeserializer<*>>(context, remoteMessage.data)
            push?.let {
                when (it.action) {
                    //Silent push handling
                    PushAction.ADDED -> broadcastIntent(context, ConciergeConstants.BROADCAST_CONTENT_ADD)
                    PushAction.REMOVED -> broadcastIntent(context, ConciergeConstants.BROADCAST_CONTENT_REMOVE)
                    PushAction.UPDATED -> broadcastIntent(context,ConciergeConstants.BROADCAST_CONTENT_UPDATE)
                    PushAction.RULE_UPDATED -> broadcastIntent(context,ConciergeConstants.BROADCAST_RULE_UPDATE)
                    PushAction.STATUS_CHANGED -> broadcastIntent(context, ConciergeConstants.BROADCAST_RULE_STATE)

                    PushAction.CUSTOM -> {
                        //build push
                        val builder = FlybitsNotificationManager.Simplifier(
                                context,
                                it.id,
                                it.title,
                                it.message,
                                notificationIconRes)
                                .build()

                        val pushIntent = conciergeIntentCreator.onCreateIntent(push)
                        val pendingIntent = conciergeIntentCreator.onCreatePendingIntent(pushIntent, push)
                        builder.get().setContentIntent(pendingIntent)
                        builder.show()
                        pushAnalytics.trackViewed(it)
                    }
                    // unknown + other unimplemented values
                    else -> {
                        throw FlybitsException("Received unhandled flybits push notification action ${it.action}")
                    }
                }

                return true //push was handled properly

            } ?: return false //push is null meaning its not flybits

        }

        /**
         * Create [PendingIntent] for the associated [Push].
         *
         * @param context Context of the application.
         * @param push [Push] that will be used in constructing the PendingIntent.

         * @return The created [PendingIntent].
         */
        @Throws(FlybitsException::class)
        fun createPendingIntent(context: Context, push: Push, intent: Intent): PendingIntent{
            return PendingIntent.getActivity(context
            , push.id.hashCode(), intent, PendingIntent.FLAG_UPDATE_CURRENT)
        }

        /**
         * Create [Intent] which will contain the launcher activity and the passed [Push].
         *
         * @param context Context of the application.
         * @param push [Push] that will be inserted into the Intent.
         *
         * @return The created [Intent].
         */
        @Throws(FlybitsException::class)
        fun createIntent(context: Context, push: Push): Intent{
            val activityClass = Utils.launcherActivity(context) ?: throw FlybitsException("No launcher activity found!")
            return Intent(context, activityClass).apply {
                flags = Intent.FLAG_ACTIVITY_CLEAR_TOP or
                        Intent.FLAG_ACTIVITY_SINGLE_TOP or
                        Intent.FLAG_ACTIVITY_NEW_TASK
                putExtra(ConciergeConstants.PUSH_EXTRA, push)
            }
        }
        /**
         * Save the push token locally.
         */
        @JvmStatic
        fun savePushToken(recentToken: String, context: Context) {
            InternalPreferences.savePushToken(context, recentToken)
        }

        private fun broadcastIntent(context: Context, action: String) {
            val intent = Intent()
            intent.action = action
            context.sendBroadcast(intent)

        }
    }


}
