package com.flybits.concierge.repository.content

import android.content.Context
import com.flybits.android.kernel.models.Content
import com.flybits.android.kernel.utilities.ContentParameters
import com.flybits.commons.library.api.results.callbacks.ListResultCallback
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback
import com.flybits.concierge.models.BaseTemplate

/**
 * This class is responsible for retrieving content and performing other operations on content.
 * We use this so that context doesn't have to be passed as a dependency to classes that shouldn't
 * be holding a reference to it(cohesion).
 *
 * @param context The application's context
 */
class ContentGetter(private val context: Context){

    /**
     * Retrieve paged content using parameters
     *
     * @param contentParameters Parameters specifying the query type
     * @param pagedResultCallback Callback where the results will be provided
     *
     * @return PagedResult
     */
    fun getContent(contentParameters: ContentParameters, pagedResultCallback: PagedResultCallback<Content>)
            = Content.get(context, contentParameters, pagedResultCallback)!!

    /**
     * Retrieve [Content] list using id list
     *
     * @param ids list of content ids to retrieve
     * @param listResultCallback Callback where the results will be provided
     *
     * @return local result object
     */
    fun getContent(ids: Collection<String>, listResultCallback: ListResultCallback<Content>)
            = Content.get(context, ids, listResultCallback)

    /**
     * Retrieve content using single id
     *
     * @param id id of the associated [Content]
     * @param objectResultCallback Callback where the result will be provided
     *
     * @return local result object
     */
    fun getContent(id: String, objectResultCallback: ObjectResultCallback<Content>)
            = Content.getById(context, id, objectResultCallback)

    /**
     * Deserialize content data into an object
     *
     * @param content Content to be deserialized
     * @param typeClass Class that will be deserialized into
     *
     * @return Deserialized class
     */
    fun getData(content: Content, typeClass: Class<BaseTemplate>): BaseTemplate? {
        val data = content.getData<BaseTemplate>(typeClass)
        content.clearDeserializedObject()
        return data
    }
}