package com.flybits.concierge;

import android.app.Activity;
import android.content.BroadcastReceiver;
import android.content.ComponentName;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.IntentFilter;
import android.media.session.PlaybackState;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.RemoteException;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.design.widget.Snackbar;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentActivity;
import android.support.v4.app.FragmentManager;
import android.support.v4.app.FragmentTransaction;
import android.support.v4.media.MediaBrowserCompat;
import android.support.v4.media.session.MediaControllerCompat;
import android.support.v4.media.session.PlaybackStateCompat;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.afollestad.materialdialogs.MaterialDialog;
import com.flybits.android.kernel.ContentAnalytics;
import com.flybits.android.kernel.models.Content;
import com.flybits.android.kernel.models.PagedArray;
import com.flybits.android.kernel.utilities.ContentParameters;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.logging.Logger;
import com.flybits.concierge.activities.NotificationsActivity;
import com.flybits.concierge.activities.SettingsActivity;
import com.flybits.concierge.exception.ConciergeException;
import com.flybits.concierge.fragments.FeedHolderFragment;
import com.flybits.concierge.fragments.ImageViewerPagerFragment;
import com.flybits.concierge.fragments.OnboardingFragment;
import com.flybits.concierge.fragments.SurveyFragment;
import com.flybits.concierge.fragments.TermsAndServicesFragment;
import com.flybits.concierge.models.Audio;
import com.flybits.concierge.models.Image;
import com.flybits.concierge.models.Onboarding;
import com.flybits.concierge.models.Survey;
import com.flybits.concierge.repository.ModelConverter;
import com.flybits.concierge.repository.content.ContentGetter;
import com.flybits.concierge.services.AudioService;
import com.flybits.concierge.viewproviders.AudiosViewProvider;
import com.pierfrancescosoffritti.androidyoutubeplayer.utils.Utils;

import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.Collection;

/**
 * This fragment is responsible for displaying all of the Concierge.
 *
 * For instantiating this fragment please see the newInstance method.
 */
public class ConciergeFragment extends Fragment implements AuthenticationStatusListener
        , OptOutListener, FlybitsNavigator, FlybitsViewProviderGetter {

    /**
     * Enum responsible for specifying where and how the options menu will appear.
     */
    public enum MenuType{
        /**
         * Menu will appear in the app bar of the hosting activity.
         */
        MENU_TYPE_APP_BAR(0)
        ,
        /**
         * Menu will appear in the
         */
        MENU_TYPE_TAB(1);

        private int value;

        MenuType(int value){
            this.value = value;
        }

        public static MenuType fromInt(int type){
            if (type == 0){
                return MENU_TYPE_APP_BAR;
            } else if (type == 1) {
                return MENU_TYPE_TAB;
            } else {
                throw new IllegalArgumentException("Type is invalid use TYPE_TAB or TYPE_APP_BAR");
            }
        }

        public int getValue(){
            return value;
        }
    }

    public static final String INSTANCE_CURRENT_FRAGMENT = "instance_current_fragment";
    public static final int MESSAGE_TNC = 1;
    public static final int MESSAGE_ONBOARDING = 2;
    public static final String ARG_MENU_TYPE = "flybits_con_menu_type";

    private LinearLayout lytLoader;
    private TextView txtLoaderText;
    private LinearLayout errorViewContainer;
    private View mainViewContainer;
    private View optInContainer;

    private Context currentContext;

    private Fragment currentFragment;
    private BroadcastReceiver optOutReceiver;

    private IConciergeFragmentCallbacks currentCallback;

    //Audio Handling
    private MediaBrowserCompat mediaBrowserCompat;
    private MediaControllerCompat mediaControllerCompat;
    private MediaControllerCompat.Callback pendingCallback;
    private Audio lastAudio = null;

    private boolean initializing = false;
    private boolean actionBarItemsVisible = true;

    private FlybitsConcierge flybitsConcierge;

    private MediaControllerCompat.Callback controllerCallback = new MediaControllerCompat.Callback()
    {
        @Override
        public void onPlaybackStateChanged(PlaybackStateCompat state)
        {
            Activity activity = getActivity();

            if (activity != null){
                Intent intent = new Intent(ConciergeConstants.BROADCAST_AUDIO_STATE);
                intent.putExtra(ConciergeConstants.INTENT_EXTRA_AUDIO_STATE, state);
                activity.sendBroadcast(intent);
            }
        }
    };

    private MediaBrowserCompat.ConnectionCallback mediaBrowserCompatConnectionCallback = new MediaBrowserCompat.ConnectionCallback()
    {

        @Override
        public void onConnected()
        {
            super.onConnected();

            final Context context = getContext();
            if (context != null){
                try
                {
                    mediaControllerCompat = new MediaControllerCompat(context, mediaBrowserCompat.getSessionToken());
                    mediaControllerCompat.registerCallback(controllerCallback);
                    if (pendingCallback != null){
                        mediaControllerCompat.registerCallback(pendingCallback);
                        pendingCallback = null;
                    }
                    //MediaControllerCompat.setMediaController(getActivity(), mediaControllerCompat);

                    Bundle extras = mediaControllerCompat.getPlaybackState().getExtras();
                    //We are still playing something?
                    if (extras != null && lastAudio == null)
                    {
                        lastAudio = new Audio();
                        lastAudio.sourceURL = extras.getString(AudioService.ARG_URL);
                    }else if (lastAudio != null){
                        playFromMediaId(lastAudio);
                    }

                }
                catch (RemoteException e)
                {
                    Logger.exception(null, e);
                }
            }
        }
    };

    //Messages back from child fragments
    public void childMessage(int code, Object data)
    {
        switch (code)
        {
            case MESSAGE_TNC:
                boolean result = (Boolean) data;
                if (result)
                {
                    initializeState(false);
                }
                else if (currentCallback != null)
                {
                    currentCallback.onTNCDecline();
                }
                break;
            case MESSAGE_ONBOARDING:
                initializeState(false);
                break;
        }
    }

    /**
     * Create instance of {@link ConciergeFragment}.
     *
     * @param menuType The {@link MenuType} appearance that will be used.
     * @return new instance of {@link ConciergeFragment}.
     */
    public static ConciergeFragment newInstance(MenuType menuType)
    {
        ConciergeFragment fragment = new ConciergeFragment();
        Bundle bundle = new Bundle();
        bundle.putInt(ARG_MENU_TYPE, menuType.value);
        fragment.setArguments(bundle);
        return fragment;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);

        if (savedInstanceState != null)
        {
            currentFragment = getChildFragmentManager().getFragment(savedInstanceState, INSTANCE_CURRENT_FRAGMENT);
        }
    }

    @Override
    public void onSaveInstanceState(@NonNull Bundle outState)
    {
        super.onSaveInstanceState(outState);

        if (currentFragment != null)
        {
            getChildFragmentManager().putFragment(outState, INSTANCE_CURRENT_FRAGMENT, currentFragment);
        }
        outState.putBoolean(ConciergeConstants.STATE_ERROR_PRESENT, errorViewContainer.getVisibility() == View.VISIBLE);

    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, Bundle savedInstanceState)
    {
        return inflater.inflate(R.layout.flybits_con_fragment_concierge, container, false);
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState)
    {
        super.onViewCreated(view, savedInstanceState);

        /*Code below makes it so that the options menu appears in the app bar if that setting is
        * being used. It is hosted in the fragment since we do not know what the hosting activity
        * is. */
        Bundle args = getArguments();
        if (args != null){
            MenuType menuType = MenuType.fromInt(args.getInt(ARG_MENU_TYPE, MenuType.MENU_TYPE_APP_BAR.value));
            if (menuType == MenuType.MENU_TYPE_APP_BAR){
                setHasOptionsMenu(true);
            }
        } else {
            setHasOptionsMenu(true);
        }

        lytLoader = view.findViewById(R.id.concierge_fragment_lytLoader);
        txtLoaderText = view.findViewById(R.id.concierge_fragment_txtLoadingText);
        errorViewContainer = view.findViewById(R.id.concierge_fragment_error_holder);
        mainViewContainer = view.findViewById(contentLayout());

        IntentFilter intentFilter = new IntentFilter();
        intentFilter.addAction(ConciergeConstants.BROADCAST_OPT_OUT);

        optOutReceiver = new BroadcastReceiver()
        {
            @Override
            public void onReceive(Context context, Intent intent)
            {
                initializeState(false);
            }
        };

        getActivity().registerReceiver(optOutReceiver, intentFilter);

        optInContainer = view.findViewById(R.id.concierge_fragment_opt_in_container);

        view.findViewById(R.id.flybits_con_fragment_btnOptIn).setOnClickListener(new View.OnClickListener()
        {
            @Override
            public void onClick(View v)
            {
                FlybitsConcierge.with(currentContext).optInRetryAuthentication();
            }
        });

        view.findViewById(R.id.concierge_fragment_retry_button).setOnClickListener(new View.OnClickListener()
        {
            @Override
            public void onClick(View view)
            {
                initializeState(false);
            }
        });

        ContentAnalytics contentAnalytics = new ContentAnalytics(getContext());
        FlybitsViewPopulator flybitsViewPopulator = new FlybitsViewPopulator(contentAnalytics);

        flybitsConcierge = FlybitsConcierge.with(currentContext);
        flybitsConcierge.registerAuthenticationStateListener(this);
        flybitsConcierge.registerOptOutListener(this);

        //Registration should take place in the FlybitsConcierge initialize method but due to the concierge fragment coupling it is done here instead temporarily.
        flybitsConcierge.registerFlybitsViewProvider(new AudiosViewProvider(this, flybitsViewPopulator));

        initializeState(false);
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
        if (flybitsConcierge != null){
            flybitsConcierge.unregisterAuthenticationStateListener(this);
            flybitsConcierge.unregisterOptOutListener(this);
        }

        getActivity().unregisterReceiver(optOutReceiver);
    }

    @Override
    public void onAttach(Context context)
    {
        super.onAttach(context);
        currentContext = context;
    }

    @Override
    public void onDetach()
    {
        super.onDetach();
        if (mediaBrowserCompat != null && mediaBrowserCompat.isConnected())
        {
            mediaBrowserCompat.disconnect();
        }

        disconnectFromAudioService();
    }

    /**
     * Attempts to resolve all requirements to see the main feed.
     *
     * @param skipOnboarding Skips onboarding (due to it not existing)
     */
    private void initializeState(final boolean skipOnboarding)
    {
        if (initializing) return;
        initializing = true;

        // reset the views to be in their default visibility
        errorViewContainer.setVisibility(View.GONE);
        optInContainer.setVisibility(View.GONE);
        mainViewContainer.setVisibility(View.INVISIBLE);

        //Check if authorized first
        if (flybitsConcierge.isAuthenticated() && !InternalPreferences.getDidOptOut(currentContext))
        {
            mainViewContainer.setVisibility(View.VISIBLE);
            flybitsConcierge.unregisterAuthenticationStateListener(this);

            //Are TNCs Accepted?
            if (FlybitsConcierge.with(currentContext).getConfiguration().getTNCUrl() != null && !InternalPreferences.isTNCAccepted(currentContext))
            {
                // if TNCs haven't been shown yet, and the link is initialized in the config file
                hideLoader();
                currentFragment = TermsAndServicesFragment.newInstance(false);
                FragmentManager manager = getChildFragmentManager();
                manager.beginTransaction()
                        .replace(contentLayout(), currentFragment)
                        .commit();
                setActionBarItemsVisibility(false);
            }
            else if (!skipOnboarding && !InternalPreferences.isOnBoardingDone(currentContext))
            {
                // if you should show onboarding and onboarding hasn't been shown yet
                hideLoader();
                findContentAndTakeover(ConciergeConstants.ONBOARDING_CONTENT_TYPE);
                InternalPreferences.saveOnBoardingDone(currentContext, true);
                setActionBarItemsVisibility(false);
            }
            else if (!InternalPreferences.isSurveyDone(currentContext))
            {
                // if you should show a survey
                hideLoader();
                findContentAndTakeover(ConciergeConstants.SURVEY_CONTENT_TYPE);
                InternalPreferences.saveSurveyDone(currentContext, true);
                setActionBarItemsVisibility(false);
            }
            else
            {
                // Otherwise, show CategoryFragment
                hideLoader();
                Bundle arguments = getArguments();
                boolean showMoreTab = false;
                if (arguments != null){
                    MenuType menuType = MenuType.fromInt(arguments.getInt(ARG_MENU_TYPE, MenuType.MENU_TYPE_APP_BAR.value));
                    showMoreTab =  menuType != MenuType.MENU_TYPE_APP_BAR;
                }
                currentFragment = FeedHolderFragment.Companion.newInstance(showMoreTab);
                FragmentManager manager = getChildFragmentManager();
                manager.beginTransaction()
                        .replace(contentLayout(), currentFragment, currentFragment.getClass().getSimpleName())
                        .commit();
                setActionBarItemsVisibility(true);
            }
        }
        else if (InternalPreferences.getDidOptOut(currentContext))
        {
            setActionBarItemsVisibility(false);
            // register for the authentication state listener that was unregistered for above.
            flybitsConcierge.registerAuthenticationStateListener(this);
            optInContainer.setVisibility(View.VISIBLE);
            errorViewContainer.setVisibility(View.GONE);
        }
        else
        {
            //Retry authentication since we are not currently authenticated, wait for callback(AuthenticationStatusListener)'s methods to be invoked
            boolean retrySuccess = flybitsConcierge.retryAuthentication();
            if (retrySuccess){
                showLoader("");
            }
            errorViewContainer.setVisibility(View.VISIBLE);
            setActionBarItemsVisibility(false);
        }

        initializing = false;
    }

    @Override
    public void onCreateOptionsMenu(Menu menu, MenuInflater inflater) {
        inflater.inflate(R.menu.menu, menu);
        menu.findItem(R.id.notifications).setVisible(actionBarItemsVisible);
        menu.findItem(R.id.settings).setVisible(actionBarItemsVisible);
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        int id = item.getItemId();
        if (id == R.id.settings) {
            Intent startSettingsIntent = new Intent(getContext(), SettingsActivity.class);
            startActivityForResult(startSettingsIntent, SettingsActivity.REQUEST_CODE);
            return true;
        } else if (id == R.id.notifications) {
            Intent startNotificationsIntent = new Intent(getContext(), NotificationsActivity.class);
            startActivity(startNotificationsIntent);
            return true;
        }

        return super.onOptionsItemSelected(item);
    }

    private void setActionBarItemsVisibility(boolean visible){
        actionBarItemsVisible = visible;
        Activity activity = getActivity();
        if (activity != null){
            activity.invalidateOptionsMenu();
        }

    }

    /**
     * Replaces the current content to the TNC page
     */
    private void showTNCs()
    {
        currentFragment = TermsAndServicesFragment.newInstance(false);
        FragmentManager manager = getChildFragmentManager();
        manager.beginTransaction()
                .replace(contentLayout(), currentFragment)
                .commit();
    }

    /**
     * Replaces the current content to some Flybits Content Fragment (Onboarding, Surveys)
     */
    private void findContentAndTakeover(final String contentTemplateType)
    {
        ContentParameters params = new ContentParameters.Builder().setTemplateType(contentTemplateType)
                .build();

        Content.get(currentContext, params, new PagedResultCallback<Content>()
        {
            @Override
            public void onSuccess(ArrayList<Content> items)
            {
                if (items.size() != 0)
                {
                    showContent(items.get(0));
                }
                else
                {
                    initializeState(true);
                }
            }

            @Override
            public void onException(FlybitsException exception)
            {

            }

            @Override
            public void onLoadedAllItems()
            {

            }
        });
    }

    /**
     * Replaces the current content with a content.
     */
    private void showContent(Content contentToShow)
    {
        ModelConverter modelConverter = new ModelConverter(new ContentGetter(getContext()), this);
        hideLoader();
        FragmentManager manager = getChildFragmentManager();
        switch (contentToShow.getType())
        {
            case ConciergeConstants.ONBOARDING_CONTENT_TYPE:
            {
                Onboarding onboarding = (Onboarding)modelConverter.contentToBaseTemplate(contentToShow);
                if (onboarding == null) {
                    initializeState(true);
                    return;
                }

                OnboardingFragment onboardingFragment = OnboardingFragment.newInstance(onboarding);

                onboardingFragment.setOnboardingListener(new OnboardingListener()
                {
                    @Override
                    public void onSkip()
                    {
                        initializeState(false);
                    }

                    @Override
                    public void onFinished()
                    {
                        initializeState(false);
                    }
                });

                currentFragment = onboardingFragment;
                manager.beginTransaction()
                        .replace(contentLayout(), currentFragment, currentFragment.getClass().getSimpleName())
                        .commit();
                break;
            }
            case ConciergeConstants.SURVEY_CONTENT_TYPE:
            {
                Survey survey = (Survey) modelConverter.contentToBaseTemplate(contentToShow);
                if (survey == null || survey.questions.getList().isEmpty()) {
                    new MaterialDialog.Builder(getActivity())
                            .title(R.string.flybits_con_error)
                            .content(R.string.flybits_con_dialog_survey_error)
                            .positiveText(android.R.string.ok)
                            .show();
                    InternalPreferences.saveSurveyDone(getContext(), true);
                    initializeState(false);
                    return;
                }

                SurveyFragment surveyFragment = SurveyFragment.newInstance(survey);

                surveyFragment.setSurveyListener(new SurveyListener()
                {
                    @Override
                    public void onFinished()
                    {
                        initializeState(false);
                        InternalPreferences.saveSurveyDone(currentContext, true);
                    }
                });

                currentFragment = surveyFragment;
                manager.beginTransaction()
                        .replace(contentLayout(), currentFragment, currentFragment.getClass().getSimpleName())
                        .commit();
                break;
            }
            default:
                initializeState(false);
                break;
        }
    }

    public void openImageViewer(PagedArray<Image> images, int offset)
    {
        Fragment imageFragment = ImageViewerPagerFragment.newInstance(images, offset);
        openFragment(imageFragment, true);
    }

    private void connectToAudioService()
    {
        Context context = getContext();
        if (context == null)
        {
            return;
        }
        if (mediaBrowserCompat == null)
        {
            mediaBrowserCompat = new MediaBrowserCompat(context.getApplicationContext()
                    , new ComponentName(context.getApplicationContext(), AudioService.class), mediaBrowserCompatConnectionCallback, null);
        }
        mediaBrowserCompat.connect();
    }

    private void disconnectFromAudioService()
    {
        if (mediaBrowserCompat != null && mediaBrowserCompat.isConnected())
        {
            mediaControllerCompat.unregisterCallback(controllerCallback);
            mediaBrowserCompat.disconnect();
        }
    }

    public void startAudio(Audio audio)
    {
        lastAudio = audio;
        boolean isOnline = Utils.isOnline(currentContext);
        View view = getView();
        if (isOnline && (mediaBrowserCompat == null || !mediaBrowserCompat.isConnected())){
            connectToAudioService();
        }else if (isOnline){
            playFromMediaId(audio);
        }else if (view != null){
            Snackbar.make(view,R.string.flybits_con_audio_error_message,Snackbar.LENGTH_LONG).show();
        }
    }

    public void registerMediaControllerCallback(MediaControllerCompat.Callback mediaControllerCallback){
        if (mediaControllerCallback != null && mediaBrowserCompat != null &&  mediaBrowserCompat.isConnected()){
            pendingCallback = null;
            mediaControllerCompat.registerCallback(mediaControllerCallback);
        }
        //register callback once AudioService is connected
        else{
            pendingCallback = mediaControllerCallback;
        }

    }

    public void resumeAudio()
    {
        View view = getView();
        Context context = getContext();
        if (mediaBrowserCompat != null && mediaBrowserCompat.isConnected()
                && context!= null && Utils.isOnline(context))
        {
            mediaControllerCompat.getTransportControls().play();
        }else if (view != null){
            Snackbar.make(view,R.string.flybits_con_audio_error_message,Snackbar.LENGTH_LONG).show();
        }
    }

    public void pauseAudio(){
        if (mediaControllerCompat != null && mediaControllerCompat.getTransportControls() != null)
        {
            mediaControllerCompat.getTransportControls().pause();
        }
    }

    public void stopAudio()
    {
        lastAudio = null;

        if (mediaControllerCompat != null && mediaControllerCompat.getTransportControls() != null)
        {
            mediaControllerCompat.getTransportControls().stop();
        }
    }

    public int getAudioState()
    {
        if (mediaBrowserCompat == null || !mediaBrowserCompat.isConnected())
        {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                return PlaybackState.STATE_NONE;
            }else{
                return 0;
            }
        }
        return mediaControllerCompat.getPlaybackState().getState();
    }

    private void playFromMediaId(Audio audio){
        Bundle args = new Bundle();

        //For some reason, parcelables ain't workin, do it manually
        if (audio.title != null)
        {
            args.putString(AudioService.ARG_TITLE, audio.title.getValue());
        }
        if (audio.description != null)
        {
            args.putString(AudioService.ARG_DESC, audio.description.getValue());
        }

        if (mediaBrowserCompat != null && mediaBrowserCompat.isConnected())
        {
            mediaControllerCompat.getTransportControls().playFromMediaId(audio.sourceURL, args);
        }
    }

    public Audio getCurrentAudio()
    {
        return lastAudio;
    }

    @Override
    public void openFragment(@NotNull Fragment fragment, boolean addToBackStack)
    {
        FragmentActivity fragmentActivity = getActivity();
        if (fragmentActivity != null){
            FragmentTransaction transaction = fragmentActivity.getSupportFragmentManager()
                    .beginTransaction();
            if (addToBackStack){
                transaction.addToBackStack(null);
            }
            transaction.replace(contentLayout(), fragment)
                    .commit();
        }

    }

    @Override
    public void openActivity(Class activity, Bundle extras) {
        Intent intent = new Intent(currentContext, activity);
        intent.putExtras(extras);
        startActivity(intent);
    }

    /**
     * Removes the current content and shows the loader
     *
     * @param title Message to display
     */
    private void showLoader(String title)
    {
        if (currentFragment != null)
        {
            getChildFragmentManager().beginTransaction()
                    .remove(currentFragment)
                    .commit();
        }
        txtLoaderText.setText(title);
        lytLoader.setVisibility(View.VISIBLE);
    }

    /**
     * Hides the loader
     */
    private void hideLoader()
    {
        txtLoaderText.setText("");
        lytLoader.setVisibility(View.GONE);
    }

    private int contentLayout()
    {
        return R.id.concierge_fragment_lytContent;
    }

    public void setCallback(IConciergeFragmentCallbacks callback)
    {
        currentCallback = callback;
    }

    @Override
    public void onAuthenticated() {
        initializeState(false);
    }

    @Override
    public void onAuthenticationStarted() {

    }

    @Override
    public void onAuthenticationError(ConciergeException e) {
        //Don't display error view if opt out view is already visible
        if (!InternalPreferences.getDidOptOut(currentContext)){
            errorViewContainer.setVisibility(View.VISIBLE);
            hideLoader();
        }
    }

    @Override
    public void onUserOptedOut() {
        if (getActivity() != null){
            initializeState(true);
        }
    }

    @Override
    public boolean openUrl(@NotNull String url) {
        Activity activity = getActivity();
        if (activity == null) {
            return false;
        }

        Intent intent = new Intent(Intent.ACTION_VIEW);
        intent.setData(Uri.parse(url));
        //verify that you're able to open url, if URL is malformed then exception is thrown
        if (intent.resolveActivity(activity.getPackageManager()) != null) {
            startActivity(intent);
            return true;
        } else {
            return false;
        }
    }

    @Override
    public void openDialog(@NotNull String content, @NotNull String title) {
        new MaterialDialog.Builder(getActivity())
                .title(title)
                .content(content)
                .positiveText(android.R.string.ok)
                .dismissListener(new DialogInterface.OnDismissListener(){
                    @Override
                    public void onDismiss(DialogInterface dialogInterface) {
                        getActivity().finish();
                    }
                })
        .show();
    }

    public interface IConciergeFragmentCallbacks
    {
        void onTNCDecline();
    }

    @Override
    public Collection<FlybitsViewProvider> getFlybitsViewProviders() {
        Context context = getContext();
        if (context == null) return new ArrayList<>();
        else return FlybitsConcierge.with(context).getFlybitsViewProviders();
    }

    @Override
    public FlybitsViewProvider getFlybitsViewProvider(String contentType) {
        Context context = getContext();
        if (context == null) return null;
        return FlybitsConcierge.with(context).getFlybitsViewProvider(contentType);
    }

}
