package com.flybits.concierge.fragments

import android.arch.lifecycle.Observer
import android.os.Bundle
import android.support.design.widget.Snackbar
import android.support.v4.app.Fragment
import android.support.v4.widget.SwipeRefreshLayout
import android.support.v7.widget.LinearLayoutManager
import android.support.v7.widget.RecyclerView
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import com.afollestad.materialdialogs.MaterialDialog
import com.flybits.android.push.models.Push
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.*
import com.flybits.concierge.adapters.NotificationAdapter
import com.flybits.concierge.models.NotificationContent
import com.flybits.concierge.repository.RepositoryResponse
import com.flybits.concierge.viewmodels.NotificationsViewModel
import com.flybits.concierge.viewmodels.NotificationsViewModelFactory

/**
 * This fragment displays up to 20 most recent [NotificationContent] that the user has received.
 * Only notifications that are tied to content are shown, and these content must
 * have a [FlybitsViewProvider] registered for them to be shown.
 */
class NotificationsFragment: Fragment() {

    companion object {
        @JvmStatic
        fun newInstance(push: Push? = null): NotificationsFragment {
            val fragment = NotificationsFragment()
            if (push != null){
                val bundle = Bundle()
                bundle.putParcelable(ConciergeConstants.PUSH_EXTRA, push)
                fragment.arguments = bundle
            }
            return fragment
        }
    }

    private var notificationsViewModel: NotificationsViewModel? = null
    private var notificationsAdapter: NotificationAdapter? = null
    private var swipeToRefresh: SwipeRefreshLayout? = null

    override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?): View? {
        return inflater.inflate(R.layout.flybits_con_fragment_category, container, false)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        val navigator: FlybitsNavigator = when {
            activity is FlybitsNavigator -> activity as FlybitsNavigator
            parentFragment is FlybitsNavigator -> parentFragment as FlybitsNavigator
            else -> throw FlybitsException("No flybits navigator defined. Activity or hosting " +
                    "fragment must implement FlybitsNavigator interface")
        }

        notificationsViewModel = NotificationsViewModelFactory(context!!
                , activity as FlybitsViewProviderGetter, navigator).create(NotificationsViewModel::class.java)

        swipeToRefresh = view.findViewById(R.id.flybits_con_swipe_ref_layout)
        swipeToRefresh?.setOnRefreshListener {
            loadNotifications()
        }
        swipeToRefresh?.isRefreshing = true

        val recyclerView: RecyclerView = view.findViewById(R.id.flybits_con_fragment_feed_lytRecycler)
        recyclerView.visibility = View.VISIBLE
        recyclerView.layoutManager = LinearLayoutManager(context!!)

        notificationsAdapter = NotificationAdapter(activity as FlybitsViewProviderGetter, navigator)
        recyclerView.adapter = notificationsAdapter

        loadNotifications()

        //Let view model handle passed push
        arguments?.get(ConciergeConstants.PUSH_EXTRA)?.let { push  ->
            arguments?.remove(ConciergeConstants.PUSH_EXTRA)
        notificationsViewModel?.handlePassedPush(push)?.observe(this, Observer {
                if (it?.data == false){
                    context?.let { nonNullContext ->
                        MaterialDialog.Builder(nonNullContext)
                                .title(R.string.flybits_con_error)
                                .content(getString(R.string.flybits_con_notification_content_error))
                                .positiveText(android.R.string.ok)
                                .show()
                    }

                }
            })
        }
    }

    //Don't reuse this pattern, it sucks, ViewModel should have full control, this should only be called once at the start to bind
    private fun loadNotifications(){
        notificationsViewModel?.getNotificationContents()?.observe(this
                , Observer<RepositoryResponse<List<NotificationContent>>> {
            Logger.d("Notifications: loaded = $it")
            it?.data?.let { notificationBaseTemplateList ->
                toggleEmptyState(notificationBaseTemplateList.isEmpty())
                notificationsAdapter?.submitList(notificationBaseTemplateList)
            }
            it?.error?.let {
                view?.let { nonNullView ->
                    toggleEmptyState(true)
                    Snackbar.make(nonNullView, getString(R.string.flybits_con_notifications_error), Snackbar.LENGTH_LONG).show()
                }
            }
            swipeToRefresh?.isRefreshing = false
        })
    }

    private fun toggleEmptyState(on: Boolean){
        if (on){
            view?.findViewById<View>(R.id.flybits_con_fragment_feed_lytNoData)?.visibility = View.VISIBLE
            view?.findViewById<View>(R.id.flybits_con_fragment_feed_lytRecycler)?.visibility = View.GONE
        }else{
            view?.findViewById<View>(R.id.flybits_con_fragment_feed_lytNoData)?.visibility = View.GONE
            view?.findViewById<View>(R.id.flybits_con_fragment_feed_lytRecycler)?.visibility = View.VISIBLE
        }
    }
}