package com.flybits.concierge.viewholders;

import android.view.LayoutInflater;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.flybits.concierge.FlybitsNavigator;
import com.flybits.concierge.FlybitsViewPopulator;
import com.flybits.concierge.R;
import com.flybits.concierge.Utils;
import com.flybits.concierge.fragments.ScheduleFeedFragment;
import com.flybits.concierge.fragments.ScheduleSegmentFragment;
import com.flybits.concierge.models.Schedule;
import com.flybits.concierge.models.ScheduleSegment;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

public class ScheduleViewHolder extends FlybitsViewHolder<Schedule>
{
    private static final int DAYS_SHOWN = 7;

    private final LinearLayout container;
    private TextView showMoreTextView;
    private TextView viewAllTextView;

    private boolean isExpanded;
    private final List<View> segmentViews;
    private boolean hasMultipleDays;

    private View lastVisibleView;

    private FlybitsViewPopulator flybitsViewPopulator;

    public ScheduleViewHolder(View view, FlybitsViewPopulator flybitsViewPopulator)
    {
        super(view);
        segmentViews = new ArrayList<>();

        showMoreTextView = view.findViewById(R.id.flybits_con_item_template_schedule_txtShowMore);
        viewAllTextView = view.findViewById(R.id.flybits_con_item_template_schedule_txtViewAll);
        container = view.findViewById(R.id.flybits_con_item_template_schedule_lytContainer);
        this.flybitsViewPopulator = flybitsViewPopulator;
    }

    @Override
    public void bindData(final Schedule data, final FlybitsNavigator flybitsNavigator)
    {
        isExpanded = false;

        container.removeAllViews();
        segmentViews.clear(); // clear in the event that the cell is recycled
        viewAllTextView.setVisibility(View.GONE);
        lastVisibleView = null;

        buildCells(data, flybitsNavigator);

        showMoreTextView.setVisibility(segmentViews.size() < 3 ? View.GONE : View.VISIBLE);

        for (View view : segmentViews)
        {
            if (container.getChildCount() == segmentViews.size() - 1)
            {
                // remove the last element's bottom rule
                View rule = view.findViewById(R.id.flybits_con_item_template_schedule_segment_viewRuleBottom);
                rule.setVisibility(View.INVISIBLE);
            }

            View rule = view.findViewById(R.id.flybits_con_item_template_schedule_title_viewTopRule);
            if (rule != null && container.getChildCount() > 0)
            {
                rule.setVisibility(View.VISIBLE);
            }

            container.addView(view);
            view.setVisibility(container.getChildCount() < 3 ? View.VISIBLE : View.GONE);

            if (view.getVisibility() == View.VISIBLE)
            {
                lastVisibleView = view;
            }
        }

        // Expand and retract
        showMoreTextView.setOnClickListener(new View.OnClickListener()
        {
            @Override
            public void onClick(View v)
            {

                isExpanded = !isExpanded;
                showMoreTextView.setText(isExpanded
                        ? R.string.flybits_con_schedule_show_less
                        : R.string.flybits_con_schedule_show_more);
                viewAllTextView.setVisibility(isExpanded && hasMultipleDays
                        ? View.VISIBLE
                        : View.GONE);

                handleLastVisibleRule();

                for (int i = 2; i < segmentViews.size(); i++)
                {
                    segmentViews.get(i).setVisibility(isExpanded ? View.VISIBLE : View.GONE);
                }
            }
        });

        // open the view all mode.
        viewAllTextView.setOnClickListener(new View.OnClickListener()
        {
            @Override
            public void onClick(View v)
            {
                ScheduleFeedFragment fragment = ScheduleFeedFragment.newInstance(data);
                flybitsNavigator.openFragment(fragment, true);
            }
        });

        handleLastVisibleRule();
    }

    private void buildCells(final Schedule data, final FlybitsNavigator flybitsNavigator)
    {
        Map<String, LinkedList<ScheduleSegment>> weekdays = data.generateSegmentList();

        hasMultipleDays = weekdays.keySet().size() > 1;

        Calendar now = Calendar.getInstance();
        for (int i = 0; i < DAYS_SHOWN; i++)
        {
            String weekDay = Utils.getWeekDayString(now.getTime());

            LinkedList<ScheduleSegment> todaySegments = weekdays.get(weekDay);

            if (todaySegments != null) // only handle days that have an element
            {
                View dateTitleView = LayoutInflater.from(container.getContext())
                        .inflate(R.layout.flybits_con_item_template_schedule_title, container, false);
                TextView textView = dateTitleView.findViewById(R.id.flybits_con_item_template_schedule_title_txtTitle);
                textView.setText(Utils.getDateString(now.getTime()));
                segmentViews.add(dateTitleView);

                for (final ScheduleSegment segment : todaySegments)
                {
                    View segmentView = LayoutInflater.from(container.getContext())
                            .inflate(R.layout.flybits_con_item_template_schedule_segment, container, false);

                    TextView timeRangeView = segmentView.findViewById(R.id.flybits_con_item_template_schedule_segment_txtTimeRange);
                    TextView titleView = segmentView.findViewById(R.id.flybits_con_item_template_schedule_segment_txtTitle);
                    TextView descriptionView = segmentView.findViewById(R.id.flybits_con_item_template_schedule_segment_txtDescription);
                    ImageView bulletOutView = segmentView.findViewById(R.id.flybits_con_item_template_schedule_segment_imgBulletOut);
                    ImageView bulletInView = segmentView.findViewById(R.id.flybits_con_item_template_schedule_segment_imgBulletIn);
                    View clickableArea = segmentView.findViewById(R.id.flybits_con_item_template_schedule_segment_lytContainer);

                    clickableArea.setOnClickListener(new View.OnClickListener()
                    {
                        @Override
                        public void onClick(View v)
                        {
                            ScheduleSegmentFragment fragment = ScheduleSegmentFragment.newInstance(segment);
                            flybitsNavigator.openFragment(fragment, true);
                        }
                    });

                    flybitsViewPopulator.populateScheduleSegmentViews(segment, timeRangeView, titleView, descriptionView, bulletOutView, bulletInView, clickableArea);

                    segmentViews.add(segmentView);
                }

                // only do one day?
                break;
            }

            // increment day by 1
            if (now.get(Calendar.DAY_OF_YEAR) == now.getActualMaximum(Calendar.DAY_OF_YEAR))
            {
                now.set(Calendar.DAY_OF_YEAR, 1);
                now.set(Calendar.YEAR, now.get(Calendar.YEAR) + 1);
            }
            else
            {
                now.set(Calendar.DAY_OF_YEAR, now.get(Calendar.DAY_OF_YEAR) + 1);
            }
        }
    }

    private void handleLastVisibleRule()
    {
        if (lastVisibleView != null)
        {
            View bottomRule = lastVisibleView.findViewById(R.id.flybits_con_item_template_schedule_segment_viewRuleBottom);
            if (bottomRule != null)
            {
                bottomRule.setVisibility(isExpanded ? View.VISIBLE : View.INVISIBLE);
            }
        }
    }
}
