package com.flybits.concierge.viewmodels

import android.arch.lifecycle.LiveData
import android.arch.lifecycle.MutableLiveData
import android.arch.lifecycle.ViewModel
import com.flybits.android.push.models.Push
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.FlybitsNavigator
import com.flybits.concierge.FlybitsViewProviderGetter
import com.flybits.concierge.models.BaseTemplate
import com.flybits.concierge.models.NotificationContent
import com.flybits.concierge.repository.RepositoryResponse
import com.flybits.concierge.repository.push.NotificationContentRepository

/**
 * Responsible for retrieving and handling data/actions related to [NotificationContent]
 * using [LiveData].
 */
class NotificationsViewModel(private val notificationContentRepository: NotificationContentRepository
                             , private val flybitsViewProviderGetter: FlybitsViewProviderGetter
                             , private val flybitsNavigator: FlybitsNavigator): ViewModel() {


    /**
     * Returns LiveData which streams [RepositoryResponse] that holds a list of [NotificationContent]
     * to be displayed.
     */
    fun getNotificationContents(): LiveData<RepositoryResponse<List<NotificationContent>>>{
        return notificationContentRepository.getNotificationContentRemote(10)
    }

    /**
     * Handle individual [Push] that has been passed down. If the [Push] is tied to Content
     * and a FlybitsViewProvider is registered, its push handling method will be invoked.
     */
    fun handlePassedPush(push: Any?): LiveData<RepositoryResponse<Boolean>>{
        val liveData = MutableLiveData<RepositoryResponse<Boolean>>()
        if (push is Push){

            //proccess url
            notificationContentRepository.getUrl(push)?.let {
                flybitsNavigator.openUrl(it)
                liveData.value = RepositoryResponse(data = true)
                return liveData
            }

            //process content
            notificationContentRepository.getContentId(push)?.let { contentId ->
                notificationContentRepository.getBaseTemplate(contentId, object: ObjectResultCallback<BaseTemplate> {
                    override fun onSuccess(item: BaseTemplate) {
                        val provider = flybitsViewProviderGetter.getFlybitsViewProvider(item.content?.type ?: "")
                        if (provider == null) {
                            liveData.value = RepositoryResponse(error = FlybitsException("No associated FlybitsViewProvider for content"))
                        } else {
                            Logger.d("NotificationsViewModel: content type found = ${item.content?.type}")
                            provider.onPushOpened(item, flybitsNavigator)
                            liveData.value = RepositoryResponse(data = true)
                        }
                    }

                    override fun onException(exception: FlybitsException) {
                        liveData.value = RepositoryResponse(error = exception)
                    }
                })
                return liveData
            }

        } else {
            liveData.value = RepositoryResponse(error = FlybitsException("Expected push, got something else"))
        }

        return liveData
    }
}