package com.flybits.concierge.repository

import android.arch.persistence.db.SimpleSQLiteQuery
import com.flybits.android.kernel.db.dao.ContentDao
import com.flybits.android.kernel.models.Content
import com.flybits.android.kernel.models.results.ContentResult
import com.flybits.android.kernel.utilities.ContentParameters
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback
import com.flybits.internal.db.CachingEntryDAO
import java.util.*

class ContentRepository(private val contentGetter: ContentGetter
                        , private val contentDao: ContentDao, private val cachingEntryDao: CachingEntryDAO) {

    fun getContent(contentParameters: ContentParameters, pagedResultCallback: PagedResultCallback<Content>): ContentResult{
        return contentGetter.getContent(contentParameters, pagedResultCallback)
    }

    private fun List<String>.joinListToString()
            = joinToString(prefix = "(", separator = ",", postfix = ")", transform = {"\'$it\'"})

    private fun getQuery(including: List<String>, includingType: List<String>, limit: Int, offset: Int)
            = SimpleSQLiteQuery("SELECT * FROM " +
                "content WHERE " +
                "id IN ${including.joinListToString()} " +
                "AND type IN ${includingType.joinListToString()} " +
                "LIMIT $limit " +
                "OFFSET $offset")

    private fun getCountQuery(including: List<String>, includingType: List<String>)
            = SimpleSQLiteQuery("SELECT COUNT(*) FROM " +
                "content WHERE " +
                "id IN ${including.joinListToString()} " +
                "AND type IN ${includingType.joinListToString()}")

    fun getByCacheIdCount(cacheId: String, includingType: List<String>): Long{
        val cachingEntryList = cachingEntryDao.getByKey(cacheId)

        val cachedIds = ArrayList<String>()
        for (e in cachingEntryList) {
            cachedIds.add(e.contentId)
        }
        return contentDao.getCountRaw(getCountQuery(cachedIds, includingType))
    }

    fun getByCacheId(cacheId: String, includingType: List<String>, limit: Int, offset: Int): List<Content> {

        //kotlin lets you nest classes in functions how cool is that?
        data class ContentWrapper(val sequence: Int, val content: Content)

        val cachingEntryList = cachingEntryDao.getByKey(cacheId)

        val cachedIds = ArrayList<String>()
        cachingEntryList.forEach {
            cachedIds.add(it.contentId)
        }

        val contentWrapperList = mutableListOf<ContentWrapper>()
        val contentList = contentDao.getRaw(getQuery(cachedIds, includingType, limit, offset))

        contentList.forEach { content ->
            cachingEntryList.find { entry ->
                entry.contentId == content.id
            }.let { entry ->
                entry?.let { nonNullEntry ->
                    contentWrapperList.add(ContentWrapper(nonNullEntry.sequence, content))
                }
            }
        }

        return contentWrapperList.sortedBy{ it.sequence }.map { it.content }
    }

}