package com.flybits.concierge.repository

import android.arch.paging.DataSource
import android.arch.paging.PositionalDataSource
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.ConciergeConstants
import com.flybits.concierge.models.BaseTemplate

class BaseTemplateDataSource(private val cacheId: String, private val contentRepository: ContentRepository
                             , private val modelConverter: ModelConverter): PositionalDataSource<BaseTemplate>() {

    companion object {

        val excludeList = listOf(ConciergeConstants.SURVEY_CONTENT_TYPE, ConciergeConstants.ONBOARDING_CONTENT_TYPE)

        fun getFactory(cacheId: String, contentRepository: ContentRepository, modelConverter: ModelConverter
                       , onCreate: (dataSource: DataSource<Int, BaseTemplate>) -> Unit): DataSource.Factory<Int, BaseTemplate>{
            return object: DataSource.Factory<Int, BaseTemplate>(){
                override fun create(): DataSource<Int, BaseTemplate> {
                    val dataSource = BaseTemplateDataSource(cacheId, contentRepository, modelConverter)
                    onCreate(dataSource)
                    return dataSource
                }
            }
        }
    }

    //Do not run this on background thread or content will stop appearing
    override fun loadRange(params: LoadRangeParams, callback: LoadRangeCallback<BaseTemplate>) {
        Logger.d("Category: $cacheId, loadRange() start: ${params.startPosition} size: ${params.loadSize}")
        val queryResult = query.invoke(params.startPosition, params.loadSize)
        callback.onResult(queryResult)
    }

    override fun loadInitial(params: LoadInitialParams, callback: LoadInitialCallback<BaseTemplate>) {
        val totalCount = contentRepository.getByCacheIdCount(cacheId, excludeList).toInt()
        val initialLoadPosition = PositionalDataSource.computeInitialLoadPosition(params, totalCount)
        val loadSize = PositionalDataSource.computeInitialLoadSize(params, initialLoadPosition, totalCount)
        Logger.d("Category: $cacheId, loadInitial() startPos: $initialLoadPosition, totalCount: $totalCount, loadSize: $loadSize")
        val queryResult = query(initialLoadPosition, loadSize)
        if (queryResult.size == loadSize){
            callback.onResult(queryResult, initialLoadPosition, totalCount)
        }else{
            Logger.d("Category: $cacheId, invalidating data source!")
            invalidate()
        }
    }

    private val query: (offset: Int, limit: Int) -> List<BaseTemplate> = { offset, limit ->
        val contentList = contentRepository.getByCacheId(cacheId, excludeList,  limit, offset)
        Logger.d("limit: $limit, offset: $offset, got content: $contentList")
        val templates: List<BaseTemplate> = contentList.asSequence().map {
            val baseTemplate = modelConverter.contentToBaseTemplate(it)
            Logger.d("basetemplate mapped: ${baseTemplate?.content?.id}")
            baseTemplate
        }.filterNotNull().toList()
        templates
    }

}