package com.flybits.concierge.services;

import android.app.PendingIntent;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.media.AudioManager;
import android.media.MediaPlayer;
import android.os.Build;
import android.os.Bundle;
import android.os.PowerManager;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.app.NotificationCompat;
import android.support.v4.app.NotificationManagerCompat;
import android.support.v4.media.MediaBrowserCompat;
import android.support.v4.media.MediaBrowserServiceCompat;
import android.support.v4.media.session.MediaButtonReceiver;
import android.support.v4.media.session.MediaSessionCompat;
import android.support.v4.media.session.PlaybackStateCompat;

import com.flybits.commons.library.logging.Logger;
import com.flybits.concierge.R;

import java.io.IOException;
import java.util.List;

public class AudioService extends MediaBrowserServiceCompat implements AudioManager.OnAudioFocusChangeListener
{

    private static final String LOG_TAG = "ConciergeAudio";

    public final static String ARG_TITLE = "arg_title";
    public final static String ARG_DESC = "arg_desc";
    public final static String ARG_URL = "arg_currentAudioURL";

    private static String playingURL;
    private static int playingState;

    private MediaSessionCompat mediaSession;
    private PlaybackStateCompat.Builder stateBuilder;
    private MediaPlayer mediaPlayer;

    private String currentMusicTitle = null;
    private String currentMusicDescription = null;
    private String currentMusicArtURL = null;
    public String currentMusicURL = null;

    public static final String AUDIO_CHANNEL = "FLYBITS_CONCIERGE_AUDIO";

    private static final String PLAYING_ACTION = "com.flybits.concierge.action.playing";
    private static final String STOPPING_ACTION = "com.flybits.concierge.action.stopping";
    private static final String PAUSING_ACTION = "com.flybits.concierge.action.pausing";

    private MediaSessionCompat.Callback mediaSessionCallback = new MediaSessionCompat.Callback()
    {

        @Override
        public void onPlay()
        {
            super.onPlay();

            if (!successfullyRetrievedAudioFocus())
            {
                return;
            }

            startService(new Intent(getApplicationContext(), AudioService.class));

            //Setup Session
            mediaSession.setActive(true);
            setMediaPlaybackState(PlaybackStateCompat.STATE_PLAYING);

            //Show Notification
            NotificationCompat.Builder builder = new NotificationCompat.Builder(getBaseContext(), AUDIO_CHANNEL);
            builder.setContentTitle(currentMusicTitle != null ? currentMusicTitle : "Untitled")
                    .setContentText(currentMusicDescription != null ? currentMusicDescription : "")
                    .setSubText(getApplicationName(getApplicationContext()))
                    .setContentIntent(mediaSession.getController()
                            .getSessionActivity())
                    .setOngoing(true)
                    .setOnlyAlertOnce(true)
                    .setDeleteIntent(MediaButtonReceiver.buildMediaButtonPendingIntent(getBaseContext(), PlaybackStateCompat.ACTION_STOP))
                    .setVisibility(NotificationCompat.VISIBILITY_PUBLIC)
                    .addAction(new NotificationCompat.Action(R.drawable.ic_baseline_pause_24px, "Pause", MediaButtonReceiver.buildMediaButtonPendingIntent(getBaseContext(), PlaybackStateCompat.ACTION_PLAY_PAUSE)))
                    .addAction(new NotificationCompat.Action(R.drawable.ic_baseline_close_24px, "Stop", MediaButtonReceiver.buildMediaButtonPendingIntent(getBaseContext(), PlaybackStateCompat.ACTION_STOP)))
                    .setStyle(new android.support.v4.media.app.NotificationCompat.MediaStyle().setShowActionsInCompactView(0,1)
                            .setMediaSession(mediaSession.getSessionToken()))
                    .setSmallIcon(R.drawable.ic_audio);
            startForeground(1, builder.build());

            //Start Playback
            mediaPlayer.start();

            Intent playIntent = new Intent(PLAYING_ACTION);
            getApplicationContext().sendBroadcast(playIntent);

            Logger.i("Playing audio.");
        }

        @Override
        public void onPause()
        {
            super.onPause();

            if (mediaPlayer.isPlaying())
            {
                mediaPlayer.pause();
                setMediaPlaybackState(PlaybackStateCompat.STATE_PAUSED);
                NotificationCompat.Builder builder = new NotificationCompat.Builder(getBaseContext(), AUDIO_CHANNEL);
                builder.setContentTitle(currentMusicTitle != null ? currentMusicTitle : "Untitled")
                        .setContentText(currentMusicDescription != null ? currentMusicDescription : "")
                        .setSubText(getApplicationName(getApplicationContext()))
                        .setOnlyAlertOnce(true)
                        .setOngoing(false)
                        .setContentIntent(mediaSession.getController()
                                .getSessionActivity())
                        .setDeleteIntent(MediaButtonReceiver.buildMediaButtonPendingIntent(getBaseContext(), PlaybackStateCompat.ACTION_STOP))
                        .setVisibility(NotificationCompat.VISIBILITY_PUBLIC)
                        .addAction(new NotificationCompat.Action(R.drawable.ic_baseline_play_arrow_24px, "Play", MediaButtonReceiver.buildMediaButtonPendingIntent(getBaseContext(), PlaybackStateCompat.ACTION_PLAY_PAUSE)))
                        .addAction(new NotificationCompat.Action(R.drawable.ic_baseline_close_24px, "Stop", MediaButtonReceiver.buildMediaButtonPendingIntent(getBaseContext(), PlaybackStateCompat.ACTION_STOP)))
                        .setStyle(new android.support.v4.media.app.NotificationCompat.MediaStyle().setShowActionsInCompactView(0,1)
                                .setMediaSession(mediaSession.getSessionToken()))
                        .setSmallIcon(R.drawable.ic_baseline_volume_up_24px);
                startForeground(1, builder.build());
            }

            Intent pausingIntent = new Intent(PAUSING_ACTION);
            getApplicationContext().sendBroadcast(pausingIntent);
            Logger.i("Pausing audio.");
        }

        @Override
        public void onPlayFromMediaId(String mediaId, Bundle extras)
        {
            super.onPlayFromMediaId(mediaId, extras);

            AudioService.playingURL = mediaId;

            try
            {
                mediaPlayer.reset();
                mediaPlayer.setDataSource(mediaId);
            }
            catch (IllegalStateException e)
            {

                mediaPlayer.setWakeMode(getApplicationContext(), PowerManager.PARTIAL_WAKE_LOCK);
                mediaPlayer.setAudioStreamType(AudioManager.STREAM_MUSIC);
                mediaPlayer.setVolume(1.0f, 1.0f);
                try
                {
                    mediaPlayer.setDataSource(mediaId);
                }
                catch (IOException e1)
                {
                    Logger.exception(AudioService.class.getSimpleName(), e1);
                }
            }
            catch (IOException e)
            {
                Logger.exception(AudioService.class.getSimpleName(), e);
            }

            mediaPlayer.prepareAsync();

            mediaPlayer.setOnPreparedListener(new MediaPlayer.OnPreparedListener()
            {
                @Override
                public void onPrepared(MediaPlayer mediaPlayer)
                {
                    mediaSession.getController()
                            .getTransportControls()
                            .play();
                }
            });

            currentMusicTitle = extras.getString(ARG_TITLE, null);
            currentMusicDescription = extras.getString(ARG_DESC, null);
            currentMusicURL = mediaId;

            Logger.i("Loading and playing audio.");
        }

        @Override
        public void onStop()
        {
            super.onStop();
            setMediaPlaybackState(PlaybackStateCompat.STATE_STOPPED);
            mediaPlayer.stop();
            mediaPlayer.reset();
            stopForeground(true);
            stopSelf();

            Intent stopIntent = new Intent(STOPPING_ACTION);
            getApplicationContext().sendBroadcast(stopIntent);

            Logger.i("Stopping audio.");

            playingURL = null;
        }

    };

    public static String getPlayingURL()
    {
        return playingURL;
    }

    public static int getPlayingState()
    {
        return playingState;
    }

    @Override
    public void onTaskRemoved(Intent rootIntent)
    {
        super.onTaskRemoved(rootIntent);
        stopSelf();
    }

    @Override
    public void onCreate()
    {
        super.onCreate();

        mediaPlayer = new MediaPlayer();
        mediaPlayer.setWakeMode(getApplicationContext(), PowerManager.PARTIAL_WAKE_LOCK);
        mediaPlayer.setAudioStreamType(AudioManager.STREAM_MUSIC);
        mediaPlayer.setVolume(1.0f, 1.0f);

        ComponentName mediaButtonReceiver = new ComponentName(getApplicationContext(), MediaButtonReceiver.class);
        mediaSession = new MediaSessionCompat(getApplicationContext(), LOG_TAG, mediaButtonReceiver, null);
        mediaSession.setCallback(mediaSessionCallback);
        mediaSession.setFlags(MediaSessionCompat.FLAG_HANDLES_MEDIA_BUTTONS | MediaSessionCompat.FLAG_HANDLES_TRANSPORT_CONTROLS);

        stateBuilder = new PlaybackStateCompat.Builder().setActions(PlaybackStateCompat.ACTION_PLAY | PlaybackStateCompat.ACTION_PLAY_PAUSE | PlaybackStateCompat.ACTION_STOP);
        mediaSession.setPlaybackState(stateBuilder.build());

        Intent mediaButtonIntent = new Intent(Intent.ACTION_MEDIA_BUTTON);
        mediaButtonIntent.setClass(this, MediaButtonReceiver.class);
        PendingIntent pendingIntent = PendingIntent.getBroadcast(this, 0, mediaButtonIntent, 0);
        mediaSession.setMediaButtonReceiver(pendingIntent);

        setSessionToken(mediaSession.getSessionToken());

        NotificationCompat.Builder builder = new NotificationCompat.Builder(getBaseContext(), AUDIO_CHANNEL);
        builder.setContentTitle(currentMusicTitle != null ? currentMusicTitle : "Untitled")
                .setContentText(currentMusicDescription != null ? currentMusicDescription : "")
                .setSubText(getApplicationName(getApplicationContext()))
                .setContentIntent(mediaSession.getController()
                        .getSessionActivity())
                .setOngoing(true)
                .setDeleteIntent(MediaButtonReceiver.buildMediaButtonPendingIntent(getBaseContext(), PlaybackStateCompat.ACTION_STOP))
                .setVisibility(NotificationCompat.VISIBILITY_PUBLIC)
                .addAction(new NotificationCompat.Action(R.drawable.ic_baseline_pause_24px, "Pause", MediaButtonReceiver.buildMediaButtonPendingIntent(getBaseContext(), PlaybackStateCompat.ACTION_PLAY_PAUSE)))
                .setStyle(new android.support.v4.media.app.NotificationCompat.MediaStyle().setShowActionsInCompactView(0)
                        .setMediaSession(mediaSession.getSessionToken()))
                .setSmallIcon(R.drawable.ic_audio);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            startForeground(1,builder.build());
        }

        Logger.i("Created");
    }

    @Override
    public void onDestroy()
    {
        super.onDestroy();
        mediaPlayer.stop();
        mediaPlayer.reset();
        mediaSession.release();
        NotificationManagerCompat.from(this)
                .cancel(1);
        Logger.i("Destroyed");
    }

    private void setMediaPlaybackState(int state)
    {
        AudioService.playingState = state;

        PlaybackStateCompat.Builder playbackstateBuilder = new PlaybackStateCompat.Builder();
        if (state == PlaybackStateCompat.STATE_PLAYING)
        {
            playbackstateBuilder.setActions(PlaybackStateCompat.ACTION_PLAY_PAUSE | PlaybackStateCompat.ACTION_PAUSE | PlaybackStateCompat.ACTION_STOP);
        }
        else if (state == PlaybackStateCompat.STATE_PAUSED)
        {
            playbackstateBuilder.setActions(PlaybackStateCompat.ACTION_PLAY_PAUSE | PlaybackStateCompat.ACTION_PLAY | PlaybackStateCompat.ACTION_STOP);
        }
        else if (state == PlaybackStateCompat.STATE_STOPPED)
        {
            playbackstateBuilder.setActions(PlaybackStateCompat.ACTION_STOP);
        }
        playbackstateBuilder.setState(state, PlaybackStateCompat.PLAYBACK_POSITION_UNKNOWN, 0);
        Bundle extraBundle = new Bundle();
        extraBundle.putString(ARG_URL, currentMusicURL);
        playbackstateBuilder.setExtras(extraBundle);
        mediaSession.setPlaybackState(playbackstateBuilder.build());
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId)
    {
        MediaButtonReceiver.handleIntent(mediaSession, intent);
        return START_NOT_STICKY;
    }

    @Nullable
    @Override
    public BrowserRoot onGetRoot(@NonNull String clientPackageName, int clientUid, @Nullable Bundle rootHints)
    {
        return new BrowserRoot("Audio_Concierge_Root", null);
    }

    @Override
    public void onLoadChildren(@NonNull String parentId, @NonNull Result<List<MediaBrowserCompat.MediaItem>> result)
    {
        result.sendResult(null);
    }

    private boolean successfullyRetrievedAudioFocus()
    {
        AudioManager audioManager = (AudioManager) getSystemService(Context.AUDIO_SERVICE);

        int result = audioManager.requestAudioFocus(this, AudioManager.STREAM_MUSIC, AudioManager.AUDIOFOCUS_GAIN);

        return result == AudioManager.AUDIOFOCUS_GAIN;
    }

    private static String getApplicationName(Context context)
    {
        ApplicationInfo applicationInfo = context.getApplicationInfo();
        int stringId = applicationInfo.labelRes;
        return stringId == 0 ? applicationInfo.nonLocalizedLabel.toString() : context.getString(stringId);
    }

    @Override
    public void onAudioFocusChange(int focusChange)
    {
        switch (focusChange)
        {
            case AudioManager.AUDIOFOCUS_LOSS:
            {
                if (mediaPlayer.isPlaying())
                {
                    mediaSession.getController()
                            .getTransportControls()
                            .pause();
                }
                break;
            }
            case AudioManager.AUDIOFOCUS_LOSS_TRANSIENT:
            {
                mediaSession.getController()
                        .getTransportControls()
                        .pause();
                break;
            }
            case AudioManager.AUDIOFOCUS_LOSS_TRANSIENT_CAN_DUCK:
            {
                if (mediaPlayer != null)
                {
                    mediaPlayer.setVolume(0.3f, 0.3f);
                }
                break;
            }
            case AudioManager.AUDIOFOCUS_GAIN:
            {
                if (mediaPlayer != null)
                {
                    if (!mediaPlayer.isPlaying())
                    {
                        mediaSession.getController()
                                .getTransportControls()
                                .play();
                    }
                    mediaPlayer.setVolume(1.0f, 1.0f);
                }
                break;
            }
        }
    }
}
