package com.flybits.concierge.viewmodels

import android.app.Activity
import android.app.Application
import android.arch.lifecycle.AndroidViewModel
import android.arch.lifecycle.LiveData
import android.arch.paging.LivePagedListBuilder
import android.arch.paging.PagedList
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.os.Handler
import android.text.TextUtils
import android.util.Log
import com.flybits.android.kernel.ContentAnalytics

import com.flybits.android.kernel.db.KernelDatabase
import com.flybits.android.kernel.db.dao.ContentDao
import com.flybits.android.kernel.models.Content
import com.flybits.android.kernel.models.results.ContentResult
import com.flybits.android.kernel.utilities.ContentParameters
import com.flybits.commons.library.api.FlybitsManager
import com.flybits.commons.library.api.results.callbacks.ConnectionResultCallback
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback
import com.flybits.commons.library.exceptions.APIUsageExceededException
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.exceptions.NotConnectedException
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.ConciergeConfiguration
import com.flybits.concierge.ConciergeConstants
import com.flybits.concierge.FlybitsConcierge
import com.flybits.concierge.R
import com.flybits.concierge.models.BaseTemplate
import com.twitter.sdk.android.core.DefaultLogger
import com.twitter.sdk.android.core.TwitterAuthConfig
import com.twitter.sdk.android.core.TwitterConfig
import java.util.*
import java.util.concurrent.*

import kotlin.concurrent.schedule

class FeedViewModel(application: Application) : AndroidViewModel(application), PagedResultCallback<Content> {

    data class ScrollEvent(val visibleItems: List<BaseTemplate>)

    interface FeedErrorDisplayer {
        fun onError(err: String)
    }

    interface RefreshResultListener {
        fun onFinished()
    }

    companion object {
        const val PAGE_SIZE = 15
        const val PRE_FETCH_DISTANCE = 5
        const val INITIAL_LOAD_SIZE = PAGE_SIZE * 3
        const val VIEWING_THRESHOLD = 250L
    }

    private var contentResult: ContentResult? = null
    private var refreshResultListener: RefreshResultListener? = null
    private var feedErrorDisplayer: FeedErrorDisplayer? = null
    private val uiHandler: Handler
    private val contentDao: ContentDao
    private val receiver: BroadcastReceiver?
    private val executorService: ExecutorService
    private var loadedAll: Boolean = false
    private val currentTaskList: ConcurrentHashMap<String, TimerTask> = ConcurrentHashMap()
    private var contentAnalytics: ContentAnalytics? = null
    private val scrollEventQueue: BlockingQueue<ScrollEvent> = LinkedBlockingQueue()

    val feedContent: LiveData<PagedList<BaseTemplate?>>
        get() {
            loadedAll = false
            val config = PagedList.Config.Builder()
                    .setInitialLoadSizeHint(INITIAL_LOAD_SIZE)
                    .setPageSize(PAGE_SIZE)
                    .setEnablePlaceholders(false)
                    .setPrefetchDistance(PRE_FETCH_DISTANCE)
                    .build()

            val typeList = ArrayList<String>()
            for (viewProvider in FlybitsConcierge.with(getApplication()).flybitsViewProviders) {
                typeList.add(viewProvider.getContentType())
            }
            return LivePagedListBuilder(contentDao.getDataSourceContentByTypeList(typeList).map { input ->
                val flybitsViewProvider = FlybitsConcierge.with(getApplication()).getFlybitsViewProvider(input.type)
                if (flybitsViewProvider == null) {
                    Logger.w("DEVELOPER: Received content type for which no flybits view providers are registered. Content-type: " + input.type)
                    null
                } else
                    BaseTemplate.fromContent(flybitsViewProvider.getClassType(), input, getApplication())
            }, config).setBoundaryCallback(object : PagedList.BoundaryCallback<BaseTemplate?>() {

                override fun onZeroItemsLoaded() {
                    Logger.d("onZeroItemsLoaded()")
                    requestContent()
                }

                override fun onItemAtEndLoaded(itemAtEnd: BaseTemplate) {
                    Logger.d("onItemAtEndLoaded() ")
                    if (!loadedAll) {
                        contentResult = contentResult!!.getMore(this@FeedViewModel) as ContentResult
                    }
                }
            }).build()
        }

    init {

        setupScrollHandler()

        loadedAll = false

        contentAnalytics = ContentAnalytics(application.applicationContext)

        val twitterConfig = FlybitsConcierge.with(application)
                .configuration
        val twitterKey = twitterConfig.getApiKey(ConciergeConfiguration.KEY_TWITTER_CONSUMER)
        val twitterSecret = twitterConfig.getApiKey(ConciergeConfiguration.KEY_TWITTER_SECRET)

        if (!TextUtils.isEmpty(twitterKey) && !TextUtils.isEmpty(twitterSecret)) {
            val config = TwitterConfig.Builder(application).logger(DefaultLogger(Log.DEBUG))
                    .twitterAuthConfig(TwitterAuthConfig(twitterKey, twitterSecret))
                    .debug(true)
                    .build()
            com.twitter.sdk.android.core.Twitter.initialize(config)
        }
        uiHandler = Handler(application.mainLooper)
        executorService = Executors.newSingleThreadExecutor()

        contentDao = KernelDatabase.getDatabase(application).contentDao()

        // register for new updates from push notifications
        receiver = object : BroadcastReceiver() {
            override fun onReceive(context: Context, intent: Intent) {
                refresh(null)
            }
        }

        val intentFilter = IntentFilter()
        intentFilter.addAction(ConciergeConstants.BROADCAST_CONTENT_ADD)
        intentFilter.addAction(ConciergeConstants.BROADCAST_CONTENT_UPDATE)
        intentFilter.addAction(ConciergeConstants.BROADCAST_CONTENT_REMOVE)
        intentFilter.addAction(ConciergeConstants.BROADCAST_RULE_UPDATE)
        intentFilter.addAction(ConciergeConstants.BROADCAST_RULE_STATE)

        application.registerReceiver(receiver, intentFilter)

        refresh(null)
    }

    override fun onCleared() {
        if (receiver != null) {
            getApplication<Application>().unregisterReceiver(receiver)
        }

        super.onCleared()
    }

    fun refresh(refreshResultListener: RefreshResultListener?) {
        loadedAll = false
        contentResult = null
        requestContent()
        this.refreshResultListener = refreshResultListener
    }

    private fun requestContent() {
        if (contentResult == null) {
            val params = ContentParameters.Builder()
                    .setPaging(PAGE_SIZE.toLong(), 0)
                    .setCaching(getApplication<Application>().getString(R.string.flybits_con_cache_key), INITIAL_LOAD_SIZE)
                    .build()
            contentResult = Content.get(getApplication<Application>().applicationContext, params, this)
        }
    }

    fun setFeedErrorDisplayer(feedErrorDisplayer: FeedErrorDisplayer) {
        this.feedErrorDisplayer = feedErrorDisplayer
    }

    override fun onSuccess(items: ArrayList<Content>) {
        Logger.d("onSuccess() items count: " + items.size)
        uiHandler.post {
            if (refreshResultListener != null) {
                refreshResultListener!!.onFinished()
                refreshResultListener = null
            }
        }

        // Remove content types that should not be in the feed.
        val ids = ArrayList<String>()
        for (content in items) {
            when (content.type) {
                ConciergeConstants.SURVEY_CONTENT_TYPE, ConciergeConstants.ONBOARDING_CONTENT_TYPE -> ids.add(content.id)
            }
        }

        if (!ids.isEmpty()) {
            executorService.submit { contentDao.deleteByIds(ids) }
        }
    }

    override fun onException(exception: FlybitsException) {
        uiHandler.post(object : Runnable {
            override fun run() {
                Logger.exception(javaClass.simpleName, exception)

                if (exception is NotConnectedException || exception is APIUsageExceededException) {
                    FlybitsConcierge.with(getApplication()).unauthenticateWithoutLogout(exception)
                }
                if (refreshResultListener != null) {
                    refreshResultListener!!.onFinished()
                    refreshResultListener = null
                }
                if (feedErrorDisplayer != null) {
                    feedErrorDisplayer!!.onError(getApplication<Application>().getString(R.string.flybits_con_generic_error))
                }
            }
        })

    }

    override fun onLoadedAllItems() {
        loadedAll = true
    }

    fun connectAndDownLoad(activity: Activity, metaDataId: String, callback: ObjectResultCallback<Content>?) {
        FlybitsManager.isConnected(activity, true, object : ConnectionResultCallback {
            override fun onConnected() {
                Content.getById(activity, metaDataId, callback)
            }

            override fun notConnected() {

            }

            override fun onException(exception: FlybitsException) {
                uiHandler.post {
                    callback?.onException(exception)
                }

            }
        })

    }

    //these scroll events need to be queued to prevent a concurrent modification exception
    fun onScroll(visibleItems: List<BaseTemplate>){
        scrollEventQueue.add(ScrollEvent(visibleItems))
    }

    private fun setupScrollHandler(){
        Executors.newSingleThreadExecutor().execute {
            while (true){
                val visibleItems = scrollEventQueue.take().visibleItems

                val visibleContentList = visibleItems.asSequence().map { it.content }.filter { it != null }.toList()

                //Cancel then remove tasks which disappeared from the screen after scrolling
                currentTaskList.keys.asSequence()
                        .filter {
                            //Filter out all the content that can be found in the visible list
                            visibleContentList.find { item -> item.id == it } == null
                        }.forEach {
                            //Cancel the timer for the view that disappeared
                            currentTaskList[it]?.cancel()
                            currentTaskList.remove(it)
                        }

                //Schedule the view actions for views which just appeared on the screen after scrolling
                visibleContentList.forEach {
                    //This timer will be canceled if this view disappears before the 250ms timer is done
                    if (currentTaskList[it.id] == null){
                        currentTaskList[it.id] = Timer().schedule(VIEWING_THRESHOLD){
                            contentAnalytics?.trackViewed(it, System.currentTimeMillis())
                            currentTaskList.remove(it.id)
                        }
                    }
                }
            }
        }
    }
}
