package com.flybits.concierge.viewmodels

import android.app.Activity
import android.arch.lifecycle.LiveData
import android.arch.lifecycle.ViewModel
import android.arch.paging.DataSource
import android.arch.paging.LivePagedListBuilder
import android.arch.paging.PagedList
import com.flybits.android.kernel.ContentAnalytics

import com.flybits.android.kernel.models.Content
import com.flybits.android.kernel.models.results.ContentResult
import com.flybits.android.kernel.utilities.ContentParameters
import com.flybits.commons.library.api.FlybitsManager
import com.flybits.commons.library.api.results.callbacks.ConnectionResultCallback
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback
import com.flybits.commons.library.exceptions.APIUsageExceededException
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.exceptions.NotConnectedException
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.*
import com.flybits.concierge.models.BaseTemplate
import java.util.*
import java.util.concurrent.*

import kotlin.concurrent.schedule

class CategoryViewModel(private val category: String
                        , private val flybitsConcierge: FlybitsConcierge
                        , private val contentRepository: ContentRepository
                        , private val contentAnalytics: ContentAnalytics
                        , private val executorService: ExecutorService
                        , private val resourceProvider: ResourceProvider) : ViewModel(), PagedResultCallback<Content> {

    data class ScrollEvent(val visibleItems: List<BaseTemplate>)

    interface FeedErrorDisplayer {
        fun onError(err: String)
    }

    interface RefreshResultListener {
        fun onFinished()
    }

    companion object {
        const val PAGE_SIZE = 20
        const val PRE_FETCH_DISTANCE = 10
        const val INITIAL_LOAD_SIZE = PAGE_SIZE
        const val VIEWING_THRESHOLD = 250L
    }

    private var contentResult: ContentResult? = null
    private var refreshResultListener: RefreshResultListener? = null
    private var feedErrorDisplayer: FeedErrorDisplayer? = null
    private var loadedAll: Boolean = false
    private val currentTaskList: ConcurrentHashMap<String, TimerTask> = ConcurrentHashMap()
    private val scrollEventQueue: BlockingQueue<ScrollEvent> = LinkedBlockingQueue()
    private var contentDataSource: DataSource<Int, BaseTemplate>? = null

    val feedContent: LiveData<PagedList<BaseTemplate?>>
        get() {
            loadedAll = false
            val config = PagedList.Config.Builder()
                    .setInitialLoadSizeHint(INITIAL_LOAD_SIZE)
                    .setPageSize(PAGE_SIZE)
                    .setEnablePlaceholders(true)
                    .setPrefetchDistance(PRE_FETCH_DISTANCE)
                    .build()

            val typeList = ArrayList<String>()
            for (viewProvider in flybitsConcierge.flybitsViewProviders) {
                typeList.add(viewProvider.getContentType())
            }
            Logger.d("Category: $category, returning LivePagedList")
            val contentDataSourceFactory = contentRepository.getContentLocal(category){
                contentDataSource = it
            }
            return LivePagedListBuilder(contentDataSourceFactory, config)
                    .setBoundaryCallback(object : PagedList.BoundaryCallback<BaseTemplate?>() {

                override fun onZeroItemsLoaded() {
                    Logger.d("onZeroItemsLoaded()")
                    requestContent()
                }

                override fun onItemAtEndLoaded(itemAtEnd: BaseTemplate) {
                    Logger.d("Category: $category, onItemAtEndLoaded() ")
                    if (!loadedAll) {
                        contentResult = contentResult?.getMore(this@CategoryViewModel) as ContentResult?
                    }
                }
            }).build()
        }

    init {

        setupScrollHandler()

        loadedAll = false

//        val intentFilter = IntentFilter()
//        intentFilter.addAction(ConciergeConstants.BROADCAST_CONTENT_ADD)
//        intentFilter.addAction(ConciergeConstants.BROADCAST_CONTENT_UPDATE)
//        intentFilter.addAction(ConciergeConstants.BROADCAST_CONTENT_REMOVE)
//        intentFilter.addAction(ConciergeConstants.BROADCAST_RULE_UPDATE)
//        intentFilter.addAction(ConciergeConstants.BROADCAST_RULE_STATE)
//
//        application.registerReceiver(receiver, intentFilter)
    }

    override fun onCleared() {
//        if (receiver != null) {
//            getApplication<Application>().unregisterReceiver(receiver)
//        }

        super.onCleared()
    }

    fun refresh(refreshResultListener: RefreshResultListener?) {
        loadedAll = false
        contentResult = null
        requestContent()
        this.refreshResultListener = refreshResultListener
    }

    private fun requestContent() {
        if (contentResult == null) {
            val paramsBuilder = ContentParameters.Builder()
                    .setPaging(PAGE_SIZE.toLong(), 0)
                    .setCaching(category, INITIAL_LOAD_SIZE)

            if (category != "Home"){
                paramsBuilder.setLabels(category)
            }

            val params = paramsBuilder.build()
            contentResult = contentRepository.getContent(params,this)
        }
    }

    fun setFeedErrorDisplayer(feedErrorDisplayer: FeedErrorDisplayer) {
        this.feedErrorDisplayer = feedErrorDisplayer
    }

    override fun onSuccess(items: ArrayList<Content>) {
        Logger.d("Category: $category, onSuccess() items count: " + items.size)
        if (refreshResultListener != null) {
            refreshResultListener!!.onFinished()
            refreshResultListener = null
        }

        // Remove content types that should not be in the feed.
        val ids = ArrayList<String>()
        for (content in items) {
            when (content.type) {
                ConciergeConstants.SURVEY_CONTENT_TYPE, ConciergeConstants.ONBOARDING_CONTENT_TYPE -> ids.add(content.id)
            }
        }

        if (!ids.isEmpty()) {
            executorService.submit {
                contentRepository.removeContent(ids)
            }
        }
        contentDataSource?.invalidate()
    }

    override fun onException(exception: FlybitsException) {
        Logger.exception(javaClass.simpleName, exception)

        if (exception is NotConnectedException || exception is APIUsageExceededException) {
            flybitsConcierge.unauthenticateWithoutLogout(exception)
        }
        if (refreshResultListener != null) {
            refreshResultListener!!.onFinished()
            refreshResultListener = null
        }
        if (feedErrorDisplayer != null) {
            feedErrorDisplayer!!.onError(resourceProvider.getString(R.string.flybits_con_generic_error))
        }

    }

    override fun onLoadedAllItems() {
        loadedAll = true
    }

    fun connectAndDownLoad(activity: Activity, metaDataId: String, callback: ObjectResultCallback<Content>?) {
        FlybitsManager.isConnected(activity, true, object : ConnectionResultCallback {
            override fun onConnected() {
                Content.getById(activity, metaDataId, callback)
            }

            override fun notConnected() {

            }

            override fun onException(exception: FlybitsException) {
                callback?.onException(exception)
            }
        })

    }

    //these scroll events need to be queued to prevent a concurrent modification exception
    fun onScroll(visibleItems: List<BaseTemplate>){
        scrollEventQueue.add(ScrollEvent(visibleItems))
    }

    private fun setupScrollHandler(){
        Executors.newSingleThreadExecutor().execute {
            while (true){
                val visibleItems = scrollEventQueue.take().visibleItems

                val visibleContentList = visibleItems.asSequence().map { it.content }.filter { it != null }.toList()

                //Cancel then remove tasks which disappeared from the screen after scrolling
                currentTaskList.keys.asSequence()
                        .filter {
                            //Filter out all the content that can be found in the visible list
                            visibleContentList.find { item -> item.id == it } == null
                        }.forEach {
                            //Cancel the timer for the view that disappeared
                            currentTaskList[it]?.cancel()
                            currentTaskList.remove(it)
                        }

                //Schedule the view actions for views which just appeared on the screen after scrolling
                visibleContentList.forEach {
                    //This timer will be canceled if this view disappears before the 250ms timer is done
                    if (currentTaskList[it.id] == null){
                        currentTaskList[it.id] = Timer().schedule(VIEWING_THRESHOLD){
                            contentAnalytics.trackViewed(it, System.currentTimeMillis())
                            currentTaskList.remove(it.id)
                        }
                    }
                }
            }
        }
    }
}
