package com.flybits.concierge

import android.arch.paging.DataSource
import android.arch.paging.PositionalDataSource
import android.content.Context
import com.flybits.android.kernel.db.dao.ContentDao
import com.flybits.android.kernel.models.Content
import com.flybits.android.kernel.models.results.ContentResult
import com.flybits.android.kernel.utilities.ContentParameters
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.models.BaseTemplate
import com.flybits.internal.db.CommonsDatabase

class ContentRepository(private val context: Context, val contentDao: ContentDao
                        , private val flybitsViewProviderGetter: FlybitsViewProviderGetter) {

    class BaseTemplateDataSource(private val cacheId: String, private val flybitsViewProviderGetter: FlybitsViewProviderGetter
                                 , private val contentDao: ContentDao, private val context: Context): PositionalDataSource<BaseTemplate>() {

        companion object {

            fun getFactory(cacheId: String, flybitsViewProviderGetter: FlybitsViewProviderGetter
                           , contentDao: ContentDao, context: Context
                           , onCreate: (dataSource: DataSource<Int, BaseTemplate>) -> Unit): DataSource.Factory<Int, BaseTemplate>{
                return object: DataSource.Factory<Int, BaseTemplate>(){
                    override fun create(): DataSource<Int, BaseTemplate> {
                        val datasource = BaseTemplateDataSource(cacheId, flybitsViewProviderGetter, contentDao, context)
                        onCreate(datasource)
                        return datasource
                    }
                }
            }
        }

        fun count() = CommonsDatabase.getDatabase(context).cachingEntryDAO().getIdsByCachingKey(cacheId).size

        //Do not run this on background thread or content will stop appearing
        override fun loadRange(params: LoadRangeParams, callback: LoadRangeCallback<BaseTemplate>) {
            Logger.d("Category: $cacheId, loadRange() start: ${params.startPosition} size: ${params.loadSize}")
            val queryResult = query.invoke(params.startPosition, params.loadSize)
            callback.onResult(queryResult)
        }

        override fun loadInitial(params: LoadInitialParams, callback: LoadInitialCallback<BaseTemplate>) {
            val totalCount = count()
            val initialLoadPosition = PositionalDataSource.computeInitialLoadPosition(params, totalCount)
            val loadSize = PositionalDataSource.computeInitialLoadSize(params, initialLoadPosition, totalCount)
            Logger.d("Category: $cacheId, loadInitial() startPos: $initialLoadPosition, totalCount: $totalCount, loadSize: $loadSize")
            val queryResult = query(initialLoadPosition, loadSize)
            if (queryResult.size == loadSize){
                callback.onResult(queryResult, initialLoadPosition, totalCount)
            }else{
                Logger.d("Category: $cacheId, invalidating data source!")
                invalidate()
            }
        }

        private val query: (offset: Int, limit: Int) -> List<BaseTemplate> = { offset, limit ->
            val contentList = contentDao.getByCacheId(context, cacheId, limit, offset)
            Logger.d("limit: $limit, offset: $offset, got content: $contentList")
            val templates = contentList.asSequence().map {
                val baseTemplate = BaseTemplate.fromContent(flybitsViewProviderGetter, it, context)
                Logger.d("basetemplate mapped: ${baseTemplate?.content?.id}")
                baseTemplate
            }.filter { it != null }.toList()
            Logger.d("Category: $cacheId, templates after mapping: ${templates.map { it.content.sequence }}")
            templates
        }

    }

    fun getContent(contentParameters: ContentParameters, pagedResultCallback: PagedResultCallback<Content>): ContentResult{
        return Content.get(context.applicationContext, contentParameters, pagedResultCallback)
    }

    fun getContentLocal(cacheId: String, onCreate: (dataSource: DataSource<Int, BaseTemplate>) -> Unit): DataSource.Factory<Int, BaseTemplate>{
        return BaseTemplateDataSource.getFactory(cacheId, flybitsViewProviderGetter, contentDao, context, onCreate)
    }

    fun removeContent(ids: List<String>){
        contentDao.deleteByIds(ids)
    }
}