package com.flybits.concierge.viewmodels;

import android.app.Activity;
import android.app.Application;
import android.arch.core.util.Function;
import android.arch.lifecycle.AndroidViewModel;
import android.arch.lifecycle.LiveData;
import android.arch.paging.LivePagedListBuilder;
import android.arch.paging.PagedList;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Handler;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.util.Log;

import com.flybits.android.kernel.db.KernelDatabase;
import com.flybits.android.kernel.db.dao.ContentDao;
import com.flybits.android.kernel.models.Content;
import com.flybits.android.kernel.models.results.ContentResult;
import com.flybits.android.kernel.utilities.ContentParameters;
import com.flybits.commons.library.api.FlybitsManager;
import com.flybits.commons.library.api.results.callbacks.ConnectionResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.exceptions.APIUsageExceededException;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.exceptions.NotConnectedException;
import com.flybits.commons.library.logging.Logger;
import com.flybits.concierge.ConciergeConfiguration;
import com.flybits.concierge.ConciergeConstants;
import com.flybits.concierge.FlybitsConcierge;
import com.flybits.concierge.FlybitsViewProvider;
import com.flybits.concierge.R;
import com.flybits.concierge.models.BaseTemplate;
import com.twitter.sdk.android.core.DefaultLogger;
import com.twitter.sdk.android.core.TwitterAuthConfig;
import com.twitter.sdk.android.core.TwitterConfig;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public class FeedViewModel extends AndroidViewModel implements PagedResultCallback<Content>
{

    public static final int PAGE_SIZE = 15;
    public static final int PRE_FETCH_DISTANCE = 5;
    public static final int INITIAL_LOAD_SIZE = PAGE_SIZE * 3;

    private ContentResult contentResult;
    private RefreshResultListener refreshResultListener;
    private FeedErrorDisplayer feedErrorDisplayer;
    private Handler uiHandler;
    private ContentDao contentDao;
    private BroadcastReceiver receiver;
    private ExecutorService executorService;
    private boolean loadedAll;

    public interface FeedErrorDisplayer{
        void onError(String err);
    }

    public interface RefreshResultListener
    {
        void onFinished();
    }

    public FeedViewModel(@NonNull Application application)
    {
        super(application);

        loadedAll = false;

        ConciergeConfiguration twitterConfig = FlybitsConcierge.with(application)
                .getConfiguration();
        String twitterKey = twitterConfig.getApiKey(ConciergeConfiguration.KEY_TWITTER_CONSUMER);
        String twitterSecret = twitterConfig.getApiKey(ConciergeConfiguration.KEY_TWITTER_SECRET);

        if (!TextUtils.isEmpty(twitterKey) && !TextUtils.isEmpty(twitterSecret))
        {
            TwitterConfig config = new TwitterConfig.Builder(application).logger(new DefaultLogger(Log.DEBUG))
                    .twitterAuthConfig(new TwitterAuthConfig(twitterKey, twitterSecret))
                    .debug(true)
                    .build();
            com.twitter.sdk.android.core.Twitter.initialize(config);
        }
        uiHandler = new Handler(application.getMainLooper());
        executorService = Executors.newSingleThreadExecutor();

        contentDao = KernelDatabase.getDatabase(application).contentDao();

        // register for new updates from push notifications
        receiver = new BroadcastReceiver()
        {
            @Override
            public void onReceive(Context context, Intent intent)
            {
                refresh(null);
            }
        };

        IntentFilter intentFilter = new IntentFilter();
        intentFilter.addAction(ConciergeConstants.BROADCAST_CONTENT_ADD);
        intentFilter.addAction(ConciergeConstants.BROADCAST_CONTENT_UPDATE);
        intentFilter.addAction(ConciergeConstants.BROADCAST_CONTENT_REMOVE);
        intentFilter.addAction(ConciergeConstants.BROADCAST_RULE_UPDATE);
        intentFilter.addAction(ConciergeConstants.BROADCAST_RULE_STATE);

        application.registerReceiver(receiver, intentFilter);

        refresh(null);
    }

    @Override
    protected void onCleared()
    {
        if (receiver != null){
            getApplication().unregisterReceiver(receiver);
        }

        super.onCleared();
    }

    public void refresh(RefreshResultListener refreshResultListener)
    {
        loadedAll = false;
        contentResult = null;
        requestContent();
        this.refreshResultListener = refreshResultListener;
    }

    private void requestContent()
    {
        if (contentResult == null){
            ContentParameters params = new ContentParameters.Builder()
                    .setPaging(PAGE_SIZE, 0)
                    .setCaching(getApplication().getString(R.string.flybits_con_cache_key), INITIAL_LOAD_SIZE)
                    .build();
            contentResult = Content.get(getApplication().getApplicationContext(), params, this);
        }
    }

    public void setFeedErrorDisplayer(FeedErrorDisplayer feedErrorDisplayer){
        this.feedErrorDisplayer = feedErrorDisplayer;
    }

    public LiveData<PagedList<BaseTemplate>> getFeedContent()
    {

        loadedAll = false;
        PagedList.Config config = new PagedList.Config.Builder()
                .setInitialLoadSizeHint(INITIAL_LOAD_SIZE)
                .setPageSize(PAGE_SIZE)
                .setEnablePlaceholders(false)
                .setPrefetchDistance(PRE_FETCH_DISTANCE)
                .build();

        List<String> typeList = new ArrayList<>();
        for (FlybitsViewProvider f: FlybitsConcierge.with(getApplication()).getFlybitsViewProviders()){
            typeList.add(f.getContentType());
        }
        return new LivePagedListBuilder<>(contentDao.getDataSourceContentByTypeList(typeList).map(new Function<Content,BaseTemplate>() {
            @Override
            public BaseTemplate apply(Content input) {
                FlybitsViewProvider flybitsViewProvider = FlybitsConcierge.with(getApplication()).getFlybitsViewProvider(input.getType());
                if (flybitsViewProvider == null){
                    Logger.w("DEVELOPER: Received content type for which no flybits view providers are registered. Content-type: "+input.getType());
                    return null;
                }
                else return BaseTemplate.fromContent(flybitsViewProvider.getClassType(), input, getApplication());
            }
        }),config).setBoundaryCallback(new PagedList.BoundaryCallback<BaseTemplate>() {
            @Override
            public void onZeroItemsLoaded() {
                Logger.d("onZeroItemsLoaded()");
                requestContent();
            }

            @Override
            public void onItemAtEndLoaded(@NonNull BaseTemplate itemAtEnd) {
                Logger.d("onItemAtEndLoaded() ");
                if (!loadedAll){
                    contentResult = (ContentResult) contentResult.getMore(FeedViewModel.this);
                }
            }
        }).build();
    }

    @Override
    public void onSuccess(ArrayList<Content> items)
    {
        Logger.d("onSuccess() items count: " + items.size());
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                if (refreshResultListener != null)
                {
                    refreshResultListener.onFinished();
                    refreshResultListener = null;
                }
            }
        });

        // Remove content types that should not be in the feed.
        final List<String> ids = new ArrayList<>();
        for (Content content : items)
        {
            switch (content.getType())
            {
                case ConciergeConstants.SURVEY_CONTENT_TYPE:
                case ConciergeConstants.ONBOARDING_CONTENT_TYPE:
                    ids.add(content.getId());
                    break;
            }
        }

        if (!ids.isEmpty())
        {
            executorService.submit(new Runnable()
            {
                @Override
                public void run()
                {
                    contentDao.deleteByIds(ids);
                }
            });
        }
    }

    @Override
    public void onException(final FlybitsException exception)
    {
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                Logger.exception(getClass().getSimpleName(), exception);

                if (exception instanceof NotConnectedException || exception instanceof APIUsageExceededException){
                    FlybitsConcierge.with(getApplication()).unauthenticateWithoutLogout(exception);
                }
                if (refreshResultListener != null){
                    refreshResultListener.onFinished();
                    refreshResultListener = null;
                }
                if (feedErrorDisplayer != null){
                    feedErrorDisplayer.onError(getApplication().getString(R.string.flybits_con_generic_error));
                }
            }
        });

    }

    @Override
    public void onLoadedAllItems()
    {
        loadedAll = true;
    }

    public void connectAndDownLoad(final Activity activity, final String metaDataId, final ObjectResultCallback<Content> callback)
    {
        FlybitsManager.isConnected(activity, true, new ConnectionResultCallback()
        {
            @Override
            public void onConnected()
            {
                Content.getById(activity, metaDataId, callback);
            }

            @Override
            public void notConnected()
            {

            }

            @Override
            public void onException(final FlybitsException exception)
            {
                uiHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        if (callback != null)
                        {
                            callback.onException(exception);
                        }
                    }
                });

            }
        });

    }
}
