package com.flybits.concierge.models;

import android.os.Parcel;
import android.os.Parcelable;
import android.support.v4.content.ContextCompat;
import android.view.View;
import android.widget.ImageView;
import android.widget.TextView;

import com.flybits.android.kernel.models.LocalizedValue;
import com.flybits.commons.library.logging.Logger;
import com.flybits.concierge.R;
import com.flybits.concierge.Utils;
import com.flybits.concierge.enums.FeedView;

import java.text.ParseException;
import java.util.Calendar;
import java.util.Date;

public class ScheduleSegment extends BaseTemplate implements Parcelable
{
    public LocalizedValue title;
    public String url;
    public String startTime;
    public String endTime;
    public LocalizedValue description;

    private static Calendar today = Calendar.getInstance();

    private Date startDayTime;
    private Date endDayTime;

    public ScheduleSegment()
    {
    }

    protected ScheduleSegment(Parcel in)
    {
        title = in.readParcelable(LocalizedValue.class.getClassLoader());
        url = in.readString();
        startTime = in.readString();
        endTime = in.readString();
        description = in.readParcelable(LocalizedValue.class.getClassLoader());
    }

    public static final Creator<ScheduleSegment> CREATOR = new Creator<ScheduleSegment>()
    {
        @Override
        public ScheduleSegment createFromParcel(Parcel in)
        {
            return new ScheduleSegment(in);
        }

        @Override
        public ScheduleSegment[] newArray(int size)
        {
            return new ScheduleSegment[size];
        }
    };

    @Override
    public int describeContents()
    {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags)
    {
        dest.writeParcelable(title, flags);
        dest.writeString(url);
        dest.writeString(startTime);
        dest.writeString(endTime);
        dest.writeParcelable(description, flags);
    }

    public Date getStartDayTime()
    {
        if (startDayTime != null)
        {
            return startDayTime;
        }

        try
        {
            Calendar calendar = Utils.toCalendar(startTime);
            calendar.set(Calendar.DAY_OF_YEAR, today.get(Calendar.DAY_OF_YEAR));
            startDayTime = calendar.getTime();
        }
        catch (ParseException e)
        {
            Logger.exception(ScheduleSegment.class.getSimpleName(), e);
        }

        return startDayTime;
    }


    public Date getEndDayTime()
    {
        if (endDayTime != null)
        {
            return endDayTime;
        }

        try
        {
            Calendar calendar = Utils.toCalendar(endTime);
            calendar.set(Calendar.DAY_OF_YEAR, today.get(Calendar.DAY_OF_YEAR));
            endDayTime = calendar.getTime();
        }
        catch (ParseException e)
        {
            Logger.exception(ScheduleSegment.class.getSimpleName(), e);
        }

        return endDayTime;
    }

    public void populateViews(TextView timeRangeView, TextView titleView, TextView descriptionView, ImageView bulletOutView, ImageView bulletInView, View containerView)
    {
        String range = Utils.getTimeRange(getStartDayTime(), getEndDayTime());
        timeRangeView.setText(range);
        titleView.setText(title.getValue());
        descriptionView.setText(description.getValue());

        boolean isCurrent = today.getTime().after(startDayTime) && today.getTime().before(endDayTime);
        boolean isPast = today.getTime().after(endDayTime);

        bulletOutView.setColorFilter(ContextCompat.getColor(bulletOutView.getContext(), isCurrent ? R.color.flybits_con_schedule_bullet_out_selected : R.color.flybits_con_schedule_bullet_out), android.graphics.PorterDuff.Mode.MULTIPLY);
        bulletInView.setColorFilter(ContextCompat.getColor(bulletInView.getContext(), isCurrent ? R.color.flybits_con_schedule_bullet_in_selected : R.color.flybits_con_schedule_bullet_in), android.graphics.PorterDuff.Mode.MULTIPLY);

        containerView.setBackgroundColor(ContextCompat.getColor(containerView.getContext(), isPast ? R.color.flybits_con_schedule_segment_item_background_past : (isCurrent ? R.color.flybits_con_schedule_segment_item_background_selected : R.color.flybits_con_schedule_segment_item_background)));

        int textColor = ContextCompat.getColor(titleView.getContext(), isCurrent ? R.color.flybits_white : R.color.flybits_con_body_text_color);
        timeRangeView.setTextColor(textColor);
        titleView.setTextColor(textColor);
        descriptionView.setTextColor(textColor);
    }
}
