package com.flybits.concierge.models;

import android.content.Intent;
import android.net.Uri;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.v4.app.Fragment;
import android.text.SpannableString;
import android.text.TextUtils;
import android.text.style.UnderlineSpan;
import android.view.View;
import android.widget.ImageView;
import android.widget.TextView;

import com.bumptech.glide.Glide;
import com.bumptech.glide.load.DecodeFormat;
import com.bumptech.glide.request.RequestOptions;
import com.flybits.android.kernel.models.LocalizedValue;
import com.flybits.commons.library.logging.Logger;
import com.flybits.concierge.FlybitsNavigator;
import com.flybits.concierge.Utils;
import com.flybits.concierge.fragments.EventDetailsFragment;

import java.text.ParseException;
import java.util.Date;

public class Event extends BaseTemplate implements Parcelable
{
    public LocalizedValue title;
    public String coverPhotoURL;
    public LocalizedValue venueDescription;
    public String url;
    public String address;
    public String phoneNumber;
    public String startDate;
    public String endDate;
    public LocalizedValue description;

    public Event()
    {
    }

    public Date getStartDate()
    {
        try
        {
            return Utils.toCalendar(startDate).getTime();
        }
        catch (ParseException e)
        {
            Logger.exception(Event.class.getSimpleName(), e);
        }

        return null;
    }

    public Date getEndDate()
    {
        try
        {
            return Utils.toCalendar(endDate).getTime();
        }
        catch (ParseException e)
        {
            Logger.exception(Event.class.getSimpleName(), e);
        }

        return null;
    }

    protected Event(Parcel in)
    {
        title = in.readParcelable(LocalizedValue.class.getClassLoader());
        coverPhotoURL = in.readString();
        venueDescription = in.readParcelable(LocalizedValue.class.getClassLoader());
        url = in.readString();
        address = in.readString();
        phoneNumber = in.readString();
        startDate = in.readString();
        endDate = in.readString();
        description = in.readParcelable(LocalizedValue.class.getClassLoader());
    }

    public static final Creator<Event> CREATOR = new Creator<Event>()
    {
        @Override
        public Event createFromParcel(Parcel in)
        {
            return new Event(in);
        }

        @Override
        public Event[] newArray(int size)
        {
            return new Event[size];
        }
    };

    @Override
    public int describeContents()
    {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags)
    {
        dest.writeParcelable(title, flags);
        dest.writeString(coverPhotoURL);
        dest.writeParcelable(venueDescription, flags);
        dest.writeString(url);
        dest.writeString(address);
        dest.writeString(phoneNumber);
        dest.writeString(startDate);
        dest.writeString(endDate);
        dest.writeParcelable(description, flags);
    }

    public void populateViews(FlybitsNavigator flybitsNavigator, TextView titleView, TextView locationView, TextView dateView, ImageView imgHeader, View expandView)
    {
        populateViews(flybitsNavigator, titleView, locationView, dateView, imgHeader, expandView, null, null, null);
    }

    public void populateViews(FlybitsNavigator flybitsNavigator, TextView titleView, TextView locationView, TextView dateView, ImageView imgHeader, final View expandView, TextView addressView, TextView ctaView, TextView phoneView)
    {
        populateViews(flybitsNavigator, titleView, locationView, dateView, imgHeader, expandView, addressView, ctaView, phoneView, null);
    }

    public void populateViews(final FlybitsNavigator flybitsNavigator, TextView titleView, TextView locationView, TextView dateView, ImageView imgHeader, final View expandView, TextView addressView, TextView ctaView, TextView phoneView, TextView descriptionView)
    {
        RequestOptions options = new RequestOptions().centerCrop().format(DecodeFormat.PREFER_RGB_565);

        Glide.with(imgHeader.getContext())
                .load(coverPhotoURL)
                .apply(options)
                .into(imgHeader);

        titleView.setText(!TextUtils.isEmpty(title.getValue()) ? title.getValue() : "");
        locationView.setText(!TextUtils.isEmpty(venueDescription.getValue()) ? venueDescription.getValue() : "");
        dateView.setText(Utils.getDateRange(getStartDate(), getEndDate()));

        if (expandView != null && flybitsNavigator != null)
        {
            expandView.setOnClickListener(new View.OnClickListener()
            {
                @Override
                public void onClick(View v)
                {
                    Fragment fragment = EventDetailsFragment.newInstance(Event.this);
                    flybitsNavigator.openFragment(fragment);
                }
            });
        }

        if (addressView != null)
        {
            if (descriptionView != null)
            {
                SpannableString content = new SpannableString(address);
                content.setSpan(new UnderlineSpan(), 0, content.length(), 0);
                addressView.setText(content);
            }
            else
            {
                addressView.setText(address);
            }
        }

        if (ctaView != null)
        {
            ctaView.setVisibility(!TextUtils.isEmpty(url) ? View.VISIBLE : View.INVISIBLE);

            ctaView.setOnClickListener(new View.OnClickListener()
            {
                @Override
                public void onClick(View v)
                {
                    String url = Event.this.url;
                    Intent intent = new Intent(Intent.ACTION_VIEW);
                    intent.setData(Uri.parse(url));
                    v.getContext().startActivity(intent);
                }
            });
        }

        if (phoneView != null)
        {
            if (!TextUtils.isEmpty(phoneNumber))
            {
                if (descriptionView != null)
                {
                    SpannableString content = new SpannableString(phoneNumber);
                    content.setSpan(new UnderlineSpan(), 0, content.length(), 0);
                    phoneView.setText(content);
                }
                else
                {
                    phoneView.setText(phoneNumber);
                }
            }
            else
            {
                phoneView.setText(android.R.string.unknownName);
            }
        }

        if (descriptionView != null)
        {
            descriptionView.setText(description.getValue());
        }
    }
}
