package com.flybits.concierge.fragments;


import android.app.Activity;
import android.arch.lifecycle.Observer;
import android.arch.lifecycle.ViewModelProviders;
import android.arch.paging.PagedList;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.Uri;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.design.widget.Snackbar;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentActivity;
import android.support.v4.widget.SwipeRefreshLayout;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import com.afollestad.materialdialogs.MaterialDialog;
import com.flybits.android.kernel.models.Content;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.logging.Logger;
import com.flybits.concierge.ConciergeConstants;
import com.flybits.concierge.ConciergeFragment;
import com.flybits.concierge.ContentHelper;
import com.flybits.concierge.R;
import com.flybits.concierge.activities.ConciergeActivity;
import com.flybits.concierge.activities.ConciergePopupActivity;
import com.flybits.concierge.adapters.FeedAdapter;
import com.flybits.concierge.models.Articles;
import com.flybits.concierge.models.BaseTemplate;
import com.flybits.concierge.models.Events;
import com.flybits.concierge.models.Images;
import com.flybits.concierge.models.Links;
import com.flybits.concierge.models.Onboarding;
import com.flybits.concierge.models.Schedule;
import com.flybits.concierge.models.Survey;
import com.flybits.concierge.models.Text;
import com.flybits.concierge.models.Videos;
import com.flybits.concierge.viewmodels.FeedViewModel;
import com.flybits.concierge.viewmodels.TwitterViewModel;

public class FeedFragment extends BasePagerFragment implements FeedViewModel.FeedErrorDisplayer
{
    private RecyclerView recyclerView;
    private FeedViewModel feedViewModel;
    private FeedAdapter feedAdapter;
    private View noDataView;
    private BroadcastReceiver refreshReceiver;
    private SwipeRefreshLayout swipeRefreshLayout;

    public FeedFragment()
    {
        // Required empty public constructor
    }

    public static FeedFragment newInstance()
    {
        FeedFragment fragment = new FeedFragment();
        return fragment;
    }

    @Override
    public int titleRes()
    {
        return R.string.flybits_con_fragment_title_feed;
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState)
    {
        // Inflate the layout for this fragment
        return inflater.inflate(R.layout.flybits_con_fragment_feed, container, false);
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState)
    {
        super.onViewCreated(view, savedInstanceState);

        view.findViewById(R.id.flybits_con_fragment_feed_settings_button).setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                Fragment parentFragment = getParentFragment();
                if (parentFragment != null && parentFragment instanceof ConciergeFragment){
                    ((ConciergeFragment)parentFragment).startSettingsActivity();
                }
            }
        });

        recyclerView = view.findViewById(R.id.flybits_con_fragment_feed_lytRecycler);
        recyclerView.setLayoutManager(new LinearLayoutManager(getContext()));
        feedAdapter = new FeedAdapter((ConciergeFragment) getParentFragment());
        recyclerView.setAdapter(feedAdapter);
        recyclerView.setHasFixedSize(false);

        swipeRefreshLayout = view.findViewById(R.id.flybits_con_swipe_ref_layout);
        swipeRefreshLayout.setOnRefreshListener(new SwipeRefreshLayout.OnRefreshListener() {
            @Override
            public void onRefresh() {
                Logger.d(FeedFragment.class.getSimpleName()+": onRefresh()");
                swipeRefreshLayout.setEnabled(false);
                feedViewModel.refresh(new FeedViewModel.RefreshResultListener() {
                    @Override
                    public void onFinished() {
                        swipeRefreshLayout.setEnabled(true);
                        swipeRefreshLayout.setRefreshing(false);
                    }
                });

                if (getContext() != null)
                {
                    TwitterViewModel twitterViewModel = ViewModelProviders
                            .of((FragmentActivity) getContext())
                            .get(TwitterViewModel.class);

                    twitterViewModel.clearTweetLists();
                }
            }
        });

        noDataView = view.findViewById(R.id.flybits_con_fragment_feed_lytNoData);

        feedViewModel = ViewModelProviders.of(this)
                .get(FeedViewModel.class);

        swipeRefreshLayout.setRefreshing(true);
        feedViewModel.getFeedContent()
                .observe(this, new Observer<PagedList<BaseTemplate>>()
                {
                    @Override
                    public void onChanged(@Nullable PagedList<BaseTemplate> baseTemplates)
                    {
                        Logger.d("Got data! data: "+baseTemplates);
                        feedAdapter.submitList(baseTemplates);
                        boolean baseTemplatesEmpty = baseTemplates.isEmpty();
                        recyclerView.setVisibility(baseTemplatesEmpty ? View.GONE : View.VISIBLE);
                        noDataView.setVisibility(baseTemplatesEmpty ? View.VISIBLE : View.GONE);
                        swipeRefreshLayout.setRefreshing(false);
                    }
                });
        feedViewModel.setFeedErrorDisplayer(this);

        refreshReceiver = new BroadcastReceiver()
        {
            @Override
            public void onReceive(Context context, Intent intent)
            {
                if (feedAdapter != null){
                    try{
                        feedAdapter.notifyDataSetChanged();
                    }catch(Exception e){
                        Logger.exception("FeedFragment.RefreshReceiver.onReceiver()",e);
                    }
                }
            }
        };

        handlePushInvoke();
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        try{
            Activity activity = getActivity();
            if (activity != null){
                activity.registerReceiver(refreshReceiver, new IntentFilter(ConciergeConstants.BROADCAST_AUDIO_STATE));

            }
        }catch(Exception e){
            Logger.exception("FeedFragment.onCreate()",e);
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        //Receiver may already be unregistered or may have never been registered in that case catch the exception
        try{
            Activity activity = getActivity();
            if (activity != null){
                activity.unregisterReceiver(refreshReceiver);
            }
        }catch(Exception e){
            Logger.exception("FeedFragment.onDestroy()",e);
        }
    }

    private void handlePushInvoke()
    {
        final Activity activity = getActivity();

        if (activity == null || activity.getIntent() == null)
        {
            return;
        }

        final String metaDataId = activity.getIntent().getStringExtra(ConciergeConstants.PUSH_CONTENT_ID);
        final String url = activity.getIntent().getStringExtra(ConciergeConstants.PUSH_CONTENT_URL);

        if (metaDataId != null)
        {
            feedViewModel.connectAndDownLoad(activity, metaDataId, new ObjectResultCallback<Content>()
            {

                @Override
                public void onSuccess(Content item)
                {
                    presentContent(item);
                }

                @Override
                public void onException(FlybitsException e)
                {
                    //Make sure activity isn't null once the callback is invoked
                    final Activity currentActivity = getActivity();
                    if (currentActivity != null){
                        new MaterialDialog.Builder(currentActivity)
                                .title(R.string.flybits_con_error)
                                .content(R.string.flybits_con_could_not_load)
                                .positiveText(android.R.string.ok)
                                .dismissListener(new DialogInterface.OnDismissListener()
                                {
                                    @Override
                                    public void onDismiss(DialogInterface dialog)
                                    {
                                        currentActivity.finish();
                                    }
                                }).show();
                    }

                    Logger.exception(FeedFragment.class.getSimpleName(), e);
                }
            });
        }
        else if (url != null)
        {
            openUrl(url);
        }
    }

    private void openUrl(String url)
    {
        Activity activity = getActivity();
        if (activity == null || url == null)
        {
            return;
        }

        Intent intent = new Intent(Intent.ACTION_VIEW);
        intent.setData(Uri.parse(url));
        startActivity(intent);
        activity.finish();
    }

    private void presentContent(Content content)
    {
        BaseTemplate contentObject = ContentHelper.contentToBaseTemplate(content, getContext());
        final Activity activity = getActivity();

        if (activity == null || contentObject == null)
        {
            return;
        }

        Fragment fragment = null;

        switch (content.getType())
        {
            case ConciergeConstants.ARTICLE_CONTENT_TYPE:
                Articles articles = (Articles) contentObject;
                if (articles.articles.getList().size() > 0)
                {
                    openUrl(articles.articles.getList().get(0).url);
                }
                break;
            case ConciergeConstants.EVENT_CONTENT_TYPE:
                Events events = (Events) contentObject;
                fragment = (events.events.getList().size() > 1)
                        ? EventDetailsFragment.newInstance(events.events.getList().get(0))
                        : null;
                break;
            case ConciergeConstants.IMAGE_CONTENT_TYPE:
                Images images = (Images) contentObject;
                fragment = ImageViewerPagerFragment.newInstance(images.images, 0);
                break;
            case ConciergeConstants.LINK_CONTENT_TYPE:
                Links links = (Links) contentObject;
                if (links.links.getList().size() > 0)
                {
                    openUrl(links.links.getList().get(0).url);
                }
                break;
            case ConciergeConstants.ONBOARDING_CONTENT_TYPE:
                Onboarding onboarding = (Onboarding) contentObject;
                fragment = OnboardingFragment.newInstance(onboarding);
                break;
            case ConciergeConstants.SCHEDULE_CONTENT_TYPE:
                Schedule schedule = (Schedule) contentObject;
                fragment = ScheduleFeedFragment.newInstance(schedule);
                break;
            case ConciergeConstants.SURVEY_CONTENT_TYPE:
                Survey survey = (Survey) contentObject;
                fragment = SurveyFragment.newInstance(survey);
                break;
            case ConciergeConstants.TEXT_CONTENT_TYPE:
                Text text = (Text) contentObject;
                new MaterialDialog.Builder(activity)
                        .title(text.header.getValue())
                        .content(text.description.getValue())
                        .positiveText(android.R.string.ok)
                        .dismissListener(new DialogInterface.OnDismissListener()
                        {
                            @Override
                            public void onDismiss(DialogInterface dialog)
                            {
                                activity.finish();
                            }
                        })
                        .show();
                break;
            case ConciergeConstants.VIDEO_CONTENT_TYPE:
                Videos videos = (Videos) contentObject;
                fragment = VideoPlaylistFragment.newInstance(videos.videos.getList());
                break;
            default:
                new MaterialDialog.Builder(activity)
                        .title(R.string.flybits_con_error)
                        .content(R.string.flybits_con_not_supported)
                        .positiveText(android.R.string.ok)
                        .dismissListener(new DialogInterface.OnDismissListener()
                        {
                            @Override
                            public void onDismiss(DialogInterface dialog)
                            {
                                activity.finish();
                            }
                        })
                        .show();
                return;
        }

        ConciergeFragment conciergeFragment = (ConciergeFragment) getParentFragment();
        if (fragment != null && conciergeFragment != null)
        {
            conciergeFragment.openFragment(fragment);
        }
    }

    @Override
    public void onError(String err) {
        Logger.d("FeedFragment: onAuthenticationError() err: "+err);
        Activity activity = getActivity();
        View v = getView();
        /*The instanceof check is done to make sure that the user is still within
        * the concierge feature at the time the toast is displayed.*/
        if (activity != null && (activity instanceof ConciergeActivity
                || activity instanceof ConciergePopupActivity) && v != null){
            Snackbar.make(v,err,Snackbar.LENGTH_LONG).show();
        }
    }
}
