package com.flybits.concierge.fragments;


import android.app.Activity;
import android.arch.lifecycle.ViewModelProviders;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.design.widget.TabLayout;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentManager;
import android.support.v4.app.FragmentPagerAdapter;
import android.support.v4.view.ViewPager;
import android.util.LongSparseArray;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.LinearLayout;

import com.afollestad.materialdialogs.MaterialDialog;
import com.flybits.android.kernel.reservedtemplates.survey.models.Answer;
import com.flybits.android.kernel.reservedtemplates.survey.models.Question;
import com.flybits.android.kernel.reservedtemplates.survey.models.SurveyAnswer;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.logging.Logger;
import com.flybits.concierge.R;
import com.flybits.concierge.Utils;
import com.flybits.concierge.enums.QuestionType;
import com.flybits.concierge.enums.QuestionUIType;
import com.flybits.concierge.models.Survey;
import com.flybits.concierge.viewmodels.SurveyViewModel;
import com.flybits.concierge.views.CustomTabsViewPager;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Locale;

import static com.flybits.concierge.ConciergeConstants.SURVEY_QUESTION_REQUIRED;

public class SurveyFragment extends Fragment
{
    private static final String ARG_PARAM_SURVEY = "flybits_con_arg_survey";

    private SurveyViewModel surveyViewModel;
    private Survey survey;
    private CustomTabsViewPager viewPager;
    private TabLayout tabLayout;
    private Button btnBack;
    private Button btnNext;
    private MaterialDialog progressDialog;
    private Handler uiHandler;

    public SurveyFragment()
    {
        // Required empty public constructor
    }

    public static SurveyFragment newInstance(Survey survey)
    {
        SurveyFragment fragment = new SurveyFragment();
        Bundle args = new Bundle();
        args.putParcelable(ARG_PARAM_SURVEY, survey);
        fragment.setArguments(args);
        return fragment;
    }

    @Override
    public void onCreate(Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);
        if (getArguments() != null)
        {
            survey = getArguments().getParcelable(ARG_PARAM_SURVEY);
        }
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState)
    {
        // Inflate the layout for this fragment
        return inflater.inflate(R.layout.flybits_con_fragment_survey, container, false);
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState)
    {
        super.onViewCreated(view, savedInstanceState);

        uiHandler = new Handler(Looper.getMainLooper());

        surveyViewModel = ViewModelProviders.of(getActivity()).get(SurveyViewModel.class);

        // view pager
        viewPager = view.findViewById(R.id.flybits_con_fragment_survey_lytPager);
        SurveyPagerAdapter adapter = new SurveyPagerAdapter(getChildFragmentManager());
        viewPager.setAdapter(adapter);

        viewPager.addOnPageChangeListener(new ViewPager.OnPageChangeListener()
        {
            @Override
            public void onPageScrolled(int position, float positionOffset, int positionOffsetPixels)
            {

            }

            @Override
            public void onPageSelected(int position)
            {
                changeState();
            }

            @Override
            public void onPageScrollStateChanged(int state)
            {

            }
        });

        // tab layout
        tabLayout = view.findViewById(R.id.flybits_con_fragment_survey_lytTabs);
        tabLayout.setupWithViewPager(viewPager);

        // disable touch on the tab bar on the survey
        View.OnTouchListener disabledListener =  new View.OnTouchListener()
        {
            @Override
            public boolean onTouch(View v, MotionEvent event)
            {
                return true;
            }
        };

        LinearLayout tabStrip = ((LinearLayout)tabLayout.getChildAt(0));
        for(int i = 0; i < tabStrip.getChildCount(); i++)
        {
            tabStrip.getChildAt(i).setOnTouchListener(disabledListener);
        }

        // bottom buttons
        btnBack = view.findViewById(R.id.flybits_con_fragment_survey_btnBack);
        btnBack.setOnClickListener(new View.OnClickListener()
        {
            @Override
            public void onClick(View v)
            {
                int current = viewPager.getCurrentItem();

                if (current == 0)
                {
                    return;
                }

                viewPager.setCurrentItem(current - 1);

                Utils.dismissKeyboard(getActivity());
            }
        });

        btnNext = view.findViewById(R.id.flybits_con_fragment_survey_btnNext);
        btnNext.setOnClickListener(new View.OnClickListener()
        {
            @Override
            public void onClick(View v)
            {
                int current = viewPager.getCurrentItem();

                //Added this to prevent crash if next is pressed on empty survey
                if (survey.questions.getList().isEmpty()){
                    submitAnswers();
                    return;
                }

                // check to see if the question is required
                Question question = survey.questions.getList().get(current);
                try
                {
                    Object answer = surveyViewModel.getAnswers().get(question.questionNumber);
                    Activity activity = getActivity();
                    if (question.getMetadata().getBoolean(SURVEY_QUESTION_REQUIRED) && answer == null && activity != null)
                    {
                        new MaterialDialog.Builder(activity)
                                .title(R.string.flybits_con_fragment_survey_required)
                                .content(String.format(Locale.getDefault(), getString(R.string.flybits_con_fragment_survey_required_message_format), (current + 1)))
                                .positiveText(android.R.string.ok)
                                .show();

                        return;
                    }
                }
                catch (Exception e)
                {
                    Logger.exception(SurveyFragment.class.getSimpleName(), e);
                }

                if (current == viewPager.getAdapter().getCount() - 1)
                {
                    submitAnswers();
                    return;
                }

                viewPager.setCurrentItem(current + 1);

                Utils.dismissKeyboard(getActivity());
            }
        });

        changeState();

        BasicResultCallback resultCallback = new BasicResultCallback()
        {

            @Override
            public void onSuccess()
            {
                uiHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        Activity activity = getActivity();
                        //Activity could be null by the time the callback gets called if user exits it
                        if (activity != null){
                            progressDialog.dismiss();
                            new MaterialDialog.Builder(activity).title(R.string.flybits_con_thank_you)
                                    .content(R.string.flybits_con_fragment_survey_message_success)
                                    .positiveText(android.R.string.ok)
                                    .show();

                            // successful submission means answers are no longer necessary.
                            surveyViewModel.getAnswers().clear();

                            // go back to the main feed page
                            getFragmentManager().popBackStack();
                        }
                    }
                });
            }

            @Override
            public void onException(final FlybitsException exception)
            {
                uiHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        Activity activity = getActivity();
                        //Activity could be null by the time the callback gets called if user exits it
                        if (activity != null){
                            progressDialog.dismiss();
                            new MaterialDialog.Builder(activity).title(R.string.flybits_con_sorry)
                                    .content(getString(R.string.flybits_con_fragment_survey_message_failed) + exception.getLocalizedMessage())
                                    .positiveText(android.R.string.ok)
                                    .show();
                        }
                    }
                });
            }
        };

        surveyViewModel.setBasicResultCallback(resultCallback);

        // define the loading screen
        progressDialog = new MaterialDialog.Builder(getContext())
                .title(R.string.flybits_con_dialog_loading_title)
                .content(R.string.flybits_con_dialog_loading_message)
                .progress(true, 0)
                .build();
    }

    @Override
    public void onResume()
    {
        super.onResume();

        if (surveyViewModel.isSubmitting())
        {
            progressDialog.show();
        }
    }

    private void submitAnswers()
    {
        ArrayList<Answer> submittedAnswers = new ArrayList<>();

        LongSparseArray<Object> answers = surveyViewModel.getAnswers();

        for (int i = 0; i < survey.questions.getList().size(); i++)
        {
            Question question = survey.questions.getList().get(i);
            Object answer = answers.get(question.questionNumber);

            if (answer == null)
            {
                continue;
            }

            QuestionType questionType = QuestionType.fromString(question.type);
            QuestionUIType questionUIType = QuestionUIType.UNKNOWN;

            try
            {
                questionUIType = QuestionUIType.fromString(question.getMetadata().get(QuestionUIType.ARG_JSON_KEY_UI_TYPE).toString());
            }
            catch (JSONException e)
            {
                Logger.exception(SurveyFragment.class.getSimpleName(), e);
            }

            // prepare answers to be submitted.
            Answer preparedAnswer = new Answer(question.questionNumber);

            switch (questionType)
            {
                case STRING:
                case NUMBER:
                    if (answer instanceof Calendar)
                    {
                        Calendar calendar = (Calendar) answer;

                        if (questionUIType == QuestionUIType.DATE)
                        {
                            preparedAnswer.addAnswer(Utils.getDateString(calendar.getTime()));
                        }
                        else if (questionUIType == QuestionUIType.TIME)
                        {
                            preparedAnswer.addAnswer(Utils.getTimeString(calendar.getTime()));
                        }
                    }
                    else if (questionUIType == QuestionUIType.LOCATION)
                    {
                        // location is stored in JSON. Parse out only the name and submit as a string
                        try
                        {
                            JSONObject jsonObject = new JSONObject(answer.toString());
                            String locationName = jsonObject.getString("name");
                            preparedAnswer.addAnswer(locationName);
                        }
                        catch (JSONException e)
                        {
                            Logger.exception(SurveyFragment.class.getSimpleName(), e);
                        }
                    }
                    else
                    {
                        preparedAnswer.addAnswer(answer.toString());
                    }
                    break;
                case SINGLE:
                    preparedAnswer.addAnswer(question.choices.get((Integer) answer).value.getValue());
                    break;
                case MULTIPLE:
                    List<Integer> list = (List<Integer>) answer;
                    for (Integer selection : list)
                    {
                        preparedAnswer.addAnswer(question.choices.get(selection).value.getValue());
                    }
                    break;
            }

            submittedAnswers.add(preparedAnswer);
        }

        // Submit Survey
        SurveyAnswer surveyAnswer = new SurveyAnswer(survey.resultId, submittedAnswers);
        progressDialog.show();
        surveyViewModel.submitSurveyAnswers(surveyAnswer, survey.getContent());
    }

    private void changeState()
    {
        int position = viewPager.getCurrentItem();

        btnBack.setVisibility(View.VISIBLE);
        btnNext.setText(R.string.flybits_con_fragment_survey_next);

        if (position == 0)
        {
            btnBack.setVisibility(View.GONE);
        }

        if (position == viewPager.getAdapter().getCount() - 1)
        {
            btnNext.setText(R.string.flybits_con_fragment_survey_finish);
        }
    }

    private class SurveyPagerAdapter extends FragmentPagerAdapter
    {
        SurveyPagerAdapter(FragmentManager childFragmentManager)
        {
            super(childFragmentManager);
        }

        @Override
        public Fragment getItem(int position)
        {
            Question question = survey.questions.getList().get(position);
            return QuestionFragment.newInstance(question, position + 1);
        }

        @Override
        public int getCount()
        {
            return survey != null? survey.questions.getList().size() : 0;
        }
    }
}
