package com.flybits.concierge;

import android.app.Activity;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.media.session.PlaybackState;
import android.os.Build;
import android.os.Bundle;
import android.os.RemoteException;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.design.widget.Snackbar;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentActivity;
import android.support.v4.app.FragmentManager;
import android.support.v4.media.MediaBrowserCompat;
import android.support.v4.media.session.MediaControllerCompat;
import android.support.v4.media.session.PlaybackStateCompat;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.afollestad.materialdialogs.MaterialDialog;
import com.flybits.android.kernel.models.Content;
import com.flybits.android.kernel.models.PagedArray;
import com.flybits.android.kernel.utilities.ContentParameters;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.logging.Logger;
import com.flybits.concierge.activities.SettingsActivity;
import com.flybits.concierge.exception.ConciergeException;
import com.flybits.concierge.fragments.FeedFragment;
import com.flybits.concierge.fragments.ImageViewerPagerFragment;
import com.flybits.concierge.fragments.OnboardingFragment;
import com.flybits.concierge.fragments.SurveyFragment;
import com.flybits.concierge.fragments.TermsAndServicesFragment;
import com.flybits.concierge.models.Audio;
import com.flybits.concierge.models.Image;
import com.flybits.concierge.models.Onboarding;
import com.flybits.concierge.models.Survey;
import com.flybits.concierge.services.AudioService;
import com.pierfrancescosoffritti.androidyoutubeplayer.utils.Utils;

import java.util.ArrayList;

public class ConciergeFragment extends Fragment implements AuthenticationStatusListener, OptOutListener {
    public static final String INSTANCE_CURRENT_FRAGMENT = "instance_current_fragment";
    public static final int MESSAGE_TNC = 1;
    public static final int MESSAGE_ONBOARDING = 2;

    private LinearLayout lytLoader;
    private TextView txtLoaderText;
    private LinearLayout errorViewContainer;

    private Context currentContext;

    private Fragment currentFragment;

    private IConciergeFragmentCallbacks currentCallback;

    //Audio Handling
    private MediaBrowserCompat mediaBrowserCompat;
    private MediaControllerCompat mediaControllerCompat;
    private MediaControllerCompat.Callback pendingCallback;
    private Audio lastAudio = null;

    private boolean initializing = false;

    private FlybitsConcierge flybitsConcierge;

    private MediaControllerCompat.Callback controllerCallback = new MediaControllerCompat.Callback()
    {
        @Override
        public void onPlaybackStateChanged(PlaybackStateCompat state)
        {
            Activity activity = getActivity();

            if (activity != null){
                Intent intent = new Intent(ConciergeConstants.BROADCAST_AUDIO_STATE);
                intent.putExtra(ConciergeConstants.INTENT_EXTRA_AUDIO_STATE, state);
                activity.sendBroadcast(intent);
            }
        }
    };

    private MediaBrowserCompat.ConnectionCallback mediaBrowserCompatConnectionCallback = new MediaBrowserCompat.ConnectionCallback()
    {

        @Override
        public void onConnected()
        {
            super.onConnected();

            final Context context = getContext();
            if (context != null){
                try
                {
                    mediaControllerCompat = new MediaControllerCompat(context, mediaBrowserCompat.getSessionToken());
                    mediaControllerCompat.registerCallback(controllerCallback);
                    if (pendingCallback != null){
                        mediaControllerCompat.registerCallback(pendingCallback);
                        pendingCallback = null;
                    }
                    //MediaControllerCompat.setMediaController(getActivity(), mediaControllerCompat);

                    Bundle extras = mediaControllerCompat.getPlaybackState().getExtras();
                    //We are still playing something?
                    if (extras != null && lastAudio == null)
                    {
                        lastAudio = new Audio();
                        lastAudio.sourceURL = extras.getString(AudioService.ARG_URL);
                    }else if (lastAudio != null){
                        playFromMediaId(lastAudio);
                    }

                }
                catch (RemoteException e)
                {
                    Logger.exception(null, e);
                }
            }
        }
    };

    public void startSettingsActivity(){
        Activity activity = getActivity();
        if (activity != null){
            Intent startSettingsIntent = new Intent(activity,SettingsActivity.class);
            startActivityForResult(startSettingsIntent, SettingsActivity.REQUEST_CODE);
        }
    }

    //Messages back from child fragments
    public void childMessage(int code, Object data)
    {
        switch (code)
        {
            case MESSAGE_TNC:
                boolean result = (Boolean) data;
                if (result)
                {
                    initializeState(false);
                }
                else if (currentCallback != null)
                {
                    currentCallback.onTNCDecline();
                }
                break;
            case MESSAGE_ONBOARDING:
                initializeState(false);
                break;
        }
    }

    private void openTNCs(){
        hideLoader();
        showTNCs();
    }

    private void findOnboarding(){
        hideLoader();
        findContentAndTakeover(ConciergeConstants.ONBOARDING_CONTENT_TYPE);
    }

    private void findSurvey(){
        hideLoader();
        findContentAndTakeover(ConciergeConstants.SURVEY_CONTENT_TYPE);
    }

    private void openFeed(){
        hideLoader();
        showFeed();
    }

    public static ConciergeFragment newInstance()
    {
        ConciergeFragment fragment = new ConciergeFragment();
        return fragment;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);

        if (savedInstanceState != null)
        {
            currentFragment = getChildFragmentManager().getFragment(savedInstanceState, INSTANCE_CURRENT_FRAGMENT);
        }
    }

    @Override
    public void onSaveInstanceState(@NonNull Bundle outState)
    {
        super.onSaveInstanceState(outState);

        if (currentFragment != null)
        {
            getChildFragmentManager().putFragment(outState, INSTANCE_CURRENT_FRAGMENT, currentFragment);
        }
        outState.putBoolean(ConciergeConstants.STATE_ERROR_PRESENT, errorViewContainer.getVisibility() == View.VISIBLE);

    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, Bundle savedInstanceState)
    {
        return inflater.inflate(R.layout.flybits_con_fragment_concierge, container, false);
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState)
    {
        super.onViewCreated(view, savedInstanceState);

        lytLoader = view.findViewById(R.id.concierge_fragment_lytLoader);
        txtLoaderText = view.findViewById(R.id.concierge_fragment_txtLoadingText);
        errorViewContainer = view.findViewById(R.id.concierge_fragment_error_holder);

        view.findViewById(R.id.concierge_fragment_retry_button).setOnClickListener(new View.OnClickListener()
        {
            @Override
            public void onClick(View view)
            {
                initializeState(false);
            }
        });

        flybitsConcierge = FlybitsConcierge.with(currentContext);
        flybitsConcierge.registerAuthenticationStateListener(this);

        flybitsConcierge.registerOptOutListener(this);

        if (savedInstanceState == null){
            initializeState(false);
        }else{
            errorViewContainer.setVisibility(savedInstanceState.getBoolean(ConciergeConstants.STATE_ERROR_PRESENT) ? View.VISIBLE: View.GONE);
        }

    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
        if (flybitsConcierge != null){
            flybitsConcierge.unregisterAuthenticationStateListener(this);
            flybitsConcierge.unregisterOptOutListener(this);
        }
    }

    @Override
    public void onAttach(Context context)
    {
        super.onAttach(context);
        currentContext = context;
    }

    @Override
    public void onDetach()
    {
        super.onDetach();
        if (mediaBrowserCompat != null && mediaBrowserCompat.isConnected())
        {
            mediaBrowserCompat.disconnect();
        }

        disconnectFromAudioService();
    }

    /**
     * Attempts to resolve all requirements to see the main feed.
     *
     * @param skipOnboarding Skips onboarding (due to it not existing)
     */
    private void initializeState(boolean skipOnboarding)
    {
        if (initializing) return;
        initializing = true;

        //Check if authorized first
        if (flybitsConcierge.isAuthenticated())
        {
            flybitsConcierge.unregisterAuthenticationStateListener(this);
            errorViewContainer.setVisibility(View.GONE);

            //Are TNCs Accepted?
            if (FlybitsConcierge.with(currentContext).getConfiguration().getTNCUrl() != null && !InternalPreferences.isTNCAccepted(currentContext))
            {
                // if TNCs haven't been shown yet, and the link is initialized in the config file
                openTNCs();
            }
            else if (!skipOnboarding && !InternalPreferences.isOnBoardingDone(currentContext))
            {
                // if you should show onboarding and onboarding hasn't been shown yet
                findOnboarding();
                InternalPreferences.saveOnBoardingDone(currentContext, true);
            }
            else if (!InternalPreferences.isSurveyDone(currentContext))
            {
                // if you should show a survey
                findSurvey();
                InternalPreferences.saveSurveyDone(currentContext, true);
            }
            else
            {
                // Otherwise, show FeedFragment
                openFeed();
            }
        }else{
            //Retry authentication since we are not currently authenticated, wait for callback(AuthenticationStatusListener)'s methods to be invoked
            boolean retrySuccess = flybitsConcierge.retryAuthentication();
            if (retrySuccess){
                showLoader("");
            }
            errorViewContainer.setVisibility(View.VISIBLE);
        }

        initializing = false;
    }

    /**
     * Replaces the current content to the TNC page
     */
    private void showTNCs()
    {
        currentFragment = TermsAndServicesFragment.newInstance(false);
        FragmentManager manager = getChildFragmentManager();
        manager.beginTransaction()
                .replace(R.id.concierge_fragment_lytContent, currentFragment)
                .commit();

    }

    /**
     * Replaces the current content to some Flybits Content Fragment (Onboarding, Surveys)
     */
    private void findContentAndTakeover(final String contentTemplateType)
    {
        ContentParameters params = new ContentParameters.Builder().setTemplateType(contentTemplateType)
                .build();

        Content.get(currentContext, params, new PagedResultCallback<Content>()
        {
            @Override
            public void onSuccess(ArrayList<Content> items)
            {
                if (items.size() != 0)
                {
                    showContent(items.get(0));
                }
                else
                {
                    initializeState(true);
                }
            }

            @Override
            public void onException(FlybitsException exception)
            {

            }

            @Override
            public void onLoadedAllItems()
            {

            }
        });
    }

    /**
     * Replaces the current content with a content.
     */
    private void showContent(Content contentToShow)
    {
        hideLoader();
        FragmentManager manager = getChildFragmentManager();
        switch (contentToShow.getType())
        {
            case ConciergeConstants.ONBOARDING_CONTENT_TYPE:
            {
                Onboarding onboarding = (Onboarding) ContentHelper.contentToBaseTemplate(contentToShow, getContext());
                if (onboarding == null) {
                    initializeState(true);
                    return;
                }

                OnboardingFragment onboardingFragment = OnboardingFragment.newInstance(onboarding);

                onboardingFragment.setOnboardingListener(new OnboardingListener()
                {
                    @Override
                    public void onSkip()
                    {
                        initializeState(false);
                    }

                    @Override
                    public void onFinished()
                    {
                        initializeState(false);
                    }
                });

                currentFragment = onboardingFragment;
                manager.beginTransaction()
                        .replace(R.id.concierge_fragment_lytContent, currentFragment, currentFragment.getClass().getSimpleName())
                        .commit();
                break;
            }
            case ConciergeConstants.SURVEY_CONTENT_TYPE:
            {
                Survey survey = (Survey) ContentHelper.contentToBaseTemplate(contentToShow, getContext());
                if (survey == null || survey.questions.getList().isEmpty()) {
                    new MaterialDialog.Builder(getActivity())
                            .title(R.string.flybits_con_error)
                            .content(R.string.flybits_con_dialog_survey_error)
                            .positiveText(android.R.string.ok)
                            .show();
                    InternalPreferences.saveSurveyDone(getContext(), true);
                    initializeState(false);
                    return;
                }

                SurveyFragment surveyFragment = SurveyFragment.newInstance(survey);

                surveyFragment.setSurveyListener(new SurveyListener()
                {
                    @Override
                    public void onFinished()
                    {
                        initializeState(false);
                        InternalPreferences.saveSurveyDone(currentContext, true);
                    }
                });

                currentFragment = surveyFragment;
                manager.beginTransaction()
                        .replace(R.id.concierge_fragment_lytContent, currentFragment, currentFragment.getClass().getSimpleName())
                        .commit();
                break;
            }
            default:
                initializeState(false);
                break;
        }
    }

    /**
     * Replaces the current content with the main feed. Common state
     */
    private void showFeed()
    {
        currentFragment = FeedFragment.newInstance();
        FragmentManager manager = getChildFragmentManager();
        manager.beginTransaction()
                .replace(R.id.concierge_fragment_lytContent, currentFragment, currentFragment.getClass().getSimpleName())
                .commit();
    }

    public void openImageViewer(PagedArray<Image> images, int offset)
    {
        ImageViewerPagerFragment imageFragment = ImageViewerPagerFragment.newInstance(images, offset);
        openFragment(imageFragment);
    }

    private void connectToAudioService()
    {
        Context context = getContext();
        if (context == null)
        {
            return;
        }
        if (mediaBrowserCompat == null)
        {
            mediaBrowserCompat = new MediaBrowserCompat(context.getApplicationContext()
                    , new ComponentName(context.getApplicationContext(), AudioService.class), mediaBrowserCompatConnectionCallback, null);
        }
        mediaBrowserCompat.connect();
    }

    private void disconnectFromAudioService()
    {
        if (mediaBrowserCompat != null && mediaBrowserCompat.isConnected())
        {
            mediaControllerCompat.unregisterCallback(controllerCallback);
            mediaBrowserCompat.disconnect();
        }
    }

    public void startAudio(Audio audio)
    {
        lastAudio = audio;
        boolean isOnline = Utils.isOnline(currentContext);
        View view = getView();
        if (isOnline && (mediaBrowserCompat == null || !mediaBrowserCompat.isConnected())){
            connectToAudioService();
        }else if (isOnline){
            playFromMediaId(audio);
        }else if (view != null){
            Snackbar.make(view,R.string.flybits_con_audio_error_message,Snackbar.LENGTH_LONG).show();
        }
    }

    public void registerMediaControllerCallback(MediaControllerCompat.Callback mediaControllerCallback){
        if (mediaControllerCallback != null && mediaBrowserCompat != null &&  mediaBrowserCompat.isConnected()){
            pendingCallback = null;
            mediaControllerCompat.registerCallback(mediaControllerCallback);
        }
        //register callback once AudioService is connected
        else{
            pendingCallback = mediaControllerCallback;
        }

    }

    public void resumeAudio()
    {
        View view = getView();
        Context context = getContext();
        if (mediaBrowserCompat != null && mediaBrowserCompat.isConnected()
                && context!= null && Utils.isOnline(context))
        {
            mediaControllerCompat.getTransportControls().play();
        }else if (view != null){
            Snackbar.make(view,R.string.flybits_con_audio_error_message,Snackbar.LENGTH_LONG).show();
        }
    }

    public void pauseAudio(){
        if (mediaControllerCompat != null && mediaControllerCompat.getTransportControls() != null)
        {
            mediaControllerCompat.getTransportControls().pause();
        }
    }

    public void stopAudio()
    {
        lastAudio = null;

        if (mediaControllerCompat != null && mediaControllerCompat.getTransportControls() != null)
        {
            mediaControllerCompat.getTransportControls().stop();
        }
    }

    public int getAudioState()
    {
        if (mediaBrowserCompat == null || !mediaBrowserCompat.isConnected())
        {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                return PlaybackState.STATE_NONE;
            }else{
                return 0;
            }
        }
        return mediaControllerCompat.getPlaybackState().getState();
    }

    private void playFromMediaId(Audio audio){
        Bundle args = new Bundle();

        //For some reason, parcelables ain't workin, do it manually
        if (audio.title != null)
        {
            args.putString(AudioService.ARG_TITLE, audio.title.getValue());
        }
        if (audio.description != null)
        {
            args.putString(AudioService.ARG_DESC, audio.description.getValue());
        }

        if (mediaBrowserCompat != null && mediaBrowserCompat.isConnected())
        {
            mediaControllerCompat.getTransportControls().playFromMediaId(audio.sourceURL, args);
        }
    }

    public Audio getCurrentAudio()
    {
        return lastAudio;
    }

    public void openFragment(Fragment fragment)
    {
        FragmentActivity fragmentActivity = getActivity();
        if (fragmentActivity != null){
            fragmentActivity.getSupportFragmentManager()
                    .beginTransaction()
                    .addToBackStack(null)
                    .replace(R.id.concierge_fragment_lytContent, fragment)
                    .commit();
        }

    }

    /**
     * Removes the current content and shows the loader
     *
     * @param title Message to display
     */
    private void showLoader(String title)
    {
        if (currentFragment != null)
        {
            getChildFragmentManager().beginTransaction()
                    .remove(currentFragment)
                    .commit();
        }
        txtLoaderText.setText(title);
        lytLoader.setVisibility(View.VISIBLE);
    }

    /**
     * Hides the loader
     */
    private void hideLoader()
    {
        txtLoaderText.setText("");
        lytLoader.setVisibility(View.GONE);
    }

    public void setCallback(IConciergeFragmentCallbacks callback)
    {
        currentCallback = callback;
    }

    @Override
    public void onAuthenticated() {
        initializeState(false);
    }

    @Override
    public void onAuthenticationStarted() {

    }

    @Override
    public void onAuthenticationError(ConciergeException e) {
        errorViewContainer.setVisibility(View.VISIBLE);
        hideLoader();
    }

    @Override
    public void onUserOptedOut() {
        if (getActivity() != null){
            initializeState(true);
        }
    }

    public interface IConciergeFragmentCallbacks
    {
        void onTNCDecline();
    }


}
