package com.flybits.concierge.services;

import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.text.TextUtils;

import com.flybits.android.push.FlybitsNotificationManager;
import com.flybits.android.push.PushManager;
import com.flybits.android.push.models.Push;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.logging.Logger;
import com.flybits.concierge.ConciergeConstants;
import com.flybits.concierge.InternalPreferences;
import com.flybits.concierge.R;
import com.flybits.concierge.activities.ConciergeActivity;
import com.google.firebase.messaging.FirebaseMessagingService;
import com.google.firebase.messaging.RemoteMessage;

import org.json.JSONObject;

public class FlybitsConMessagingService extends FirebaseMessagingService
{
    private static final String JSON_KEY_URL = "url";
    private static final String JSON_KEY_VALUE = "value";
    private static final String JSON_KEY_CONTENT = "contentId";
    private static final int NOTIFICATION_ID = 753;

    @Override
    public void onMessageReceived(RemoteMessage remoteMessage)
    {
        super.onMessageReceived(remoteMessage);

        PushManager.parsePushNotification(getApplicationContext(), remoteMessage.getData(), new ObjectResultCallback<Push>()
        {
            @Override
            public void onSuccess(Push item)
            {
                Context context = getApplicationContext();
                Intent intent = new Intent();

                switch (item.getAction())
                {
                    case ADDED:
                        intent.setAction(ConciergeConstants.BROADCAST_CONTENT_ADD);
                        break;
                    case REMOVED:
                        intent.setAction(ConciergeConstants.BROADCAST_CONTENT_REMOVE);
                        break;
                    case UPDATED:
                        intent.setAction(ConciergeConstants.BROADCAST_CONTENT_UPDATE);
                        break;
                    case RULE_UPDATED:
                        intent.setAction(ConciergeConstants.BROADCAST_RULE_UPDATE);
                        break;
                    case STATUS_CHANGED:
                        intent.setAction(ConciergeConstants.BROADCAST_RULE_STATE);
                        break;
                    case CUSTOM:
                        buildNotification(item);
                        break;
                    default: // unknown + other unimplemented values
                        break;
                }

                if (!TextUtils.isEmpty(intent.getAction()))
                {
                    context.sendBroadcast(intent);
                }
            }

            @Override
            public void onException(FlybitsException exception)
            {
                Logger.exception(FlybitsConMessagingService.class.getSimpleName(), exception);
            }
        });
    }

    public int pushNotificationIconRes()
    {
        return R.mipmap.ic_flybits_notification;
    }

    private void buildNotification(Push item)
    {
        FlybitsNotificationManager builder = new FlybitsNotificationManager.Simplifier(
                getApplicationContext(),
                item.getId(),
                item.getTitle(),
                item.getMessage(),
                pushNotificationIconRes())
                .build();

        Intent intent = new Intent(getApplicationContext(), ConciergeActivity.class);

        try
        {
            JSONObject jsonObject = new JSONObject(item.getCustomFieldsAsString());
            String url = jsonObject.isNull(JSON_KEY_URL)
                    ? null
                    : jsonObject.getString(JSON_KEY_URL);
            String contentId = jsonObject.isNull(JSON_KEY_CONTENT)
                    ? null
                    : jsonObject.getJSONObject(JSON_KEY_CONTENT).getString(JSON_KEY_VALUE);

            if (url != null)
            {
                intent.putExtra(ConciergeConstants.PUSH_CONTENT_URL, url);
            }
            else if (contentId != null)
            {
                intent.putExtra(ConciergeConstants.PUSH_CONTENT_ID, contentId);
            }
        }
        catch (Exception e)
        {
            Logger.exception(FlybitsConMessagingService.class.getSimpleName(), e);
        }

        PendingIntent pendingIntent = PendingIntent.getActivity(getApplicationContext(), NOTIFICATION_ID, intent, PendingIntent.FLAG_UPDATE_CURRENT);

        builder.get().setContentIntent(pendingIntent);

        builder.show();
    }

    @Override
    public void onNewToken(String recentToken)
    {
        super.onNewToken(recentToken);

        InternalPreferences.savePushToken(getApplicationContext(), recentToken);
    }
}
