package com.flybits.concierge;

import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.os.Build;
import android.support.annotation.IdRes;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.flybits.android.kernel.KernelScope;
import com.flybits.android.push.PushManager;
import com.flybits.android.push.PushScope;
import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.api.FlybitsManager;
import com.flybits.commons.library.api.idps.IDP;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ConnectionResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.logging.Logger;
import com.flybits.concierge.activities.ConciergeActivity;
import com.flybits.concierge.activities.ConciergePopupActivity;
import com.flybits.concierge.enums.ShowMode;
import com.flybits.concierge.exception.ConciergeException;
import com.flybits.concierge.exception.ConciergeOptedOutException;
import com.flybits.concierge.exception.ConciergeUninitializedException;
import com.flybits.concierge.services.PreloadingWorker;

import java.lang.ref.WeakReference;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import androidx.work.Constraints;
import androidx.work.NetworkType;
import androidx.work.OneTimeWorkRequest;
import androidx.work.WorkManager;
import androidx.work.WorkRequest;

import static com.flybits.concierge.services.AudioService.AUDIO_CHANNEL;

/**
 * A {@code FlybitsConcierge} is a singleton responsible for dealing with Concierge
 * SDK configuration, authentication, and display.
 *
 */
public class FlybitsConcierge
{
    private static FlybitsConcierge INSTANCE;

    private WeakReference<Context> lastProvidedContext;
    private ConciergeConfiguration currentConfig;
    private IDP idp;    //IDP set during most recent authenticate call
    private FlybitsManager flybitsManager;
    private final Set<OptOutListener> optOutListeners;
    private final Set<AuthenticationStatusListener> authenticationStatusListeners;

    private boolean authenticating = false;     //If authentication is currently happening
    private boolean authenticationRequested = false;    //Tracks whether authentication has ever been requested so we know whether to register network receiver
    private boolean networkReceiverRegistered = false;
    private boolean initialized = false;

    private BroadcastReceiver networkReceiver = new BroadcastReceiver()
    {
        @Override
        public void onReceive(Context context, Intent intent)
        {
            try
            {
                ConnectivityManager manager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);

                //Retry authentication if its been previously requested and we're not authenticated yet and phone came online
                if (manager.getActiveNetworkInfo() != null && manager.getActiveNetworkInfo().isConnected()
                        && isAutoRetryAuthenticationOnConnectedEnabled() && !isAuthenticated() && authenticationRequested)
                {
                    retryAuthentication();
                }
            }catch(Exception e)
            {
                Logger.exception("networkReceiver.onReceive",e);
            }

        }
    };

    private ConnectionResultCallback conciergeConnectionResultCallback = new ConnectionResultCallback()
    {
        @Override
        public void onConnected()
        {
            authenticating = false;

            broadcastAuthenticationState(AuthenticationStatusListener.AUTHENTICATED);

            unregisterNetworkReceiver();

            schedulerWorkers();

            enablePushMessaging(new BasicResultCallback()
            {
                @Override
                public void onSuccess()
                {
                    // Success!
                }

                @Override
                public void onException(FlybitsException exception)
                {
                    Logger.exception(FlybitsConcierge.class.getSimpleName(), exception);
                }
            });

        }

        @Override
        public void notConnected()
        {
            authenticating = false;
        }

        @Override
        public void onException(FlybitsException exception)
        {
            authenticating = false;

            //Hacky for now but only solution without making specific exceptions
            Context context = getContext();
            if (context != null && SharedElements.getSavedJWTToken(context).isEmpty())
            {
                broadcastAuthenticationError(new ConciergeException(exception.getMessage()));
            }else if (context != null && exception.getMessage().contains("Connecting") && isAuthenticated())
            {
                broadcastAuthenticationState(AuthenticationStatusListener.AUTHENTICATED);
            }

            registerNetworkReceiver();
        }
    };

    private FlybitsConcierge(WeakReference<Context> context)
    {
        optOutListeners = Collections.synchronizedSet(new HashSet<OptOutListener>());
        authenticationStatusListeners = Collections.synchronizedSet(new HashSet<AuthenticationStatusListener>());
        lastProvidedContext = context;
    }

    /**
     * Used to get reference to {@code FlybitsConcierge} singleton instance.
     *
     * @param context The context of your application.
     * @return {@code FlybitsConcierge} singleton instance.
     */
    public static FlybitsConcierge with(Context context)
    {
        if (context == null)
        {
            throw new ConciergeUninitializedException();
        }

        synchronized (FlybitsConcierge.class)
        {
            if (INSTANCE == null)
            {
                INSTANCE = new FlybitsConcierge(new WeakReference<>(context.getApplicationContext()));
                INSTANCE.createNotificationChannel();
                return INSTANCE;
            }
        }
        INSTANCE.lastProvidedContext = new WeakReference<>(context.getApplicationContext());
        return INSTANCE;
    }

    private void createNotificationChannel()
    {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O)
        {
            CharSequence name = "Concierge Channel";
            String description = "Specific for playing audio content in the concierge.";
            NotificationChannel channel = new NotificationChannel(AUDIO_CHANNEL, name, NotificationManager.IMPORTANCE_HIGH);
            channel.setDescription(description);
            NotificationManager notificationManager = getContext().getSystemService(NotificationManager.class);
            notificationManager.createNotificationChannel(channel);
        }
    }

    private Context getContext() throws ConciergeException
    {
        if (lastProvidedContext != null && lastProvidedContext.get() != null)
        {
            return lastProvidedContext.get();
        }
        throw new ConciergeException("Context was null");
    }

    private FlybitsManager createFlybitsManager(){
        FlybitsManager.Builder builder = new FlybitsManager.Builder(getContext())
                .setAccount(null)
                .setProjectId(getConfiguration().getProjectID())
                .addScope(PushScope.SCOPE)
                .addScope(KernelScope.SCOPE);

        if (BuildConfig.DEBUG)
        {
            builder.setDebug();
        }

        return builder.build();
    }

    /**
     * Initializes the {@code FlybitsConcierge} instance, allowing for further method invocations.
     *
     * @param cfgResource Resource file containing required and optional configuration fields.
     * @return False if already initialized, and true otherwise.
     */
    public boolean initialize(@IdRes int cfgResource)
    {
        if (initialized){
            return false;
        }

        initialize(ConciergeConfiguration.createFromXML(getContext(), cfgResource));

        initialized = true;

        flybitsManager = createFlybitsManager();

        return true;
    }

    void initialize(ConciergeConfiguration cfg)
    {
        currentConfig = cfg;
    }

    /**
     *
     * @return Whether the {@code FlybitsConcierge} is initialized.
     */
    public boolean isInitialized(){
        return initialized;
    }

    /**
     * Register {@code AuthenticationStateListener} that will be notified about any changes to
     * the authentication status.
     *
     * This method is purposely seperate from the authenticate method because it is needed in order
     * to get updates in the ConciergeFragment even if authenticate isn't called in there due to
     * AutoAuthenticate being set to false.
     *
     * @param authenticationStatusListener {@code AuthenticationStateListener} that will be notified about changes to the authentication status.
     */
    public void registerAuthenticationStateListener(@NonNull AuthenticationStatusListener authenticationStatusListener)
    {
        synchronized (authenticationStatusListeners) {
            authenticationStatusListeners.add(authenticationStatusListener);
        }
    }

    /**
     * Unregister {@code AuthenticationStateListener} so that it is no longer notified about changes
     * to the authentication status.
     *
     * @param authenticationStatusListener {@code AuthenticationStateListener} being unregistered.
     * @return true if unregistered successfully, and false otherwise.
     */
    public boolean unregisterAuthenticationStateListener(@NonNull AuthenticationStatusListener authenticationStatusListener){
        synchronized (authenticationStatusListeners){
            return authenticationStatusListeners.remove(authenticationStatusListener);
        }
    }

    private void broadcastAuthenticationState(String state)
    {

        synchronized (authenticationStatusListeners)
        {
            //Copy over to array to avoid ConcurrentModificationException
            Object[] copy = authenticationStatusListeners.toArray();

            for (Object o : copy)
            {
                AuthenticationStatusListener authenticationStatusListener = (AuthenticationStatusListener)o;
                switch(state){
                    case AuthenticationStatusListener.AUTHENTICATED:
                        authenticationStatusListener.onAuthenticated();
                        break;
                    case AuthenticationStatusListener.AUTHENTICATION_STARTED:
                        authenticationStatusListener.onAuthenticationStarted();
                        break;
                    default:
                }
            }
        }
    }

    private void broadcastAuthenticationError(ConciergeException err)
    {
        synchronized (authenticationStatusListeners)
        {
            //Copy over to array to avoid ConcurrentModificationException
            Object[] copy = authenticationStatusListeners.toArray();

            for (Object o : copy)
            {
                AuthenticationStatusListener authenticationStatusListener = (AuthenticationStatusListener)o;
                authenticationStatusListener.onAuthenticationError(err);
            }
        }
    }

    synchronized boolean retryAuthentication()
    {
        if (!isInitialized())
        {
            throw new ConciergeUninitializedException();
        }
        /*Do not allow for retrying authentication after user logged out or if authentication was never
            requested. That way if the view is shown and the sdk is logged out it won't re-authenticate immediately.*/
        else if(idp == null || authenticating || isAuthenticated() || !authenticationRequested)
        {
            return false;
        }

        authenticate(idp);
        return true;
    }

    /**
     * Attempt to authenticate the user. Feedback will be provided after the request is made
     * through the registered {@code AuthenticationStateListener}.
     *
     * This method will opt the user back in right away, and is the only way to do so.
     *
     * If the authentication request
     * fails it will be retried automatically upon change in connection status or "retry" button
     * click unless the autoRetryAuthenticationOnConnected flag is set to false.
     *
     * @param idp {@code IDP} for the user being authenticated.
     * @return True if the authentication request was made, and false otherwise.
     */
    public synchronized boolean authenticate(@NonNull IDP idp)
    {

        Context context = getContext();

        if (!isInitialized())
        {
            throw new ConciergeUninitializedException();
        }

        //Automatically opt in if authenticate is called.
        InternalPreferences.setOptOutState(context,false);

        if (isAuthenticated() || authenticating)
        {
            return false;
        }else{
            authenticating = true;
        }

        //Might be null if user opted out
        if (flybitsManager == null){
            flybitsManager = createFlybitsManager();
        }

        this.idp = idp;

        broadcastAuthenticationState(AuthenticationStatusListener.AUTHENTICATION_STARTED);

        authenticationRequested = true;

        flybitsManager.connect(idp,conciergeConnectionResultCallback);

        return true;
    }

    /**
     * Ends the authenticated user session. Cannot be performed offline.
     *
     * @param logOutCallback {@code LogOutCallback} that will be notified about the status of the logout request.
     */
    public void logOut(final LogOutCallback logOutCallback){
        if (!isInitialized()){
            throw new ConciergeUninitializedException();
        }else if (!isAuthenticated() || authenticating || flybitsManager == null){
            logOutCallback.onError(new ConciergeException("Authentication required before logging out"));
        }else{
            flybitsManager.disconnect(new BasicResultCallback() {
                @Override
                public void onSuccess() {
                    authenticationRequested = false;
                    unregisterNetworkReceiver();
                    if (logOutCallback != null){
                        logOutCallback.onSuccess();
                    }
                }

                @Override
                public void onException(FlybitsException exception) {
                    if (logOutCallback != null){
                        logOutCallback.onError(new ConciergeException(exception));
                    }
                }
            });
        }
    }

    /**
     * Unauthenticates the FlybitsConcierge without calling logOut(). This should only
     * be used if the JWT token expires and there's no way to log out but the application
     * still needs to be aware that the session expires.
     *
     * @param e The cause of the unauthenticate method call.
     * @return Whether unauthenticated successfully.
     */
    public boolean unauthenticateWithoutLogout(FlybitsException e){
        if (!isInitialized()){
            throw new ConciergeUninitializedException();
        }else if (authenticating){
            return false;
        }else{
            SharedElements.setJWTToken(getContext(),""); //So that isAuthenticated() returns false
            authenticationRequested = false;
            unregisterNetworkReceiver();
            broadcastAuthenticationError(new ConciergeException(e));
            return true;
        }
    }

    /**
     * Set the {@code OptOutListener} that will be notified if the user opts out.
     *
     * @param o {@code OptOutListener} that will be notified when the user opts out.
     */
    public void registerOptOutListener(@NonNull OptOutListener o)
    {
        synchronized (optOutListeners){
            optOutListeners.add(o);
        }
    }

    /**
     * Unregister {@code OptOutListener}.
     */
    public void unregisterOptOutListener(@NonNull OptOutListener o){
        synchronized (optOutListeners){
            optOutListeners.remove(o);
        }
    }

    private void broadcastOptOut() {
        synchronized (optOutListeners) {
            Object[] copy = optOutListeners.toArray();
            for (Object o : copy) {
                ((OptOutListener) o).onUserOptedOut();
            }
        }
    }

    /**
     *
     * @return Whether the user is opted out of the {@code FlybitsConcierge}.
     */
    public boolean isOptedOut()
    {
        if (isInitialized())
        {
            return InternalPreferences.getOptOutState(getContext());
        }else{
            throw new ConciergeUninitializedException();
        }
    }

    /**
     * Opts the user out of the {@code FlybitsConcierge}.
     *
     * This is different from logout! Removes all data associated with user.
     *
     * Call authenticate to opt user back in.
     *
     * @param callback {@code OptOutCallback} that will be notified about whether the opt-out request succeeded.
     */
    public void optOut(@Nullable final OptOutCallback callback)
    {
        if (!isInitialized())
        {
            throw new ConciergeUninitializedException();
        }else if(isOptedOut()){
            throw new ConciergeOptedOutException();
        }else if (!isAuthenticated() || authenticating ||flybitsManager == null){
            callback.onError(new ConciergeException("Authentication required."));
            return;
        }

        final Context context = getContext();
        //Do not allow for opting out if authentication online hasn't happened because there's no way to get a reference to flybits manager to destroy! (as far as I know)
        flybitsManager.destroy(new BasicResultCallback()
        {
            @Override
            public void onSuccess()
            {
                InternalPreferences.saveTNCAccepted(context,false);
                InternalPreferences.setOptOutState(context,true);
                authenticating = false;
                unregisterNetworkReceiver();
                authenticationRequested = false;

                broadcastOptOut();

                if (callback != null)
                {
                    callback.onSuccess();
                }
                flybitsManager = null;
            }

            @Override
            public void onException(FlybitsException exception)
            {
                if (callback != null)
                {
                    callback.onError(new ConciergeException(exception));
                }
            }
        });
    }

    private boolean unregisterNetworkReceiver(){
        if (networkReceiverRegistered){
            try{
                getContext().unregisterReceiver(networkReceiver);
                networkReceiverRegistered = false;
                return true;
            }catch(Exception e){
                Logger.exception("FlybitsConcierge.unregisterNetworkReceiver()",e);
                return false;
            }
        }else{
            return false;
        }
    }

    private boolean registerNetworkReceiver(){
        //Register receiver responsible for auto authenticating in the future once connection is established
        if (!networkReceiverRegistered && isAutoRetryAuthenticationOnConnectedEnabled())
        {
            IntentFilter intentFilter = new IntentFilter(android.net.ConnectivityManager.CONNECTIVITY_ACTION);
            try{
                getContext().registerReceiver(networkReceiver,intentFilter);
                networkReceiverRegistered = true;
                return true;
            }catch(Exception e){
                Logger.exception("FlybitsConcierge.registerNetworkReceiver()",e);
                return false;
            }
        }else{
            return false;
        }
    }

    /**
     * Show the Concierge UI elements to the user.
     *
     * Do not call this if the user is opted out! ConciergeOptedOutException will be thrown if you do so.
     * Call authenticate to opt user back in.
     *
     * @param mode Mode of display, either OVERLAY or NEW_ACTIVITY(recommended).
     * @throws ConciergeException
     */
    public void show(ShowMode mode) throws ConciergeException
    {
        Context context = getContext();
        if (!isInitialized())
        {
            throw new ConciergeUninitializedException();
        }
        else if (InternalPreferences.getOptOutState(context))
        {
            throw new ConciergeOptedOutException();
        }

        Intent activityIntent = null;

        switch (mode)
        {
            case NEW_ACTIVITY:
            {
                activityIntent = new Intent(context, ConciergeActivity.class);
                break;
            }
            case OVERLAY:
            {
                activityIntent = new Intent(context, ConciergePopupActivity.class);
                break;
            }
        }

        activityIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        context.startActivity(activityIntent);
    }

    /**
     * If set to false the {@code FlybitsConcierge} will not retry authentication on change in connection if authentication failed previously.
     * It is set to true by default, this is the recommended setting.
     *
     * @param enabled Whether to retry authentication on internet connection state change if authentication failed previously.
     */
    public void setAutoRetryAuthenticationOnConnected(boolean enabled)
    {
        Context context = getContext();
        if (isInitialized())
        {
            InternalPreferences.setAutoRetryAuthOnConnected(context,enabled);
        }else{
            throw new ConciergeUninitializedException();
        }
    }

    /**
     *
     * @return Whether the {@code FlybitsConcierge} will retry authentication on change in connection
     * if authentication failed previously. Will return true by default if flag was never altered.
     */
    public boolean isAutoRetryAuthenticationOnConnectedEnabled()
    {
        Context context = getContext();
        if (isInitialized())
        {
            return InternalPreferences.isAutoRetryAuthOnConnectedEnabled(context);
        }else{
            throw new ConciergeUninitializedException();
        }
    }

    /**
     *
     * @return {@code ConciergeConfiguration} that was set during initialization.
     */
    public ConciergeConfiguration getConfiguration()
    {
        if (!isInitialized())
        {
            throw new ConciergeUninitializedException();
        }

        return currentConfig;
    }

    private void enablePushMessaging(BasicResultCallback callback)
    {
        String token = InternalPreferences.pushToken(getContext());

        if (token == null)
        {
            return;
        }

        PushManager.enablePush(getContext(), token, new HashMap<String, String>(), callback);
    }

    /**
     *
     * @return Whether the {@code FlybitsConcierge} is currently authenticated.
     */
    public boolean isAuthenticated()
    {
        Context context = getContext();
        if (isInitialized())
        {
            return !SharedElements.getSavedJWTToken(context).isEmpty();
        }else{
            throw new ConciergeUninitializedException();
        }
    }

    /**
     *
     * @return Whether the {@code FlybitsConcierge} is currently in the process of attempting to authenticate.
     */
    public boolean isAuthenticating()
    {
        if (isInitialized())
        {
            return authenticating;
        }else{
            throw new ConciergeUninitializedException();
        }
    }

    private void schedulerWorkers()
    {
        WorkManager workManager = WorkManager.getInstance();

        Constraints workConstraints = new Constraints.Builder().setRequiredNetworkType(NetworkType.CONNECTED)
                .build();
        final WorkRequest workRequest = new OneTimeWorkRequest.Builder(PreloadingWorker.class).setConstraints(workConstraints)
                .build();

        //Worker responsible for preloading
        workManager.enqueue(workRequest);
    }

}
