package com.flybits.concierge;

import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.os.Build;
import android.support.annotation.IdRes;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.flybits.android.kernel.KernelScope;
import com.flybits.android.push.PushManager;
import com.flybits.android.push.PushScope;
import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.api.FlybitsManager;
import com.flybits.commons.library.api.idps.IDP;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ConnectionResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.logging.Logger;
import com.flybits.concierge.activities.ConciergeActivity;
import com.flybits.concierge.activities.ConciergePopupActivity;
import com.flybits.concierge.enums.ShowMode;
import com.flybits.concierge.exception.ConciergeException;
import com.flybits.concierge.exception.ConciergeOptedOutException;
import com.flybits.concierge.exception.ConciergeUninitializedException;
import com.flybits.concierge.services.PreloadingWorker;

import java.lang.ref.WeakReference;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import androidx.work.Constraints;
import androidx.work.NetworkType;
import androidx.work.OneTimeWorkRequest;
import androidx.work.WorkManager;
import androidx.work.WorkRequest;

import static com.flybits.concierge.services.AudioService.AUDIO_CHANNEL;

public class FlybitsConcierge
{
    private static FlybitsConcierge INSTANCE;

    private WeakReference<Context> lastProvidedContext;
    private ConciergeConfiguration currentConfig;
    private IDP idp;    //IDP set during most recent authenticate call
    private FlybitsManager flybitsManager;
    private OptOutListener optOutListener;
    private final Set<AuthenticationStatusListener> authenticationStatusListeners
            = Collections.synchronizedSet(new HashSet<AuthenticationStatusListener>()); //Must be final since its synchronized across threads

    private boolean authenticating = false;     //If authentication is currently happening
    private boolean authenticationRequested = false;    //Tracks whether authentication has ever been requested so we know whether to register network receiver
    private boolean networkReceiverRegistered = false;
    private boolean initialized = false;

    private BroadcastReceiver networkReceiver = new BroadcastReceiver()
    {
        @Override
        public void onReceive(Context context, Intent intent)
        {
            try
            {
                ConnectivityManager manager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);

                //Retry authentication if its been previously requested and we're not authenticated yet and phone came online
                if (manager.getActiveNetworkInfo() != null && manager.getActiveNetworkInfo().isConnected()
                        && isAutoRetryAuthenticationOnConnectedEnabled() && !isAuthenticated() && authenticationRequested)
                {
                    retryAuthentication();
                }
            }catch(Exception e)
            {
                Logger.exception("networkReceiver.onReceive",e);
            }

        }
    };

    private ConnectionResultCallback conciergeConnectionResultCallback = new ConnectionResultCallback()
    {
        @Override
        public void onConnected()
        {
            authenticating = false;

            broadcastAuthenticationState(AuthenticationStatusListener.AUTHENTICATED);

            unregisterNetworkReceiver();

            schedulerWorkers();

            enablePushMessaging(new BasicResultCallback()
            {
                @Override
                public void onSuccess()
                {
                    // Success!
                }

                @Override
                public void onException(FlybitsException exception)
                {
                    Logger.exception(FlybitsConcierge.class.getSimpleName(), exception);
                }
            });

        }

        @Override
        public void notConnected()
        {
            authenticating = false;
        }

        @Override
        public void onException(FlybitsException exception)
        {
            authenticating = false;

            //Hacky for now but only solution without making specific exceptions
            Context context = getContext();
            if (context != null && SharedElements.getSavedJWTToken(context).isEmpty())
            {
                broadcastAuthenticationError(new ConciergeException(exception.getMessage()));
            }else if (context != null && exception.getMessage().contains("Connecting") && isAuthenticated())
            {
                broadcastAuthenticationState(AuthenticationStatusListener.AUTHENTICATED);
            }

            registerNetworkReceiver();
        }
    };

    private FlybitsConcierge(WeakReference<Context> context)
    {
        lastProvidedContext = context;
    }

    public static FlybitsConcierge with(Context context)
    {
        if (context == null)
        {
            throw new ConciergeUninitializedException();
        }

        synchronized (FlybitsConcierge.class)
        {
            if (INSTANCE == null)
            {
                INSTANCE = new FlybitsConcierge(new WeakReference<>(context.getApplicationContext()));
                INSTANCE.createNotificationChannel();
                return INSTANCE;
            }
        }
        INSTANCE.lastProvidedContext = new WeakReference<>(context.getApplicationContext());
        return INSTANCE;
    }

    private void createNotificationChannel()
    {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O)
        {
            CharSequence name = "Concierge Channel";
            String description = "Specific for playing audio content in the concierge.";
            NotificationChannel channel = new NotificationChannel(AUDIO_CHANNEL, name, NotificationManager.IMPORTANCE_HIGH);
            channel.setDescription(description);
            NotificationManager notificationManager = getContext().getSystemService(NotificationManager.class);
            notificationManager.createNotificationChannel(channel);
        }
    }

    private Context getContext() throws ConciergeException
    {
        if (lastProvidedContext != null && lastProvidedContext.get() != null)
        {
            return lastProvidedContext.get();
        }
        throw new ConciergeException("Context was null");
    }

    private FlybitsManager createFlybitsManager(){
        FlybitsManager.Builder builder = new FlybitsManager.Builder(getContext())
                .setAccount(null)
                .setProjectId(getConfiguration().getProjectID())
                .addScope(PushScope.SCOPE)
                .addScope(KernelScope.SCOPE);

        if (BuildConfig.DEBUG)
        {
            builder.setDebug();
        }

        return builder.build();
    }

    public boolean initialize(@IdRes int cfgResource)
    {
        if (initialized){
            return false;
        }

        initialize(ConciergeConfiguration.createFromXML(getContext(), cfgResource));

        initialized = true;

        flybitsManager = createFlybitsManager();

        return true;
    }

    void initialize(ConciergeConfiguration cfg)
    {
        currentConfig = cfg;
    }

    public boolean isInitialized(){
        return initialized;
    }

    /*This method is purposely seperate from the authenticate method because it is needed in order
    to get updates in the ConciergeFragment even if authenticate isn't called in there due to
    AutoAuthenticate being set to false*/
    public void registerAuthenticationStateListener(@NonNull AuthenticationStatusListener authenticationStatusListener)
    {
        synchronized (authenticationStatusListeners) {
            authenticationStatusListeners.add(authenticationStatusListener);
        }
    }

    public boolean unregisterAuthenticationStateListener(@NonNull AuthenticationStatusListener authenticationStatusListener){
        synchronized (authenticationStatusListeners){
            return authenticationStatusListeners.remove(authenticationStatusListener);
        }
    }

    private void broadcastAuthenticationState(String state)
    {

        synchronized (authenticationStatusListeners)
        {
            //Copy over to array to avoid ConcurrentModificationException
            Object[] copy = authenticationStatusListeners.toArray();

            for (Object o : copy)
            {
                AuthenticationStatusListener authenticationStatusListener = (AuthenticationStatusListener)o;
                switch(state){
                    case AuthenticationStatusListener.AUTHENTICATED:
                        authenticationStatusListener.onAuthenticated();
                        break;
                    case AuthenticationStatusListener.AUTHENTICATION_STARTED:
                        authenticationStatusListener.onAuthenticationStarted();
                        break;
                    default:
                }
            }
        }
    }

    private void broadcastAuthenticationError(ConciergeException err)
    {
        synchronized (authenticationStatusListeners)
        {
            //Copy over to array to avoid ConcurrentModificationException
            Object[] copy = authenticationStatusListeners.toArray();

            for (Object o : copy)
            {
                AuthenticationStatusListener authenticationStatusListener = (AuthenticationStatusListener)o;
                authenticationStatusListener.onError(err);
            }
        }
    }

    synchronized boolean retryAuthentication()
    {
        if (!isInitialized())
        {
            throw new ConciergeUninitializedException();
        }
        /*Do not allow for retrying authentication after user logged out or if authentication was never
            requested. That way if the view is shown and the sdk is logged out it won't re-authenticate immediately.*/
        else if(idp == null || authenticating || isAuthenticated() || !authenticationRequested)
        {
            return false;
        }

        authenticate(idp);
        return true;
    }

    public synchronized boolean authenticate(@NonNull IDP idp)
    {

        Context context = getContext();

        if (!isInitialized())
        {
            throw new ConciergeUninitializedException();
        }

        //Automatically opt in if authenticate is called.
        InternalPreferences.setOptOutState(context,false);

        if (isAuthenticated() || authenticating)
        {
            return false;
        }else{
            authenticating = true;
        }

        //Might be null if user opted out
        if (flybitsManager == null){
            flybitsManager = createFlybitsManager();
        }

        this.idp = idp;

        broadcastAuthenticationState(AuthenticationStatusListener.AUTHENTICATION_STARTED);

        authenticationRequested = true;

        flybitsManager.connect(idp,conciergeConnectionResultCallback);

        return true;
    }

    //Can't logout offline
    //Using unique callback here since it may differ from the BasicResultCallback in the future
    public void logOut(final LogOutCallback logOutCallback){
        if (!isInitialized()){
            throw new ConciergeUninitializedException();
        }else if (!isAuthenticated() || authenticating || flybitsManager == null){
            logOutCallback.onError(new ConciergeException("Authentication required before logging out"));
        }else{
            flybitsManager.disconnect(new BasicResultCallback() {
                @Override
                public void onSuccess() {
                    authenticationRequested = false;
                    unregisterNetworkReceiver();
                    if (logOutCallback != null){
                        logOutCallback.onSuccess();
                    }
                }

                @Override
                public void onException(FlybitsException exception) {
                    if (logOutCallback != null){
                        logOutCallback.onError(new ConciergeException(exception));
                    }
                }
            });
        }
    }

    public void setOptOutListener(@NonNull OptOutListener optOutListener)
    {
        this.optOutListener = optOutListener;
    }

    public void clearOptOutListener(){
        this.optOutListener = null;
    }

    public boolean isOptedOut()
    {
        if (isInitialized())
        {
            return InternalPreferences.getOptOutState(getContext());
        }else{
            throw new ConciergeUninitializedException();
        }
    }

    //This is different from logout! Removes all data associated with user.
    public void optOut(@Nullable final OptOutCallback callback)
    {
        if (!isInitialized())
        {
            throw new ConciergeUninitializedException();
        }else if(isOptedOut()){
            throw new ConciergeOptedOutException();
        }else if (!isAuthenticated() || authenticating ||flybitsManager == null){
            callback.onError(new ConciergeException("Authentication required."));
            return;
        }

        final Context context = getContext();
        //Do not allow for opting out if authentication online hasn't happened because there's no way to get a reference to flybits manager to destroy! (as far as I know)
        flybitsManager.destroy(new BasicResultCallback()
        {
            @Override
            public void onSuccess()
            {
                InternalPreferences.saveTNCAccepted(context,false);
                InternalPreferences.setOptOutState(context,true);
                authenticating = false;
                unregisterNetworkReceiver();
                authenticationRequested = false;

                if (optOutListener != null)
                {
                    optOutListener.onUserOptedOut();
                }

                if (callback != null)
                {
                    callback.onSuccess();
                }
                flybitsManager = null;
            }

            @Override
            public void onException(FlybitsException exception)
            {
                if (callback != null)
                {
                    callback.onError(new ConciergeException(exception));
                }
            }
        });
    }

    private boolean unregisterNetworkReceiver(){
        if (networkReceiverRegistered){
            try{
                getContext().unregisterReceiver(networkReceiver);
                networkReceiverRegistered = false;
                return true;
            }catch(Exception e){
                Logger.exception("FlybitsConcierge.unregisterNetworkReceiver()",e);
                return false;
            }
        }else{
            return false;
        }
    }

    private boolean registerNetworkReceiver(){
        //Register receiver responsible for auto authenticating in the future once connection is established
        if (!networkReceiverRegistered && isAutoRetryAuthenticationOnConnectedEnabled())
        {
            IntentFilter intentFilter = new IntentFilter(android.net.ConnectivityManager.CONNECTIVITY_ACTION);
            try{
                getContext().registerReceiver(networkReceiver,intentFilter);
                networkReceiverRegistered = true;
                return true;
            }catch(Exception e){
                Logger.exception("FlybitsConcierge.registerNetworkReceiver()",e);
                return false;
            }
        }else{
            return false;
        }
    }

    public void show(ShowMode mode) throws ConciergeException
    {
        Context context = getContext();
        if (!isInitialized())
        {
            throw new ConciergeUninitializedException();
        }
        else if (InternalPreferences.getOptOutState(context))
        {
            throw new ConciergeOptedOutException();
        }

        Intent activityIntent = null;

        switch (mode)
        {
            case NEW_ACTIVITY:
            {
                activityIntent = new Intent(context, ConciergeActivity.class);
                break;
            }
            case OVERLAY:
            {
                activityIntent = new Intent(context, ConciergePopupActivity.class);
                break;
            }
        }

        activityIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        context.startActivity(activityIntent);
    }

    //If set to false Flybits will not try to re-authenticate later if a connection is established. Set to true by default
    public void setAutoRetryAuthenticationOnConnected(boolean enabled)
    {
        Context context = getContext();
        if (isInitialized())
        {
            InternalPreferences.setAutoRetryAuthOnConnected(context,enabled);
        }else{
            throw new ConciergeUninitializedException();
        }
    }

    public boolean isAutoRetryAuthenticationOnConnectedEnabled()
    {
        Context context = getContext();
        if (isInitialized())
        {
            return InternalPreferences.isAutoRetryAuthOnConnectedEnabled(context);
        }else{
            throw new ConciergeUninitializedException();
        }
    }

    public ConciergeConfiguration getConfiguration()
    {
        if (!isInitialized())
        {
            throw new ConciergeUninitializedException();
        }

        return currentConfig;
    }

    private void enablePushMessaging(BasicResultCallback callback)
    {
        String token = InternalPreferences.pushToken(getContext());

        if (token == null)
        {
            return;
        }

        PushManager.enablePush(getContext(), token, new HashMap<String, String>(), callback);
    }

    public boolean isAuthenticated()
    {
        Context context = getContext();
        if (isInitialized())
        {
            return !SharedElements.getSavedJWTToken(context).isEmpty();
        }else{
            throw new ConciergeUninitializedException();
        }
    }

    public boolean isAuthenticating()
    {
        if (isInitialized())
        {
            return authenticating;
        }else{
            throw new ConciergeUninitializedException();
        }
    }

    private void schedulerWorkers()
    {
        WorkManager workManager = WorkManager.getInstance();

        Constraints workConstraints = new Constraints.Builder().setRequiredNetworkType(NetworkType.CONNECTED)
                .build();
        final WorkRequest workRequest = new OneTimeWorkRequest.Builder(PreloadingWorker.class).setConstraints(workConstraints)
                .build();

        //Worker responsible for preloading
        workManager.enqueue(workRequest);
    }

}
