package com.flybits.commons.library.utils

import android.content.Context
import android.os.Build
import com.flybits.commons.library.BuildConfig
import com.flybits.commons.library.SharedElementsFactory.get
import com.flybits.commons.library.exceptions.InvalidPhysicalDeviceIdException
import com.flybits.commons.library.logging.Logger
import org.json.JSONException
import org.json.JSONObject

private const val TAG_LOGGING = "_Network"
private const val JSON_ATTRIBUTE_NAME = "name"
private const val JSON_ATTRIBUTE_MAKE = "make"
private const val JSON_ATTRIBUTE_MODEL = "model"
private const val JSON_ATTRIBUTE_OS_VERSION = "osVersion"
private const val JSON_ATTRIBUTE_DEVICE_TYPE = "deviceType"
private const val JSON_ATTRIBUTE_SDK_VERSION = "sdkVersion"
private const val JSON_ATTRIBUTE_PHYSICAL_ID = "physicalDeviceId"
private const val JSON_VALUE_ANDROID = "Android"

/**
 * This class is responsible for generating any data needed to successfully make a network request
 * to Flybits. This will eventually be added new HttpRequest class written in Kotlin.
 */
internal class RequestUtilities{

    /**
     * This function is responsible for generating the X-User-Agent header in all Flybits requests.
     *
     * @param context Application context
     * @return The [JSONObject] that contains all the X-User-Agent information for the device.
     */
    fun generateXUserAgentHeader(context: Context)
            : JSONObject{

        val obj = JSONObject()
        try{
            obj.put(JSON_ATTRIBUTE_MAKE, Build.MANUFACTURER)
        }catch (exception : JSONException){
            Logger.appendTag(TAG_LOGGING).w("The $JSON_ATTRIBUTE_MAKE attribute could not be " +
                    "added to the request")
        }

        try{
            obj.put(JSON_ATTRIBUTE_MODEL, Build.MODEL)
        }catch (exception : JSONException){
            Logger.appendTag(TAG_LOGGING).w("The $JSON_ATTRIBUTE_MODEL attribute could not be " +
                    "added to the request")
        }

        try{
            obj.put(JSON_ATTRIBUTE_OS_VERSION, Build.VERSION.SDK_INT.toString())
        }catch (exception : JSONException){
            Logger.appendTag(TAG_LOGGING).w("The $JSON_ATTRIBUTE_OS_VERSION attribute could " +
                    "not be added to the request")
        }

        try{
            obj.put(JSON_ATTRIBUTE_DEVICE_TYPE, JSON_VALUE_ANDROID)
        }catch (exception : JSONException){
            Logger.appendTag(TAG_LOGGING).w("The $JSON_ATTRIBUTE_DEVICE_TYPE attribute could " +
                    "not be added to the request")
        }

        try{
            obj.put(JSON_ATTRIBUTE_SDK_VERSION, BuildConfig.FlybitsVersion)
        }catch (exception : JSONException){
            Logger.appendTag(TAG_LOGGING).w("The $JSON_ATTRIBUTE_SDK_VERSION attribute could " +
                    "not be added to the request")
        }

        try{
            obj.put(JSON_ATTRIBUTE_NAME, "${Build.MANUFACTURER}-${Build.MODEL}")
        }catch (exception : JSONException){
            Logger.appendTag(TAG_LOGGING).w("The $JSON_ATTRIBUTE_NAME attribute could not be " +
                    "added to the request")
        }

        try{
            obj.put(JSON_ATTRIBUTE_PHYSICAL_ID, get(context).getUniqueDeviceId())
        }catch (jsonException : JSONException){
            Logger.appendTag(TAG_LOGGING).w("The $JSON_ATTRIBUTE_PHYSICAL_ID attribute could " +
                    "not be added to the request due to a json exception")
        }catch (physicalDeviceId : InvalidPhysicalDeviceIdException){
            Logger.appendTag(TAG_LOGGING).w("The $JSON_ATTRIBUTE_PHYSICAL_ID attribute could " +
                    "not be added to the request due to an invalid physical device Id")
        }catch (nullPointer : NullPointerException){
            Logger.appendTag(TAG_LOGGING).w("The $JSON_ATTRIBUTE_PHYSICAL_ID attribute could " +
                    "not be added to the request due to an invalid DefaultUniqueDeviceIdGenerator")
        }

        return obj
    }
}