package com.flybits.commons.library.api.results

import java.util.concurrent.*

/**
 * The [FlybitsResult] class is used to query the server and get a response from the server.
 *
 * For example a request to the server can result in a object being returned or a list of objects
 * returned or nothing being returned. This class defines the basic operations such as what happens
 * when a request fails/succeeds/time outs.
 *
 * @param serviceExecutor The [ExecutorService] that is used to run the network request thread.
 * @param time    The time when the [ExecutorService] should shutdown in the event that it
 *                has not been finished. The default value is 30.
 * @param unit    The unit of time used in associated with the time parameter. The default
 * value is TimeUnit.SECONDS.
 */
open class FlybitsResult(val serviceExecutor: ExecutorService = Executors.newSingleThreadExecutor(),
                         time: Long = DEFAULT_TIMEOUT,
                         unit: TimeUnit = DEFAULT_TIMEOUT_UNIT) {

    var status: QueryStatus = QueryStatus.INITIALIZED
    val timeInSeconds = TimeUnit.SECONDS.convert(time, unit)
    val serviceTimeout: ScheduledExecutorService = Executors.newScheduledThreadPool(1)
    var scheduledServiceResult: ScheduledFuture<*>? = null

    companion object {
        private const val DEFAULT_TIMEOUT: Long = 30
        private val DEFAULT_TIMEOUT_UNIT = TimeUnit.SECONDS
    }

    /**
     * Cancels the network request to get more items.
     */
    fun cancel() {
        if (status == QueryStatus.PROCESSING) {
            updateStatus(QueryStatus.CANCELLED)
        }
    }

    /**
     * Execute a command and start the timeout automatically
     *
     * @param runnable The runnable that should be executed.
     */
    fun execute(runnable: Runnable) {
        status = QueryStatus.PROCESSING
        scheduledServiceResult = serviceTimeout.schedule(r, timeInSeconds, TimeUnit.SECONDS)
        serviceExecutor.execute(runnable)
    }

    /**
     * Sets the [FlybitsException] that was thrown when request more items to be loaded.
     *
     * @return true if the error has been set, false otherwise
     */
    fun failed(): Boolean {
        return if (status != QueryStatus.CANCELLED && status != QueryStatus.TIMED_OUT) {
            updateStatus(QueryStatus.FAILED)
            true
        } else false
    }

    /**
     * Sets the result of the network request used to retrieve more items. In all cases, this should
     * include the items retrieved along with the returned [Pagination] object.
     *
     * @return true if the request was successful, false otherwise
     */
    fun success(): Boolean {
        return if (status != QueryStatus.CANCELLED && status != QueryStatus.TIMED_OUT) {
            updateStatus(QueryStatus.SUCCESS)
            true
        } else false
    }

    private fun cancelRequest() {
        if (!serviceExecutor.isShutdown) {
            serviceExecutor.shutdownNow()
        }
    }

    private fun cancelTimeout() {
        scheduledServiceResult?.cancel(true)
    }

    private fun updateStatus(status: QueryStatus) {
        this.status = status
        cancelRequest()
        cancelTimeout()
    }

    private val r = Runnable {
        if (status == QueryStatus.PROCESSING) {
            updateStatus(QueryStatus.TIMED_OUT)
        }
    }
}