package com.flybits.commons.library.encryption

import android.content.Context
import android.content.SharedPreferences
import android.content.SharedPreferences.OnSharedPreferenceChangeListener
import android.os.Build
import android.util.Log
import com.flybits.commons.library.BuildConfig
import com.flybits.commons.library.encryption.EncryptionManager
import com.flybits.commons.library.encryption.EncryptionManager.InvalidMacException
import com.flybits.commons.library.logging.Logger
import java.io.File
import java.io.IOException
import java.io.UnsupportedEncodingException
import java.security.*
import java.security.cert.CertificateException
import java.util.*
import javax.crypto.BadPaddingException
import javax.crypto.IllegalBlockSizeException
import javax.crypto.NoSuchPaddingException

/***
 *  * @param appContext application context
 * @param storeName optional name of the preference file
 * @param keyPrefix optional prefix for encryption key aliases
 * @param bitShiftingKey seed for randomization and bit shifting, enhances security on older OS versions
 *
 *
 */
const val TAG = "flybits_encryption"
const val TAG_EXCEPTION = "throw excpetion"

class SecuredPreferenceStore : SharedPreferences {
    private val RESERVED_KEYS: Array<String?>
    private val mPrefs: SharedPreferences
    val encryptionManager: EncryptionManager


    private fun isReservedKey(key: String): Boolean {
        return Arrays.asList(*RESERVED_KEYS).contains(key)
    }

    private fun isReservedHashedKey(hashedKey: String): Boolean {
        for (key in RESERVED_KEYS) {
            try {
                if (hashedKey == EncryptionManager.getHashed(key)) {
                    return true
                }
            } catch (e: NoSuchAlgorithmException) {
                Logger.appendTag(TAG).e("", e)
            } catch (e: UnsupportedEncodingException) {
                Logger.appendTag(TAG).e(TAG_EXCEPTION, e)
            }
        }
        return false
    }

    override fun getAll(): Map<String, Any?> {
        val all = mPrefs.all
        val dAll: MutableMap<String, Any?> =
            HashMap(all.size)
        if (all.size > 0) {
            for (key in all.keys) {
                if (key == VERSION_KEY || isReservedHashedKey(key)) continue
                try {
                    val value = all[key]
                    dAll[key] = encryptionManager.decrypt(value as String?)
                } catch (e: Exception) {
                    Logger.appendTag(TAG).e(TAG_EXCEPTION, e)
                }
            }
        }
        return dAll
    }

    override fun getString(key: String, defValue: String?): String? {
        if (!isReservedKey(key)) {
            try {
                val hashedKey: String = EncryptionManager.Companion.getHashed(key)
                val value = mPrefs.getString(hashedKey, null)
                if (value != null) {
                    val text = encryptionManager.decrypt(value)
                    Log.e(TAG, "decrypttext $text")
                    return text
                }
            } catch (e: Exception) {
                Logger.appendTag(TAG).e(TAG_EXCEPTION, e)
            }
        }
        return defValue
    }

    override fun getStringSet(
        key: String,
        defValues: Set<String>?
    ): Set<String>? {
        if (!isReservedKey(key)) {
            try {
                val hashedKey: String = EncryptionManager.Companion.getHashed(key)
                val eSet =
                    mPrefs.getStringSet(hashedKey, null)
                if (eSet != null) {
                    val dSet: MutableSet<String> =
                        HashSet(eSet.size)
                    for (`val` in eSet) {
                        val toAdd = encryptionManager.decrypt(`val`) ?: continue
                        dSet.add(toAdd)
                    }
                    return dSet
                }
            } catch (e: Exception) {
                Logger.appendTag(TAG).e(TAG_EXCEPTION, e)
            }
        }
        return defValues
    }

    override fun getInt(key: String, defValue: Int): Int {
        val value = getString(key, null)
        return value?.toInt() ?: defValue
    }

    override fun getLong(key: String, defValue: Long): Long {
        val value = getString(key, null)
        return value?.toLong() ?: defValue
    }

    override fun getFloat(key: String, defValue: Float): Float {
        val value = getString(key, null)
        return value?.toFloat() ?: defValue
    }

    override fun getBoolean(key: String, defValue: Boolean): Boolean {
        val value = getString(key, null)
        return if (value != null) {
            java.lang.Boolean.parseBoolean(value)
        } else defValue
    }

    fun getBytes(key: String): ByteArray? {
        val `val` = getString(key, null)
        return if (`val` != null) {
            EncryptionManager.Companion.base64Decode(`val`)
        } else null
    }

    override fun contains(key: String): Boolean {
        try {
            val hashedKey: String = EncryptionManager.Companion.getHashed(key)
            return mPrefs.contains(hashedKey)
        } catch (e: Exception) {
            Logger.appendTag(TAG).e(TAG_EXCEPTION, e)
        }
        return false
    }

    override fun edit(): Editor {
        return Editor()
    }

    override fun registerOnSharedPreferenceChangeListener(onSharedPreferenceChangeListener: OnSharedPreferenceChangeListener) {
        mPrefs.registerOnSharedPreferenceChangeListener(
            onSharedPreferenceChangeListener
        )
    }

    override fun unregisterOnSharedPreferenceChangeListener(onSharedPreferenceChangeListener: OnSharedPreferenceChangeListener) {
        mPrefs.unregisterOnSharedPreferenceChangeListener(
            onSharedPreferenceChangeListener
        )
    }

    inner class Editor : SharedPreferences.Editor {
        var mEditor: SharedPreferences.Editor
        override fun putString(
            key: String,
            value: String?
        ): SharedPreferences.Editor {
            if (isReservedKey(key)) {
                Logger.e("Trying to store value for a reserved key, value: $value")
                return this
            }
            try {
                val hashedKey: String = EncryptionManager.Companion.getHashed(key)
                val evalue = encryptionManager.encrypt(value)
                mEditor.putString(hashedKey, evalue)
            } catch (e: Exception) {
                Logger.appendTag(TAG).e(TAG_EXCEPTION, e)
            }
            return this
        }

        override fun putStringSet(
            key: String,
            values: Set<String>?
        ): SharedPreferences.Editor {
            if (isReservedKey(key)) {
                Logger.e("Trying to store value for a reserved key, value: $values")
                return this
            }
            if (values.isNullOrEmpty()) {
                Logger.e("$TAG putStringSet values is null or empty")
                return this
            }

            try {
                val hashedKey: String = EncryptionManager.Companion.getHashed(key)
                val eSet: MutableSet<String?> =
                    HashSet(values.size)
                for (`val` in values) {
                    eSet.add(encryptionManager.encrypt(`val`))
                }
                mEditor.putStringSet(hashedKey, eSet)
            } catch (e: Exception) {
                Logger.appendTag(TAG).e(TAG_EXCEPTION, e)
            }
            return this
        }

        override fun putInt(key: String, value: Int): SharedPreferences.Editor {
            val `val` = Integer.toString(value)
            return putString(key, `val`)
        }

        override fun putLong(
            key: String,
            value: Long
        ): SharedPreferences.Editor {
            val `val` = java.lang.Long.toString(value)
            return putString(key, `val`)
        }

        override fun putFloat(
            key: String,
            value: Float
        ): SharedPreferences.Editor {
            val `val` = java.lang.Float.toString(value)
            return putString(key, `val`)
        }

        override fun putBoolean(
            key: String,
            value: Boolean
        ): SharedPreferences.Editor {
            val `val` = java.lang.Boolean.toString(value)
            return putString(key, `val`)
        }

        fun putBytes(key: String, bytes: ByteArray?): SharedPreferences.Editor {
            return if (bytes != null) {
                val `val`: String = EncryptionManager.Companion.base64Encode(bytes)
                putString(key, `val`)
            } else remove(key)
        }

        override fun remove(key: String): SharedPreferences.Editor {
            if (isReservedKey(key)) {
                Logger.e("Trying to remove value for a reserved key")
                return this
            }
            try {
                val hashedKey: String = EncryptionManager.Companion.getHashed(key)
                mEditor.remove(hashedKey)
            } catch (e: Exception) {
                Logger.appendTag(TAG).e(TAG_EXCEPTION, e)
            }
            return this
        }

        override fun clear(): SharedPreferences.Editor {
            for (key in mPrefs.all.keys) {
                if (key == VERSION_KEY || isReservedHashedKey(key)) continue
                mEditor.remove(key)
            }
            return this
        }

        override fun commit(): Boolean {
            return mEditor.commit()
        }

        override fun apply() {
            mEditor.apply()
        }

        init {
            mEditor = mPrefs.edit()
        }
    }

    interface KeyStoreRecoveryNotifier {
        /**
         *
         * @param e
         * @param keyStore
         * @param keyAliases
         * @return true if the error could be resolved
         */
        fun onRecoveryRequired(
            e: Exception,
            keyStore: KeyStore?,
            keyAliases: List<String?>?
        ): Boolean
    }

    //region Migration
    private inner class MigrationHelper(
        var mContext: Context,
        var storeName: String?,
        var keyPrefix: String?,
        var bitShiftKey: ByteArray?
    ) {

        /**
         * if storeName has changed from the default and there's data in the default file then those will be moved to the new file
         * if keyPrefix has changed from the default and there aren't any other prefix stored in the file, then new keys will be stored
         * with the new prefix and existing data will be migrated
         * @throws MigrationFailedException
         * @throws
         */
        @Throws(MigrationFailedException::class)
        fun migrateToV10() {
            if (storeName == null && keyPrefix == null && bitShiftKey == null) {
                //using the defaults, so no migration needed
                return
            }
            val prefToRead: SharedPreferences
            var prefToWrite: SharedPreferences
            prefToWrite = mContext.getSharedPreferences(
                DEFAULT_PREF_FILE_NAME,
                Context.MODE_PRIVATE
            )
            prefToRead = prefToWrite
            var filenameChanged = false
            var prefixChanged = false
            if (storeName != null && storeName != DEFAULT_PREF_FILE_NAME) {
                prefToWrite =
                    mContext.getSharedPreferences(storeName, Context.MODE_PRIVATE)
                filenameChanged = true
            }
            var storedPrefix: String? = null
            storedPrefix = try {
                prefToWrite.getString(
                    EncryptionManager.Companion.getHashed(EncryptionManager.Companion.OVERRIDING_KEY_ALIAS_PREFIX_NAME),
                    null
                )
            } catch (e: NoSuchAlgorithmException) {
                throw MigrationFailedException(
                    "Migration to Version: 0.7.0: Failed to hash a key",
                    e
                )
            } catch (e: UnsupportedEncodingException) {
                throw MigrationFailedException(
                    "Migration to Version: 0.7.0: Failed to hash a key",
                    e
                )
            }
            prefixChanged =
                storedPrefix == null && keyPrefix != null && keyPrefix != EncryptionManager.Companion.DEFAULT_KEY_ALIAS_PREFIX
            if ((filenameChanged || prefixChanged) && prefToRead.all.size > 0) {
                try {
                    val readCrypto =
                        EncryptionManager(mContext, prefToRead, null)
                    val writeCrypto =
                        EncryptionManager(mContext, prefToWrite, keyPrefix, bitShiftKey, null)
                    val allData = prefToRead.all
                    val editor = prefToWrite.edit()
                    for ((hashedKey, value1) in allData) {
                        if (hashedKey == EncryptionManager.Companion.getHashed(readCrypto.AES_KEY_ALIAS) || hashedKey == EncryptionManager.Companion.getHashed(
                                readCrypto.IS_COMPAT_MODE_KEY_ALIAS
                            ) || hashedKey == EncryptionManager.Companion.getHashed(readCrypto.MAC_KEY_ALIAS)
                        ) {
                            continue
                        }
                        if (value1 == null) continue
                        if (value1 is Set<*>) { //string set
                            val values =
                                value1 as Set<String>
                            val eValues: MutableSet<String?> =
                                HashSet()
                            for (value in values) {
                                val dValue = readCrypto.decrypt(value)
                                eValues.add(writeCrypto.encrypt(dValue))
                            }
                            editor.putStringSet(hashedKey, eValues)
                        } else if (value1 is String) { //string
                            val dValue =
                                readCrypto.decrypt(value1 as String?)
                            editor.putString(hashedKey, writeCrypto.encrypt(dValue))
                        } else {
                            Logger.e("Found a value that is not String or Set, key: $hashedKey, value: $value1")
                        }
                    }
                    if (editor.commit()) {
                        editor.putInt(VERSION_KEY, 10).apply()
                        cleanupPref(DEFAULT_PREF_FILE_NAME)
                    }
                } catch (e: InvalidKeyException) {
                    throw MigrationFailedException(
                        "Migration to Version: 0.7.0: Encryption/Hashing Error",
                        e
                    )
                } catch (e: UnrecoverableEntryException) {
                    throw MigrationFailedException(
                        "Migration to Version: 0.7.0: Encryption/Hashing Error",
                        e
                    )
                } catch (e: KeyStoreException) {
                    throw MigrationFailedException(
                        "Migration to Version: 0.7.0: Encryption/Hashing Error",
                        e
                    )
                } catch (e: NoSuchAlgorithmException) {
                    throw MigrationFailedException(
                        "Migration to Version: 0.7.0: Encryption/Hashing Error",
                        e
                    )
                } catch (e: NoSuchProviderException) {
                    throw MigrationFailedException(
                        "Migration to Version: 0.7.0: Encryption/Hashing Error",
                        e
                    )
                } catch (e: CertificateException) {
                    throw MigrationFailedException(
                        "Migration to Version: 0.7.0: Encryption/Hashing Error",
                        e
                    )
                } catch (e: UnsupportedEncodingException) {
                    throw MigrationFailedException(
                        "Migration to Version: 0.7.0: Encryption/Hashing Error",
                        e
                    )
                } catch (e: InvalidAlgorithmParameterException) {
                    throw MigrationFailedException(
                        "Migration to Version: 0.7.0: Encryption/Hashing Error",
                        e
                    )
                } catch (e: InvalidMacException) {
                    throw MigrationFailedException(
                        "Migration to Version: 0.7.0: Encryption/Hashing Error",
                        e
                    )
                } catch (e: IllegalBlockSizeException) {
                    throw MigrationFailedException(
                        "Migration to Version: 0.7.0: Encryption/Hashing Error",
                        e
                    )
                } catch (e: BadPaddingException) {
                    throw MigrationFailedException(
                        "Migration to Version: 0.7.0: Encryption/Hashing Error",
                        e
                    )
                } catch (e: NoSuchPaddingException) {
                    throw MigrationFailedException(
                        "Migration to Version: 0.7.0: Encryption/Hashing Error",
                        e
                    )
                } catch (e: IOException) {
                    throw MigrationFailedException(
                        "Migration to Version: 0.7.0: Encryption/Hashing Error",
                        e
                    )
                }
            }
        }

        @Throws(MigrationFailedException::class)
        fun migrate(fromVersion: Int, toVersion: Int) {
            var fromVersion = fromVersion
            if (fromVersion >= toVersion) {
                return
            }
            for (version in VERSIONS_WITH_BREAKING_CHANGES) {
                if (fromVersion < version) {
                    migrate(version)
                    fromVersion = version
                }
            }
            mPrefs.edit().putInt(VERSION_KEY, toVersion).apply()
        }

        @Throws(MigrationFailedException::class)
        fun migrate(toVersion: Int) {
            if (toVersion == 10) {

                Logger.d("Migrating to: $toVersion")
                migrateToV10()
            }
        }

        fun cleanupPref(storeName: String) {
            val prefs =
                mContext.getSharedPreferences(storeName, Context.MODE_PRIVATE)
            if (prefs.all.size > 0) prefs.edit().clear().commit()
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                mContext.deleteSharedPreferences(storeName)
            } else {
                try {
                    File(
                        mContext.cacheDir.parent + "/shared_prefs/" + storeName + ".xml"
                    ).delete()
                } catch (e: Exception) {
                    Logger.w("Unable to remove store file completely")
                }
            }
        }

    }

    inner class MigrationFailedException(
        message: String?,
        cause: Throwable?
    ) : Exception(message, cause)  //endregion

    companion object {
        private val VERSIONS_WITH_BREAKING_CHANGES =
            intArrayOf(10) //version code in ascending order
        const val VERSION_KEY = "VERSION"
        private const val DEFAULT_PREF_FILE_NAME = "SPS_file"
        private var mRecoveryHandler: RecoveryHandler? = null
        private var mInstance: SecuredPreferenceStore? = null
        fun setRecoveryHandler(recoveryHandler: RecoveryHandler?) {
            mRecoveryHandler = recoveryHandler
        }


        @Synchronized
        fun getSharedInstance(): SecuredPreferenceStore? {
//            if (mInstance == null)
//                throw IllegalStateException("Must call init() before using the store")
            checkNotNull(mInstance) { "Must call init() before using the store" }
            return mInstance
        }

        /**
         * Must be called once before using the SecuredPreferenceStore to initialize the shared instance.
         * You may call it in @code{onCreate} method of your application class or launcher activity

         * @throws IOException
         * @throws CertificateException
         * @throws NoSuchAlgorithmException
         * @throws KeyStoreException
         * @throws UnrecoverableEntryException
         * @throws InvalidAlgorithmParameterException
         * @throws NoSuchPaddingException
         * @throws InvalidKeyException
         * @throws NoSuchProviderException
         */
        @Throws(
            IOException::class,
            CertificateException::class,
            NoSuchAlgorithmException::class,
            KeyStoreException::class,
            UnrecoverableEntryException::class,
            InvalidAlgorithmParameterException::class,
            NoSuchPaddingException::class,
            InvalidKeyException::class,
            NoSuchProviderException::class,
            MigrationFailedException::class
        )
        fun init(
            appContext: Context,
            storeName: String?,
            keyPrefix: String?,
            bitShiftingKey: ByteArray?,
            recoveryHandler: RecoveryHandler?
        ) {
            if (mInstance != null) {
                Logger.w("init called when there already is a non-null instance of the class")
                return
            }
            setRecoveryHandler(recoveryHandler)
            mInstance =
                SecuredPreferenceStore(appContext, storeName, keyPrefix, bitShiftingKey)
        }

        /**
         * @see .init
         */
        @Deprecated("Use the full constructor for better security, specially on older OS versions")
        @Throws(
            IOException::class,
            CertificateException::class,
            NoSuchAlgorithmException::class,
            InvalidKeyException::class,
            UnrecoverableEntryException::class,
            InvalidAlgorithmParameterException::class,
            NoSuchPaddingException::class,
            NoSuchProviderException::class,
            KeyStoreException::class,
            MigrationFailedException::class
        )
        fun init(
            appContext: Context,
            recoveryHandler: RecoveryHandler?
        ) {
            init(
                appContext,
                DEFAULT_PREF_FILE_NAME,
                null,
                null,
                recoveryHandler
            )
        }
    }

    /** *  * @param appContext application context
     * @param storeName optional name of the preference file
     * @param keyPrefix optional prefix for encryption key aliases
     * @param bitShiftingKey seed for randomization and bit shifting, enhances security on older OS versions
     *  @param recoveryHandler recovery handler to use if necessary
     * @param appContext application context
     * @param storeName optional name of the preference file
     * @param keyPrefix optional prefix for encryption key aliases
     * @param bitShiftingKey seed for randomization and bit shifting, enhances security on older OS versions
     * @throws IOException
     * @throws CertificateException
     * @throws NoSuchAlgorithmException
     * @throws KeyStoreException
     * @throws UnrecoverableEntryException
     * @throws InvalidAlgorithmParameterException
     * @throws NoSuchPaddingException
     * @throws InvalidKeyException
     * @throws NoSuchProviderException
     * @throws MigrationFailedException
     */
    private constructor(
        appContext: Context, storeName: String?, keyPrefix: String?,
        bitShiftingKey: ByteArray?
    ) {
        Logger.d("Creating store instance")
        // handle migration
        val fileName =
            storeName ?: DEFAULT_PREF_FILE_NAME
        mPrefs = appContext.getSharedPreferences(fileName, Context.MODE_PRIVATE)
//        val mRunningVersion = mPrefs.getInt(VERSION_KEY, 9)
//        if (mRunningVersion < BuildConfig.VERSION_CODE) {
//            MigrationHelper(appContext, storeName, keyPrefix, bitShiftingKey)
//                .migrate(mRunningVersion, BuildConfig.VERSION_CODE)
//        }
        encryptionManager = EncryptionManager(
            appContext,
            mPrefs,
            keyPrefix,
            bitShiftingKey,
            object : KeyStoreRecoveryNotifier {
                override fun onRecoveryRequired(
                    e: Exception,
                    keyStore: KeyStore?,
                    keyAliases: List<String?>?
                ): Boolean {
                    return mRecoveryHandler?.recover(
                        e,
                        keyStore,
                        keyAliases,
                        mPrefs
                    ) ?: throw RuntimeException(e)
                }
            })
        RESERVED_KEYS = arrayOf(
            VERSION_KEY,
            EncryptionManager.OVERRIDING_KEY_ALIAS_PREFIX_NAME,
            encryptionManager.IS_COMPAT_MODE_KEY_ALIAS,
            encryptionManager.MAC_KEY_ALIAS,
            encryptionManager.AES_KEY_ALIAS
        )
    }
}