package com.flybits.commons.library

import android.content.Context
import com.flybits.commons.library.logging.Logger
import com.flybits.internal.db.CommonsDatabase
import devliving.online.securedpreferencestore.DefaultRecoveryHandler
import devliving.online.securedpreferencestore.SecuredPreferenceStore
import kotlinx.coroutines.*

/**
 * Factory object used for creating [SharedElements]. Depending on whether or not encryption
 * is supported on the device, [SecuredSharedElements] or [UnsecuredSharedElements] will
 * be returned by the [SharedElementsFactory.get] method.
 */
object SharedElementsFactory {
    private var fallbackToMemory: Boolean = false
    private var retainSharedPref: Boolean = true

    @JvmField
    @Volatile
    internal var savedInstance: SharedElements? = null



    fun get(context: Context): SharedElements =
        savedInstance ?: get(context, fallbackToMemory, retainSharedPref)


    /**
     * Get instance of [SharedElements].
     *
     * @param context Context of the application.
     * @param  fallbackToMemory Boolean value to determine if the data needs to be stored in
     *          the memory in-case Encryption on Shared Prefs fails.
     * @param retainSharedPref  Boolean value to determine to retain the data stored in Shared Prefs
     * when [fallbackToMemory] is set to true and Encryption on Shared Prefs fails:
     * if true,  retain shared preference data
     * if false, remove the sensitive shared preference data
     * @return [SecuredSharedElements] if encryption is supported. [UnsecuredSharedElements] otherwise.
     */
    fun get(
        context: Context,
        fallbackToMemory: Boolean = false,
        retainSharedPref: Boolean = true
    ): SharedElements {
        this.fallbackToMemory = fallbackToMemory
        this.retainSharedPref = retainSharedPref
        return savedInstance ?: synchronized(this) {
            val userDAO = CommonsDatabase.getDatabase(context).userDao()
            savedInstance ?: try {
                Logger.appendTag(SharedElements.TAG)
                    .d("get(): Creating first instance of SharedElements")
                var time1 = System.currentTimeMillis()
                SecuredPreferenceStore.init(
                    context, SharedElements.FLYBITS_STORAGE_ENC_V2, null
                    , null, DefaultRecoveryHandler()
                )

                SecuredSharedElements(SecuredPreferenceStore.getSharedInstance(), userDAO)
                    .also {
                        savedInstance = it }


            } catch (e: Exception) {
                Logger.exception("SharedElementsFactory.INSTANCE.get()", e)
                if (!fallbackToMemory) {
                    Logger.appendTag(SharedElements.TAG).d("Creating instance of UnsecuredSharedElements")
                    UnsecuredSharedElements(
                        context.getSharedPreferences(
                            SharedElements.FLYBITS_STORAGE_UNENCRYPTED_V2, Context.MODE_PRIVATE
                        ), userDAO
                    ).also { savedInstance = it }
                } else {
                    Logger.appendTag(SharedElements.TAG).d("Creating instance of SharedElementsInMemory")
                    SharedElementsInMemory(
                        context.getSharedPreferences(
                            SharedElements.FLYBITS_STORAGE_UNENCRYPTED_V2, Context.MODE_PRIVATE
                        ), userDAO, this.retainSharedPref
                    ).also { savedInstance = it }
                }

            }
        }
    }

}