package com.flybits.commons.library.analytics

import android.content.Context
import androidx.work.*
import com.flybits.commons.library.api.FlyAway
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.http.RequestStatus
import com.flybits.commons.library.logging.Logger
import com.flybits.commons.library.models.CtxData
import com.flybits.internal.db.CommonsDatabase
import org.json.JSONObject
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit

/**
 * This class is responsible for tracking analytics data, this including: scheduling workers
 * , storing the data in the local database, and formatting the data in the expected format.
 *
 */
open class Analytics(private val context: Context) {

    private val TAG_LOGGING = "Analytics"
    private val UPLOAD_INTERVAL_METERED = 60L
    private val UPLOAD_INTERVAL_UNMETERED = 20L
    private val CTX_DATA_ENDPOINT = "/context/ctxdata"

    /**
     * Represents an application event that will be tracked by the Flybits Analytics system.
     *
     * @param analyticsScope Scope of the analytics e.x push or content.
     * @param action Action that occurred e.x viewed or engaged.
     * @param dataId Identification of the data associated with the event.
     * @param timestamp When the event occurred in epoch time in milliseconds.
     *
     */
    data class Event(val analyticsScope: String, val action: String, val dataId: String, val timestamp: Long)

    private val ctxDataDAO = CommonsDatabase.getDatabase(context).ctxDataDAO()

    companion object {
        const val TAG_METERED_WORK = "com.flybits.commons.library.analytics.meteredWork"
        const val TAG_UNMETERED_WORK = "com.flybits.commons.library.analytics.unmeteredWork"
    }

    private fun Event.toCtxData(): CtxData{
        val analyticsJson = JSONObject()
        analyticsJson.put("dataTypeID","ctx.flybits.$analyticsScope")
        val valueJson = JSONObject()
        valueJson.put("query.$action.$dataId", true)
        analyticsJson.put("value", valueJson)
        analyticsJson.put("timestamp",timestamp/1000) // Convert the timestamp to seconds prior sending to the server as required.
        return CtxData(timestamp = timestamp, value = analyticsJson.toString())
    }

    /**
     * Flush all data from the ctxDatabase to the server then delete the data if it was
     * sent with success.
     *
     * @param callback Callback which will be informed about whether the flush succeeded or failed.
     * @param async Whether or not to execute on a separate thread, true by default.
     *
     */
    @Throws(FlybitsException::class)
    fun flush( callback: ((success: Boolean, exception: Exception?) -> Unit)? = null, async: Boolean = true){
        Logger.appendTag(TAG_LOGGING).i("Flushing analytics data")
        if (async) Executors.newSingleThreadExecutor().execute {
            try {
                flush(callback)
            } catch (e: Exception) {
                callback?.invoke(false, e)
                Logger.appendTag(TAG_LOGGING).e("Flushing analytics data failed", e)
            }
        }else flush(callback)
    }

    @Throws(FlybitsException::class)
    private fun flush(callback: ((success: Boolean, exception: Exception?) -> Unit)? = null){
        val ctxDataDAO = CommonsDatabase.getDatabase(context).ctxDataDAO()
        val ctxData = ctxDataDAO.getAll()

        if (!ctxData.isEmpty()) {
            Logger.appendTag(TAG_LOGGING).i("Flushing analytics data - Non-Empty list")
            val body = ctxData.joinToString(separator = ",", prefix = "[", postfix = "]") { it.value }
            val result = FlyAway.post<Any>(context
                    , CTX_DATA_ENDPOINT, body, null
                    , "AnalyticsWorker.doWork", null)

            if (result.status == RequestStatus.COMPLETED) {
                //We cannot just delete all since by the time the response comes back an item which has not been sent may have been added
                ctxDataDAO.deleteMany(ctxData)
                callback?.invoke(true, null)
                Logger.appendTag(TAG_LOGGING).i("Flushing analytics data success")
            }else{
                callback?.invoke(false, result.exception)
                Logger.appendTag(TAG_LOGGING).e("Flushing analytics data failed", result.exception)
            }
        }else{
            callback?.invoke(true, null)
            Logger.appendTag(TAG_LOGGING).i("Flushing analytics data - Empty list, nothing sent")
        }
    }

    /**
     * Enqueue a {@link PeriodicWorkRequest} which involves a worker periodically
     * uploading data from the local database to the server pertaining to analytics.
     * Data will be sent to the server for both a metered and unmetered connection.
     * Metered connection data will be sent less frequently.
     *
     * @return false if both metered and unmetered workers have been scheduled, true otherwise.
     */
    fun scheduleWorkers(){
        scheduleMeteredWorker()
        scheduleUnmeteredWorker()
        Logger.appendTag(TAG_LOGGING).i("Analytics workers are scheduled")
    }

    fun cancelWorkers(){
        cancelUnmeteredWork()
        cancelMeteredWork()
        Logger.appendTag(TAG_LOGGING).i("Analytics workers are cancelled")
    }

    /**
     * Destroy all stored analytics data, and workers responsible for uploading it.
     *
     */
    fun destroy(){
        cancelWorkers()
        Executors.newSingleThreadExecutor().execute{
            ctxDataDAO.deleteAll()
            Logger.appendTag(TAG_LOGGING).i("Ctxdata database in Commons cleared")
        }
    }

    private fun cancelUnmeteredWork(){
        WorkManager.getInstance().cancelUniqueWork(TAG_UNMETERED_WORK)
    }

    private fun cancelMeteredWork(){
        WorkManager.getInstance().cancelUniqueWork(TAG_METERED_WORK)
    }

    private fun scheduleMeteredWorker(){
        val constraintsMetered = Constraints.Builder()
                .setRequiredNetworkType(NetworkType.METERED)
                .build()

        val workRequestMetered = PeriodicWorkRequest
                .Builder(AnalyticsWorker::class.java, UPLOAD_INTERVAL_METERED, TimeUnit.MINUTES)
                .setConstraints(constraintsMetered)
                .addTag(TAG_METERED_WORK)
                .build()

        WorkManager.getInstance().enqueueUniquePeriodicWork(TAG_METERED_WORK
                , ExistingPeriodicWorkPolicy.KEEP
                , workRequestMetered)
    }

    private fun scheduleUnmeteredWorker(){
        val constraintsMetered = Constraints.Builder()
                .setRequiredNetworkType(NetworkType.UNMETERED)
                .build()

        val workRequestMetered = PeriodicWorkRequest
                .Builder(AnalyticsWorker::class.java, UPLOAD_INTERVAL_UNMETERED, TimeUnit.MINUTES)
                .setConstraints(constraintsMetered)
                .addTag(TAG_UNMETERED_WORK)
                .build()

        WorkManager.getInstance().enqueueUniquePeriodicWork(TAG_UNMETERED_WORK
                , ExistingPeriodicWorkPolicy.KEEP
                , workRequestMetered)
    }

    /**
     * Queue an event for tracking which will be uploaded to the server in the future.
     *
     * Runs on a separate thread.
     *
     * @param event Analytics event that you want to track.
     */
    protected fun track(event: Event) {
        Executors.newSingleThreadExecutor().execute {
            Logger.appendTag(TAG_LOGGING)
                .i("Received event: ${event.analyticsScope} - ${event.action} - ${event.dataId}")

            val data = event.toCtxData()
            if (checkIfUnique(data)) {
                ctxDataDAO?.insert(data)
                Logger.appendTag(TAG_LOGGING)
                    .i("Recorded event: ${data.id} - ${data.value}")
            }
        }
    }

    private fun checkIfUnique(ctxData: CtxData): Boolean {
        synchronized(this) {
            val dataFromCursor = ctxDataDAO.getAll()
            dataFromCursor.let {
                dataFromCursor.forEach {
                    if (it.value == ctxData.value) {
                        return false
                    }
                }
                return true
            }
        }
    }
}