package com.flybits.internal.db;

import android.arch.persistence.db.SupportSQLiteDatabase;
import android.arch.persistence.room.Database;
import android.arch.persistence.room.Room;
import android.arch.persistence.room.RoomDatabase;
import android.arch.persistence.room.migration.Migration;
import android.content.Context;

import com.flybits.commons.library.models.CtxData;
import com.flybits.commons.library.models.User;
import com.flybits.internal.db.models.CachingEntry;
import com.flybits.internal.db.models.Preference;

@Database(entities = {User.class, CachingEntry.class, Preference.class, CtxData.class}, version = 8)
public abstract class CommonsDatabase extends RoomDatabase {

    static final Migration MIGRATION_1_4 = new Migration(1, 4) {
        @Override
        public void migrate(SupportSQLiteDatabase database) {
            database.execSQL("ALTER TABLE user ADD isVerified INTEGER NOT NULL DEFAULT 0");
            database.execSQL("DROP TABLE IF EXISTS cachingentry");
            database.execSQL("CREATE TABLE cachingentry (sequence INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL, cachingkey TEXT NOT NULL, contentID TEXT NOT NULL)");
            database.execSQL("CREATE UNIQUE INDEX index_cachingentry_cachingkey_contentID on cachingentry (cachingkey, contentID)");
            database.execSQL("CREATE TABLE preferences (id INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL, prefKey TEXT, value TEXT)");
        }
    };
    static final Migration MIGRATION_2_4 = new Migration(2, 4) {
        @Override
        public void migrate(SupportSQLiteDatabase database) {
            database.execSQL("DROP TABLE IF EXISTS cachingentry");
            database.execSQL("CREATE TABLE cachingentry (sequence INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL, cachingkey TEXT NOT NULL, contentID TEXT NOT NULL)");
            database.execSQL("CREATE UNIQUE INDEX index_cachingentry_cachingkey_contentID on cachingentry (cachingkey, contentID)");
            database.execSQL("CREATE TABLE preferences (id INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL, prefKey TEXT, value TEXT)");
        }
    };
    static final Migration MIGRATION_3_4 = new Migration(3, 4) {
        @Override
        public void migrate(SupportSQLiteDatabase database) {
            database.execSQL("CREATE TABLE preferences (id INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL, prefKey TEXT, value TEXT)");
        }
    };
    static final Migration MIGRATION_4_5 = new Migration(4, 5) {
        @Override
        public void migrate(SupportSQLiteDatabase database) {
            database.execSQL("CREATE TABLE ctxData (id INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL, timestamp INTEGER NOT NULL,value TEXT NOT NULL)");
        }
    };
    static final Migration MIGRATION_5_6 = new Migration(5, 6) {
        @Override
        public void migrate(SupportSQLiteDatabase database) {
            database.execSQL("ALTER TABLE user ADD isOptedIn INTEGER NOT NULL DEFAULT 1");
            database.execSQL("DROP TABLE IF EXISTS preferences");
            database.execSQL("CREATE TABLE IF NOT EXISTS preferences (prefKey TEXT NOT NULL, value TEXT NOT NULL, PRIMARY KEY(prefKey, value))");
        }
    };
    static final Migration MIGRATION_6_7 = new Migration(6, 7) {
        @Override
        public void migrate(SupportSQLiteDatabase database) {
            database.execSQL("ALTER TABLE user ADD isActive INTEGER NOT NULL DEFAULT 0");
            database.execSQL("UPDATE user SET isActive = 1 WHERE rowid IN (SELECT rowid FROM user ORDER BY rowid desc LIMIT 1)");
        }
    };
    static final Migration MIGRATION_7_8 = new Migration(7, 8) {
        @Override
        public void migrate(SupportSQLiteDatabase database) {
            database.execSQL("ALTER TABLE user ADD COLUMN customerID TEXT");

            database.execSQL("CREATE UNIQUE INDEX index_customerID ON user (customerID)");
        }
    };

    private static CommonsDatabase INSTANCE;

    public static CommonsDatabase getDatabase(Context context) {
        if (INSTANCE == null) {
            INSTANCE =
                    Room.databaseBuilder(context, CommonsDatabase.class, "commons-flybits-db")
                            .addMigrations(MIGRATION_1_4)
                            .addMigrations(MIGRATION_2_4)
                            .addMigrations(MIGRATION_3_4)
                            .addMigrations(MIGRATION_4_5)
                            .addMigrations(MIGRATION_5_6)
                            .addMigrations(MIGRATION_6_7)
                            .addMigrations(MIGRATION_7_8)
                            .fallbackToDestructiveMigration()
                            .build();
        }
        return INSTANCE;
    }

    public static void destroyInstance() {
        INSTANCE = null;
    }

    public abstract UserDAO userDao();

    public abstract CachingEntryDAO cachingEntryDAO();

    public abstract PreferenceDAO preferenceDAO();

    public abstract CtxDataDAO ctxDataDAO();
}
