package com.flybits.commons.library.caching;

import android.app.Activity;
import android.content.Context;
import android.os.AsyncTask;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.flybits.internal.db.CommonsDatabase;
import com.flybits.internal.db.models.CachingEntry;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;

/**
 * The {@code FlybitsCacheLoader} is responsible for obtaining cache results for a specific entity.
 * This is an important aspect as it will allow application to display content to the user that has
 * been previously saved rather than a loading screen.
 *
 * @param <T> The entity that cached content is being retrieved for.
 */
public abstract class FlybitsCacheLoader<T>{

    private Context context;
    private Activity activity;
    private AsyncTask task;

    /**
     * Default constructor used for defining a {@link FlybitsCacheLoader} object.
     *
     * @param context The Context of the application.
     * @deprecated You should use the more optimized {@link #FlybitsCacheLoader(Activity)}.
     * deprecated in version 1.19.0, will be removed in version 3.0.0
     */
    @Deprecated
    protected FlybitsCacheLoader(@NonNull Context context){
        this.context = context;
    }

    /**
     * Constructor that passes the Activity object where the cache is being loaded for.
     *
     * @param activity The activity that the cache is being loaded for.
     */
    protected FlybitsCacheLoader(@NonNull Activity activity){
        this.activity = activity;
    }

    /**
     * Cancel the AsyncTask if it is running.
     */
    public void cancel(){
        if (task != null && task.getStatus() == AsyncTask.Status.RUNNING){
            task.cancel(true);
        }
    }

    /**
     * Get the {@code Activity} of the application that is using this cache loader.
     *
     * @return The Activity of the application.
     */
    protected Activity getActivity(){
        return activity;
    }

    /**
     * Get the {@code Context} of the application.
     *
     * @return The Context of the application.
     */
    protected Context getContext(){

        if (activity != null){
            return activity.getApplicationContext();
        }
        return context;
    }

    /**
     * Abstract method that is responsible for retrieving cache data from the local DB. This method
     * will be called in a separate thread so no threading needs to take place. This method will
     * always return a list.
     *
     * @param cachedIds The list of cached ids that should be retrieved.
     * @param limit The number of records that should be returned.
     * @return The cached list of items associated to the entity defined in the class generic.
     */
    protected abstract List<T> load(List<String> cachedIds, int limit);

    /**
     * Abstract method that is responsible for retrieving cache data from the local DB. This method
     * will be called in a separate thread so no threading needs to take place. This method will
     * always return a single object.
     *
     * @param id A entity that should be retrieved based on the its id.
     * @return The cached list of items associated to the entity defined in the class generic.
     */
    protected abstract T load(String id);

    /**
     * Retrieves the cached data defined within the {@link #load(List, int)} method and loads the
     * based through the {@code callback} parameter's callback methods.
     *
     * @param cachingKey The caching key which should be used to retrieve a specific entity list.
     * @param callback The {@link CacheListLoader} callback that indicates when cached data is
     *                 loaded and then that loading was cancelled.
     */
    protected void getList(@NonNull final String cachingKey, final CacheListLoader<T> callback){
        if (activity == null) {
            task = new GetCachedDataList(callback).execute(cachingKey);
        }else{
            task = new GetCachedDataList(activity, callback).execute(cachingKey);
        }
    }

    /**
     * Retrieves the cached data defined within the {@link #load(List, int)} method and loads the
     * based through the {@code callback} parameter's callback methods.
     *
     * @param cachingKey The caching key which should be used to retrieve a specific entity list.
     * @param limit The max number of items to return.
     * @param callback The {@link CacheListLoader} callback that indicates when cached data is
     *                 loaded and then that loading was cancelled.
     */
    protected void getList(@NonNull final String cachingKey, final int limit, final CacheListLoader<T> callback){
        if (activity == null) {
            task = new GetCachedDataList(callback, limit).execute(cachingKey);
        }else{
            task = new GetCachedDataList(activity, callback, limit).execute(cachingKey);
        }
    }

    /**
     * Retrieves the cached data defined within the {@link #load(List, int)} method and loads the based
     * through the {@code callback} parameter's callback methods.
     *
     * @param id The id that should be used to retrieve cached data for.
     * @param callback The {@link CacheListLoader} callback that indicates when cached data is
     *                 loaded and then that loading was cancelled.
     */
    protected void getItem(@Nullable final String id, final CacheObjectLoader<T> callback){
        if (activity == null) {
            task = new GetCachedObject(callback).execute(id);
        }else{
            task = new GetCachedObject(activity, callback).execute(id);
        }
    }

    /**
     * Callback used to indicate when a list of cached entries is loaded from the DB.
     *
     * @param <T> The entity that cached content is being retrieved for.
     */
    public interface CacheListLoader<T>{

        /**
         * Indicates the cache retrieval has completed and the list has been retrieved.
         *
         * @param listToLoad The list of loaded data.
         */
        void onLoad(List<T> listToLoad);
    }

    /**
     * Callback used to indicate when a single cached entry is loaded from the DB.
     *
     * @param <T> The entity that cached content is being retrieved for.
     */
    public interface CacheObjectLoader<T>{

        /**
         * Indicates the cache retrieval has completed and the object has been returned
         *
         * @param objectToLoad The object representing the cached item.
         */
        void onLoad(T objectToLoad);

        /**
         * Indicates the cache retrieval has completed and nothing was found.
         */
        void onNotFound();
    }

    private class GetCachedDataList extends AsyncTask<String, Integer, List<T>> {

        private CacheListLoader<T> callback;
        private WeakReference<Activity> mActivityRef;
        private int limit = -1;

        GetCachedDataList(CacheListLoader<T> callback, int limit){
            this(callback);
            this.limit      = limit;
        }

        GetCachedDataList(CacheListLoader<T> callback){
            this.callback   = callback;
        }

        GetCachedDataList(Activity activity, CacheListLoader<T> callback, int limit){
            this(callback, limit);
            this.mActivityRef   = new WeakReference<>(activity);
        }

        GetCachedDataList(Activity activity, CacheListLoader<T> callback){
            this.callback       = callback;
            this.mActivityRef   = new WeakReference<>(activity);
        }

        protected List<T> doInBackground(String... cachingKeys) {
            List<CachingEntry> listOfCacheObjects = CommonsDatabase.getDatabase(getContext()).cachingEntryDAO().getByKey(cachingKeys[0]);
            List<String> ids                      = new ArrayList<>();
            for(CachingEntry entry : listOfCacheObjects){
                ids.add(entry.getContentId());
            }

            if (limit == -1){
                limit = listOfCacheObjects.size();
            }

            return load(ids, limit);
        }

        protected void onPostExecute(List<T> result) {

            if (callback != null){
                if (mActivityRef == null && !isCancelled()) {
                    if (result != null) {
                        callback.onLoad(result);
                    } else {
                        callback.onLoad(new ArrayList<T>());
                    }
                }else if (mActivityRef.get() != null){
                    if (result != null) {
                        callback.onLoad(result);
                    } else {
                        callback.onLoad(new ArrayList<T>());
                    }
                }
            }
        }
    }

    private class GetCachedObject extends AsyncTask<String, Integer, T> {

        private CacheObjectLoader<T> callback;
        private WeakReference<Activity> mActivityRef;

        protected GetCachedObject(CacheObjectLoader<T> callback){
            this.callback   = callback;
        }

        protected GetCachedObject(Activity activity, CacheObjectLoader<T> callback){
            this.mActivityRef   = new WeakReference<>(activity);
            this.callback       = callback;
        }

        protected T doInBackground(String... id) {
            return load(id[0]);
        }

        protected void onPostExecute(T result) {
            if (callback != null){
                if (mActivityRef == null && !isCancelled()) {
                    if (result != null) {
                        callback.onLoad(result);
                    } else {
                        callback.onNotFound();
                    }
                }else if (mActivityRef.get() != null){
                    if (result != null) {
                        callback.onLoad(result);
                    } else {
                        callback.onNotFound();
                    }
                }
            }
        }
    }
}