package com.flybits.commons.library.api;

import android.content.Context;
import android.support.annotation.NonNull;

import com.flybits.commons.library.SharedElementsFactory;
import com.flybits.commons.library.deserializations.IDeserializer;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.exceptions.InvalidFlybitsManagerException;
import com.flybits.commons.library.exceptions.InvalidJsonFormatException;
import com.flybits.commons.library.exceptions.InvalidParameterException;
import com.flybits.commons.library.exceptions.NetworkResponseException;
import com.flybits.commons.library.http.HttpDefaultClass;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.internal.PagedResponse;
import com.flybits.commons.library.models.internal.QueryParameters;
import com.flybits.commons.library.models.internal.Result;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import okhttp3.MediaType;
import okhttp3.RequestBody;

public class FlyAway {

    private static final String TAG_LOGGING = "_Network";

    public static <T> Result<PagedResponse<T>> get(@NonNull Context mContext, @NonNull String urlEndpoint, QueryParameters params,
                                                   @NonNull IDeserializer deserializer, @NonNull String uniqueIdentifier) throws FlybitsException, InvalidFlybitsManagerException {

        return get(mContext, urlEndpoint, null, params, deserializer, uniqueIdentifier);
    }

    public static <T> Result<PagedResponse<T>> get(@NonNull Context context, @NonNull String urlEndpoint, HashMap<String, String> headers,
                                                   QueryParameters params, @NonNull IDeserializer deserializer, @NonNull String uniqueIdentifier) throws FlybitsException, InvalidFlybitsManagerException {

        if (context == null) {
            Logger.appendTag(TAG_LOGGING).e("Context is null");
            throw new InvalidParameterException("context", "Null");
        }
        if (urlEndpoint == null || urlEndpoint.length() == 0) {
            Logger.appendTag(TAG_LOGGING).e("Invalid url");
            throw new InvalidParameterException("urlEndpoint", "Null or Empty");
        }
        if (deserializer == null) {
            Logger.appendTag(TAG_LOGGING).e("IDeserializer is null");
            throw new InvalidParameterException("deserializer", "Null");
        }
        if (uniqueIdentifier == null) {
            Logger.appendTag(TAG_LOGGING).e("uniqueIdentifier is null");
            throw new InvalidParameterException("uniqueIdentifier", "Null");
        }
        if (SharedElementsFactory.INSTANCE.get(context).getGatewayURL().isEmpty()) {
            Logger.appendTag(TAG_LOGGING).e("Gateway url appears to be null");
            throw new InvalidFlybitsManagerException("No Flybits gateway URL set. You must initialize and configure a FlybitsManager prior to calling Flybits APIs.");
        }

        try {

            Result<PagedResponse<T>> result = new HttpDefaultClass.Builder(context, true, urlEndpoint, params)
                    .addHeaders(headers)
                    .get()
                    .getResponse();

            if (result.getStatus() == RequestStatus.COMPLETED) {
                PagedResponse<T> response = (PagedResponse<T>) deserializer.fromJson(result.getResultAsString());
                if (response != null) {
                    result.setResponse(response);
                } else {
                    Logger.appendTag(TAG_LOGGING).e("There appears to be a parsing error with " + result.getResultAsString());
                    result = new Result<>(new InvalidJsonFormatException(result.getResultAsString()), result.getResultAsString());
                }
            } else if (result.getStatus() == RequestStatus.OPTED_OUT) {
                Logger.appendTag(TAG_LOGGING).i("User appears to be opted-out");
                ScopeOfficer.INSTANCE.onOptedStateChange(context,false);
            }

            return result;
        } catch (Exception e) {
            Logger.appendTag(TAG_LOGGING).e(uniqueIdentifier, e);
            if (e instanceof ClassCastException) {
                throw new FlybitsException("Invalid Deserializer for this request: " + deserializer.getClass());
            } else {
                throw new NetworkResponseException();
            }
        }
    }

    public static <T> Result<ArrayList<T>> get(@NonNull Context context, @NonNull String urlEndpoint, HashMap<String, String> headers,
                                               Map<String, String> queryParameters, @NonNull IDeserializer deserializer,
                                               @NonNull String uniqueIdentifier) throws FlybitsException, InvalidFlybitsManagerException {

        if (context == null) {
            Logger.appendTag(TAG_LOGGING).e("Context is null");
            throw new InvalidParameterException("context", "Null");
        }
        if (urlEndpoint == null || urlEndpoint.length() == 0) {
            Logger.appendTag(TAG_LOGGING).e("Invalid url");
            throw new InvalidParameterException("urlEndpoint", "Null or Empty");
        }
        if (deserializer == null) {
            Logger.appendTag(TAG_LOGGING).e("IDeserializer is null");
            throw new InvalidParameterException("deserializer", "Null");
        }
        if (uniqueIdentifier == null) {
            Logger.appendTag(TAG_LOGGING).e("uniqueIdentifier is null");
            throw new InvalidParameterException("uniqueIdentifier", "Null");
        }
        if (SharedElementsFactory.INSTANCE.get(context).getGatewayURL().isEmpty()) {
            Logger.appendTag(TAG_LOGGING).e("Gateway url appears to be null");
            throw new InvalidFlybitsManagerException("No Flybits gateway URL set. You must initialize and configure a FlybitsManager prior to calling Flybits APIs.");
        }

        try {

            Result<ArrayList<T>> result = new HttpDefaultClass.Builder(context, true, urlEndpoint, queryParameters)
                    .addHeaders(headers)
                    .get()
                    .getResponse();

            if (result.getStatus() == RequestStatus.COMPLETED) {
                ArrayList<T> response = (ArrayList<T>) deserializer.fromJson(result.getResultAsString());
                if (response != null) {
                    result.setResponse(response);
                } else {
                    Logger.appendTag(TAG_LOGGING).e("There appears to be a parsing error with " + result.getResultAsString());
                    result = new Result<>(new InvalidJsonFormatException(result.getResultAsString()), result.getResultAsString());
                }
            } else if (result.getStatus() == RequestStatus.OPTED_OUT) {
                Logger.appendTag(TAG_LOGGING).i("User appears to be opted-out");
                ScopeOfficer.INSTANCE.onOptedStateChange(context,false);
            }

            return result;
        } catch (Exception e) {
            Logger.appendTag(TAG_LOGGING).e(uniqueIdentifier, e);
            if (e instanceof ClassCastException) {
                throw new FlybitsException("Invalid Deserializer for this request: " + deserializer.getClass());
            } else {
                throw new NetworkResponseException();
            }
        }
    }

    public static <T> Result<T> get(@NonNull Context context, @NonNull String urlEndpoint,
                                    IDeserializer deserializer, @NonNull String uniqueIdentifier,
                                    Class<T> type) throws FlybitsException, InvalidFlybitsManagerException {

        return get(context, urlEndpoint, null, deserializer, uniqueIdentifier, type);
    }

    public static <T> Result<T> get(@NonNull Context context, @NonNull String urlEndpoint, HashMap<String, String> headers,
                                    IDeserializer deserializer, @NonNull String uniqueIdentifier,
                                    Class<T> type) throws FlybitsException {

        if (context == null) {
            Logger.appendTag(TAG_LOGGING).e("Context is null");
            throw new InvalidParameterException("context", "Null");
        }
        if (urlEndpoint == null || urlEndpoint.length() == 0) {
            Logger.appendTag(TAG_LOGGING).e("Invalid url");
            throw new InvalidParameterException("urlEndpoint", "Null or Empty");
        }
        if (uniqueIdentifier == null) {
            Logger.appendTag(TAG_LOGGING).e("uniqueIdentifier is null");
            throw new InvalidParameterException("uniqueIdentifier", "Null");
        }
        if (SharedElementsFactory.INSTANCE.get(context).getGatewayURL().isEmpty()) {
            Logger.appendTag(TAG_LOGGING).e("Gateway url appears to be null");
            throw new InvalidFlybitsManagerException("No Flybits gateway URL set. You must initialize and configure a FlybitsManager prior to calling Flybits APIs.");
        }
        if ((deserializer == null && type != null) || (deserializer != null && type == null)) {
            Logger.appendTag(TAG_LOGGING).e("IDeserializer or Type is null");
            throw new InvalidParameterException("deserializer or type", "Null");
        }

        try {

            Result<T> result = new HttpDefaultClass.Builder(context, true, urlEndpoint)
                    .addHeaders(headers)
                    .get()
                    .getResponse();
            if (result.getStatus() == RequestStatus.COMPLETED && deserializer != null) {
                T response = type.cast(deserializer.fromJson(result.getResultAsString()));
                if (response != null) {
                    result.setResponse(response);
                } else {
                    Logger.appendTag(TAG_LOGGING).e("There appears to be a parsing error with " + result.getResultAsString());
                    result = new Result<>(new InvalidJsonFormatException(result.getResultAsString()), result.getResultAsString());
                }
            } else if (result.getStatus() == RequestStatus.OPTED_OUT) {
                Logger.appendTag(TAG_LOGGING).i("User appears to be opted-out");
                ScopeOfficer.INSTANCE.onOptedStateChange(context,false);
            }

            return result;
        } catch (Exception e) {
            Logger.appendTag(TAG_LOGGING).e(uniqueIdentifier, e);
            if (e instanceof ClassCastException) {
                throw new FlybitsException("Invalid Deserializer for this request: " + deserializer.getClass());
            } else {
                throw new NetworkResponseException();
            }
        }
    }

    public static <T> Result<T> post(@NonNull Context context, @NonNull String urlEndpoint, String json,
                                     IDeserializer deserializer, @NonNull String uniqueIdentifier, Class<T> type) throws FlybitsException, InvalidFlybitsManagerException {

        if (json == null) {
            json = "";
        }
        RequestBody requestBody = RequestBody.create(MediaType.parse("application/json; charset=utf-8"), json);
        return post(context, urlEndpoint, null, requestBody, deserializer, uniqueIdentifier, type);
    }

    public static <T> Result<T> post(@NonNull Context context, @NonNull String urlEndpoint, String json,
                                     IDeserializer deserializer, @NonNull String uniqueIdentifier, Class<T> type, HashMap<String, String> headers) throws FlybitsException, InvalidFlybitsManagerException {

        if (json == null) {
            json = "";
        }
        RequestBody requestBody = RequestBody.create(MediaType.parse("application/json; charset=utf-8"), json);
        return post(context, urlEndpoint, headers, requestBody, deserializer, uniqueIdentifier, type);
    }

    public static <T> Result<T> post(@NonNull Context context, @NonNull String urlEndpoint, RequestBody requestBody,
                                     IDeserializer deserializer, @NonNull String uniqueIdentifier, Class<T> type) throws FlybitsException, InvalidFlybitsManagerException {

        if (requestBody == null) {
            requestBody = RequestBody.create(MediaType.parse("application/json; charset=utf-8"), "");
        }
        return post(context, urlEndpoint, null, requestBody, deserializer, uniqueIdentifier, type);
    }

    public static <T> Result<T> post(@NonNull Context context, @NonNull String urlEndpoint, HashMap<String, String> headers, RequestBody requestBody,
                                     IDeserializer deserializer, @NonNull String uniqueIdentifier, Class<T> type) throws FlybitsException, InvalidFlybitsManagerException {
        if (context == null) {
            Logger.appendTag(TAG_LOGGING).e("Context is null");
            throw new InvalidParameterException("context", "Null");
        }
        if (urlEndpoint == null || urlEndpoint.length() == 0) {
            Logger.appendTag(TAG_LOGGING).e("Invalid url");
            throw new InvalidParameterException("urlEndpoint", "Null or Empty");
        }
        if (uniqueIdentifier == null) {
            Logger.appendTag(TAG_LOGGING).e("uniqueIdentifier is null");
            throw new InvalidParameterException("uniqueIdentifier", "Null");
        }
        if (SharedElementsFactory.INSTANCE.get(context).getGatewayURL().isEmpty()) {
            Logger.appendTag(TAG_LOGGING).e("Gateway url appears to be null");
            throw new InvalidFlybitsManagerException("No Flybits gateway URL set. You must initialize and configure a FlybitsManager prior to calling Flybits APIs.");
        }
        if ((deserializer == null && type != null) || (deserializer != null && type == null)) {
            Logger.appendTag(TAG_LOGGING).e("IDeserializer or Type is null");
            throw new InvalidParameterException("deserializer or type", "Null");
        }

        try {
            Result<T> result = new HttpDefaultClass.Builder(context, true, urlEndpoint)
                    .addHeaders(headers)
                    .post(requestBody)
                    .getResponse();
            if (result.getStatus() == RequestStatus.COMPLETED && type != null) {
                T response = type.cast(deserializer.fromJson(result.getResultAsString()));
                if (response != null) {
                    result.setResponse(response);
                } else {
                    Logger.appendTag(TAG_LOGGING).e("There appears to be a parsing error with " + result.getResultAsString());
                    result = new Result<>(new InvalidJsonFormatException(result.getResultAsString()), result.getResultAsString());
                }
            } else if (result.getStatus() == RequestStatus.OPTED_OUT) {
                Logger.appendTag(TAG_LOGGING).i("User appears to be opted-out");
                ScopeOfficer.INSTANCE.onOptedStateChange(context,false);
            }

            return result;
        } catch (Exception e) {
            Logger.appendTag(TAG_LOGGING).e(uniqueIdentifier, e);
            if (e instanceof ClassCastException) {
                throw new FlybitsException("Invalid Deserializer for this request: " + deserializer.getClass());
            } else {
                throw new NetworkResponseException();
            }
        }
    }

    public static <T> Result<T> put(@NonNull Context context, @NonNull String urlEndpoint, @NonNull RequestBody requestBody,
                                    IDeserializer deserializer, @NonNull String uniqueIdentifier, Class<T> type) throws FlybitsException, InvalidFlybitsManagerException {
        return put(context, urlEndpoint, null, requestBody, deserializer, uniqueIdentifier, type);
    }

    public static <T> Result<T> put(@NonNull Context context, @NonNull String urlEndpoint, @NonNull String json,
                                    IDeserializer deserializer, @NonNull String uniqueIdentifier, Class<T> type) throws FlybitsException, InvalidFlybitsManagerException {

        RequestBody requestBody = (json == null) ? null : RequestBody.create(MediaType.parse("application/json; charset=utf-8"), json);
        return put(context, urlEndpoint, null, requestBody, deserializer, uniqueIdentifier, type);
    }

    public static <T> Result<T> put(@NonNull Context context, @NonNull String urlEndpoint, @NonNull String json,
                                    IDeserializer deserializer, @NonNull String uniqueIdentifier, Class<T> type, HashMap<String, String> headers) throws FlybitsException, InvalidFlybitsManagerException {

        RequestBody requestBody = (json == null) ? null : RequestBody.create(MediaType.parse("application/json; charset=utf-8"), json);
        return put(context, urlEndpoint, headers, requestBody, deserializer, uniqueIdentifier, type);
    }

    public static <T> Result<T> put(@NonNull Context context, @NonNull String urlEndpoint, HashMap<String, String> headers, @NonNull RequestBody requestBody,
                                    IDeserializer deserializer, @NonNull String uniqueIdentifier, Class<T> type) throws FlybitsException, InvalidFlybitsManagerException {
        if (context == null) {
            Logger.appendTag(TAG_LOGGING).e("Context is null");
            throw new InvalidParameterException("context", "Null");
        }
        if (urlEndpoint == null || urlEndpoint.length() == 0) {
            Logger.appendTag(TAG_LOGGING).e("Invalid url");
            throw new InvalidParameterException("urlEndpoint", "Null or Empty");
        }
        if (uniqueIdentifier == null) {
            Logger.appendTag(TAG_LOGGING).e("uniqueIdentifier is null");
            throw new InvalidParameterException("uniqueIdentifier", "Null");
        }
        if (SharedElementsFactory.INSTANCE.get(context).getGatewayURL().isEmpty()) {
            Logger.appendTag(TAG_LOGGING).e("Gateway url appears to be null");
            throw new InvalidFlybitsManagerException("No Flybits gateway URL set. You must initialize and configure a FlybitsManager prior to calling Flybits APIs.");
        }
        if ((deserializer == null && type != null) || (deserializer != null && type == null)) {
            Logger.appendTag(TAG_LOGGING).e("IDeserializer or Type is null");
            throw new InvalidParameterException("deserializer or type", "Null");
        }
        if (requestBody == null) {
            Logger.appendTag(TAG_LOGGING).e("Request body parameter is null");
            throw new InvalidParameterException("requestBody", "Null");
        }

        try {
            Result<T> result = new HttpDefaultClass.Builder(context, true, urlEndpoint)
                    .addHeaders(headers)
                    .put(requestBody)
                    .getResponse();
            if (result.getStatus() == RequestStatus.COMPLETED && type != null) {
                T response = type.cast(deserializer.fromJson(result.getResultAsString()));
                if (response != null) {
                    result.setResponse(response);
                } else {
                    Logger.appendTag(TAG_LOGGING).e("There appears to be a parsing error with " + result.getResultAsString());
                    result = new Result<>(new InvalidJsonFormatException(result.getResultAsString()), result.getResultAsString());
                }
            } else if (result.getStatus() == RequestStatus.OPTED_OUT) {
                Logger.appendTag(TAG_LOGGING).i("User appears to be opted-out");
                ScopeOfficer.INSTANCE.onOptedStateChange(context,false);
            }

            return result;
        } catch (Exception e) {
            Logger.appendTag(TAG_LOGGING).e(uniqueIdentifier, e);
            if (e instanceof ClassCastException) {
                throw new FlybitsException("Invalid Deserializer for this request: " + deserializer.getClass());
            } else {
                throw new NetworkResponseException();
            }
        }
    }

    public static Result delete(@NonNull Context context, @NonNull String urlEndpoint,
                                @NonNull String uniqueIdentifier, String id) throws FlybitsException, InvalidFlybitsManagerException {
        return delete(context, urlEndpoint, null, uniqueIdentifier, id);
    }


    public static Result delete(@NonNull Context context, @NonNull String urlEndpoint, HashMap<String, String> headers,
                                @NonNull String uniqueIdentifier, String id) throws FlybitsException, InvalidFlybitsManagerException {

        if (context == null) {
            Logger.appendTag(TAG_LOGGING).e("Context is null");
            throw new InvalidParameterException("context", "Null");
        }
        if (urlEndpoint == null || urlEndpoint.length() == 0) {
            Logger.appendTag(TAG_LOGGING).e("Invalid url");
            throw new InvalidParameterException("urlEndpoint", "Null or Empty");
        }
        if (uniqueIdentifier == null) {
            Logger.appendTag(TAG_LOGGING).e("uniqueIdentifier is null");
            throw new InvalidParameterException("uniqueIdentifier", "Null");
        }
        if (SharedElementsFactory.INSTANCE.get(context).getGatewayURL().isEmpty()) {
            Logger.appendTag(TAG_LOGGING).e("Gateway url appears to be null");
            throw new InvalidFlybitsManagerException("No Flybits gateway URL set. You must initialize and configure a FlybitsManager prior to calling Flybits APIs.");
        }

        if (id != null) {
            urlEndpoint += ("/" + id);
        }

        try {
            Result result = new HttpDefaultClass.Builder(context, true, urlEndpoint)
                    .addHeaders(headers)
                    .delete()
                    .getResponse();

            if (result.getStatus() == RequestStatus.OPTED_OUT) {
                Logger.appendTag(TAG_LOGGING).i("User appears to be opted-out");
                ScopeOfficer.INSTANCE.onOptedStateChange(context,false);
            }

            return result;
        } catch (IOException | NullPointerException e) {
            Logger.appendTag(TAG_LOGGING).e(uniqueIdentifier, e);
            throw new NetworkResponseException();
        }
    }
}
