package com.flybits.commons.library.api.idps

import android.os.Parcel
import android.os.Parcelable
import com.flybits.commons.library.exceptions.FlybitsException
import org.json.JSONException
import org.json.JSONObject

/**
 * The [OpenIDConnectIDP] class is responsible for OIDC.
 * OpenID Connect (OIDC) is an authentication layer on top of OAuth 2.0,
 * an authorization framework. The standard is controlled by the OpenID Foundation.
 */
class OpenIDConnectIDP : IDP {

    private val idToken: String

    /**
     * Constructor used for OpenIDConnectIDP IDP.
     *
     * @param idToken The ID Token is a security token that contains Claims about the Authentication
     *                  of an End-User by an Authorization Server.
     */
    constructor(idToken: String) : super("oidc") {
        this.idToken = idToken
    }

    /**
     * Constructor used to read the data
     *
     * @param parcel The parcel is for reading the attribute idToken
     *
     */
    constructor(parcel: Parcel) : super(parcel){
        idToken = parcel.readString()
    }

    override fun getAuthenticationEndPoint() = OIDC_ENDPOINT

    override fun describeContents() = 0

    override fun getBody() = try {
        val bodyObject = JSONObject()
        bodyObject.put("provider", provider)
        bodyObject.put("idToken", idToken)
        bodyObject
    } catch (e: JSONException) {
        throw FlybitsException(e)
    }

    override fun writeToParcel(parcel: Parcel, flags: Int) {
        super.writeToParcel(parcel, flags)
        parcel.writeString(idToken)
    }

    companion object {

        /**
         * Indicates the endpoint for the OpenIDConnect authentication
         */
        internal const val OIDC_ENDPOINT = "/oidc"

        @JvmField
        val CREATOR = object : Parcelable.Creator<OpenIDConnectIDP> {
            override fun createFromParcel(parcel: Parcel) = OpenIDConnectIDP(parcel)
            override fun newArray(size: Int) = arrayOfNulls<OpenIDConnectIDP>(size)
        }
    }
}