package com.flybits.commons.library.exceptions;

import androidx.annotation.NonNull;

/**
 *  The {@code NetworkResponseException} is thrown whenever a network request returns an error from
 *  the server. Along with the error the network request identifier is provided along with a
 *  {@code type} that indicates what caused the error.
 *
 *  @since 1.0.0
 */
public class NetworkResponseException extends FlybitsException{

    private String requestId;
    private String type;

    /**
     * Constructs a new {@code NetworkResponseException} that is thrown whenever a network request
     * returns a response with an error with an unknown type. In most cases this usually, happens
     * when the server responses with unknown error.
     *
     * @param message The {@code message} of the error.
     */
    public NetworkResponseException(@NonNull String message){
        this(message, "Unknown");
    }

    /**
     * Constructs a new {@code NetworkResponseException} that is thrown whenever a network request
     * returns a response with an error.
     *
     * @param message The {@code message} of the error.
     * @param type The error code that represents the network request error.
     */
    public NetworkResponseException(@NonNull String message, @NonNull String type){
        super(message);
        this.type       = type;
    }

    /**
     * Constructs a new {@code NetworkResponseException} that is thrown whenever a network request
     * returns a response with an error.
     *
     * @param message The {@code message} of the error.
     * @param requestId The unique identifier that represents a specific network request.
     * @param type The error code that represents the network request error.
     */
    public NetworkResponseException(@NonNull String message, @NonNull String requestId,
                                    @NonNull String type){
        this(message, type);
        this.requestId  = requestId;
    }

    /**
     * Get the request identifier of the network request that failed.
     *
     * @return The String representation of the identifier for the network request.
     */
    public String getRequestId() {
        return requestId;
    }

    /**
     * Get the error identifier (type) that indicated what the issue is.
     *
     * @return The String representation that indicates what type of error was created.
     */
    public String getType() {
        return type;
    }
}
