package com.flybits.commons.library.models;

import android.arch.persistence.room.ColumnInfo;
import android.arch.persistence.room.Entity;
import android.arch.persistence.room.Ignore;
import android.arch.persistence.room.PrimaryKey;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;

import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.FlybitsManager;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.deserializations.DeserializeLogin;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.internal.db.CommonsDatabase;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * This class represents a {@code User} that can be found within this application environment. A
 * {@code User} is defined by their profile attributes as well an any additional metadata associated
 * to the {@code User} that may be specific to the application.
 */

@Entity(tableName = "user")
public class User implements Parcelable {

    public static final String FIELD_IS_OPTED_IN = "isOptedIn";

    public static final String ME_ENDPOINT  = FlybitsManager.AUTHENTICATION_API+"/me";

    @ColumnInfo(name = "deviceID")
    private String deviceID;

    @ColumnInfo(name = "email")
    private String email;

    @ColumnInfo(name = "firstName")
    private String firstName;

    @PrimaryKey
    @ColumnInfo(name = "id")
    @NonNull
    private String id;

    @ColumnInfo(name = "lastName")
    private String lastName;

    @ColumnInfo(name="isVerified")
    private boolean isVerified;

    @ColumnInfo(name= FIELD_IS_OPTED_IN)
    private boolean isOptedIn;

    /**
     * Default constructor used to define a ROOM DB Entity for the {@code User} object. This
     * constructor should not be used by the application and is meant for internal use only.
     */
    public User (){}

    /**
     * Constructor used for un-flattening a {@code User} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code User} parcel.
     */
    @Ignore
    public User(Parcel in){
        id                  = in.readString();
        firstName           = in.readString();
        lastName            = in.readString();
        email               = in.readString();
        deviceID            = in.readString();
        isVerified          = in.readInt() == 1;
        isOptedIn           = in.readInt() == 1;
    }

    /**
     * Default constructor that creates {@code User} object.
     *
     * @param id The unique identifier of the {@code User}.
     * @param email The email of the {@code User}.
     * @param deviceID The device identifier for the currently logged in {@code User}.
     */
    @Ignore
    public User(@NonNull String id, @NonNull String email, @NonNull String deviceID){
        this.deviceID       = deviceID;
        this.email          = email;
        this.id             = id;
    }

    /**
     * Constructor that creates {@code User} object using a previously created {@code User} object.
     *
     * @param user The {@code User} object used to copy.
     */
    @Ignore
    public User(@NonNull User user){
        this.deviceID       = user.deviceID;
        this.email          = user.email;
        this.id             = user.id;
        this.firstName      = user.firstName;
        this.lastName       = user.lastName;
    }

    /**
     * Get the unique device identifier for the currently logged in {@code User}.
     *
     * @return The unique device identifier.
     */
    public String getDeviceID() {
        return deviceID;
    }

    /**
     * Get the email of the {@code User}.
     *
     * @return The unique email registered for the logged in {@code User}.
     */
    public String getEmail() {
        return email;
    }

    /**
     * Get the first name of the logged in {@code User}.
     *
     * @return The first name of the logged in {@code User}.
     */
    public String getFirstName() {
        return firstName;
    }

    /**
     * Get the unique identifier of the logged in {@code User}.
     *
     * @return The unique identifier of the logged in {@code User}.
     */
    public String getId() {
        return id;
    }

    /**
     * Get the last name of the logged in {@code User}.
     *
     * @return The last name of the logged in {@code User}.
     */
    public String getLastName() {
        return lastName;
    }

    /**
     * Indicates whether or not the user is verified through an email verification process.
     *
     * @return true indicates that a user is verified, false otherwise.
     */
    public boolean isVerified(){
        return isVerified;
    }


    /**
     * Indicates whether or not the user is opted in.
     *
     * @return Whether the user is opted in.
     */
    public boolean isOptedIn() {
        return isOptedIn;
    }

    /**
     * Set the Flybits Device identifier for this instance of the application for this device.
     *
     * @param deviceID The unique identifier of the device.
     */
    public void setDeviceID(String deviceID) {
        this.deviceID = deviceID;
    }

    /**
     * Set the email of the user logged in to the application.
     *
     * @param email The unique email for the user who is logged into the application.
     */
    public void setEmail(String email) {
        this.email = email;
    }

    /**
     * Set the first name of the user.
     *
     * @param firstName The first name of the user who is logged into the application.
     */
    public void setFirstName(String firstName) {
        this.firstName = firstName;
    }

    /**
     * Set the unique identifier for the user who is logged into the application.
     *
     * @param id The unique identifier for the user who is logged into the application.
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * Sets whether or not a user's account is verified through a email verification.
     *
     * @param isVerified indicates whether or not an account is verified.
     */
    public void setIsVerified(boolean isVerified) {
        this.isVerified = isVerified;
    }

    /**
     * Set the last name of the user.
     *
     * @param lastName The last name of the user who is logged into the system.
     */
    public void setLastName(String lastName) {
        this.lastName = lastName;
    }

    /**
     * Set the first and last name of the logged in {@code User}.
     *
     * @param firstName The first name of the logged in {@code User}.
     * @param lastName The last name of the logged in {@code User}.
     */
    public void setName(String firstName, String lastName) {
        this.firstName  = firstName;
        this.lastName   = lastName;
    }

    /**
     * Set the opt in status of the user.
     *
     * @param isOptedIn Whether the user is active.
     */
    public void setOptedIn(boolean isOptedIn) {
        this.isOptedIn = isOptedIn;
    }

    /**
     * Describe the kinds of special objects contained in this Parcelable's marshalled representation.
     *
     * @return a bitmask indicating the set of special object types marshalled by the Parcelable.
     */
    public int describeContents() {
        return 0;
    }

    @Override
    public boolean equals(Object o) {
        if (!(o instanceof User))
            return false;

        User another = (User) o;
        return (another.id.equals(id));
    }

    /**
     * Flatten this {@code User} into a Parcel.
     *
     * @param out The Parcel in which the {@code User} object should be written.
     * @param flags Additional flags about how the {@code User} object should be written.
     * May be 0 or {@link #PARCELABLE_WRITE_RETURN_VALUE}.
     */
    public void writeToParcel(Parcel out, int flags) {
        out.writeString(id);
        out.writeString(firstName);
        out.writeString(lastName);
        out.writeString(email);
        out.writeString(deviceID);
        out.writeInt(isVerified ? 1 : 0);
        out.writeInt(isOptedIn ? 1: 0);
    }

    /**
     * Parcelable.Creator that instantiates {@code User} objects
     */
    public static final Creator<User> CREATOR = new Creator<User>(){
        public User createFromParcel(Parcel in){
            return new User(in);
        }

        public User[] newArray(int size){
            return new User[size];
        }
    };

    /**
     * The {@code getSelf} method is responsible for retrieving user information about the Flybits
     * {@link User}. The {@link User} contains basic personal attributes such as
     * {@link User#firstName} and {@link User#lastName} as well as Flybits information such as
     * {@link User#id} and {@link User#deviceID}.
     * @param context The context of the activity that is calling this method.
     * @return The {@link ObjectResult} is an object that is returned from the Server after a HTTP
     * request is made.
     */
    public static ObjectResult<User> getSelf(final Context context){
        return getSelf(context, null);
    }

    /**
     * The {@code getSelf} method is responsible for retrieving user information about the Flybits
     * {@link User}. The {@link User} contains basic personal attributes such as
     * {@link User#firstName} and {@link User#lastName} as well as Flybits information such as
     * {@link User#id} and {@link User#deviceID}.
     * @param context The context of the activity that is calling this method.
     * @param callback Indicates whether or not the request was successful or it failed with the
     *                 corresponding reason it failed.
     * @param handler The handler that the callback will be invoked through.
     * @return The {@link ObjectResult} is an object that is returned from the Server after a HTTP
     * request is made.
     */
    public static ObjectResult<User> getSelf(final Context context, ObjectResultCallback<User> callback, @NonNull final Handler handler){
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<User> request = new ObjectResult<>(callback, handler, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    final Result<User> getUser = FlyAway.get(context, User.ME_ENDPOINT, new DeserializeLogin(), "FlybitsManager.get", User.class);
                    if (getUser.getStatus() == RequestStatus.COMPLETED) {
                        CommonsDatabase.getDatabase(context).userDao().update(getUser.getResult());
                    }
                    request.setResult(getUser);
                } catch (final FlybitsException e) {
                    request.setFailed(e);
                }
            }
        });
        return request;
    }

    /**
     * The {@code getSelf} method is responsible for retrieving user information about the Flybits
     * {@link User}. The {@link User} contains basic personal attributes such as
     * {@link User#firstName} and {@link User#lastName} as well as Flybits information such as
     * {@link User#id} and {@link User#deviceID}.
     * @param context The context of the activity that is calling this method.
     * @param callback Indicates whether or not the request was successful or it failed with the
     *                 corresponding reason it failed.
     * @return The {@link ObjectResult} is an object that is returned from the Server after a HTTP
     * request is made.
     */
    public static ObjectResult<User> getSelf(final Context context, ObjectResultCallback<User> callback){
        return getSelf(context, callback, new Handler(Looper.getMainLooper()));
    }
}
