package com.flybits.commons.library.api

import kotlinx.coroutines.*

/**
 * Responsible for notifying subscribed [OptInStateChangeListener]s about
 * changed to the opt in state of the [User].
 *
 * Introduced as a temporary solution to [FlyAway] having no reference to [FlybitsScope]s.
 */
internal object OptInStateObservable {

    private val subscribers: ArrayList<OptInStateChangeListener> = arrayListOf()

    /**
     * Subscribe to notifications about the opt in state.
     *
     * @param subscriber [OptInStateChangeListener] subscribing to updates.
     *
     */
    fun subscribe(subscriber: OptInStateChangeListener) {
        subscribers.add(subscriber)
    }

    /**
     * Unsubscribe from notifications about the opt in state.
     *
     * @param subscriber [OptInStateChangeListener] un-subscribing from updates.
     */
    fun unsubscribe(subscriber: OptInStateChangeListener) {
        subscribers.remove(subscriber)
    }

    /**
     * Broadcast change opt-in state change to all subscribers.
     *
     * @param userOptedIn New opt-in state.
     *
     */
    fun broadcastOptedInStateChange(userOptedIn: Boolean) {
        CoroutineScope(Dispatchers.Default).launch {
            subscribers.forEach {
                it.onOptedInStateChange(userOptedIn)
            }
        }
    }

}