package com.flybits.commons.library.api.results

import android.os.Handler
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.http.RequestStatus
import com.flybits.commons.library.models.internal.Result
import java.util.concurrent.ExecutorService
import java.util.concurrent.Executors

/**
 * The [ObjectResult] class is used to query the server when a single object is expected to
 * be returned, a [onSuccess] method or onException with an exception as the reason for the
 * failure will be returned.
 *
 * @param service The [ExecutorService] that is used to run the network request thread.
 * @param handler The handler that allows methods to be run in the UI.
 * @param callback The [ObjectResultCallback] that indicates whether or not the network
 *                 request was successful or not, if successful an object will be returned.
 */
open class ObjectResult<T>(val callback : ObjectResultCallback<T>? = null,
                      val handler : Handler? = null,
                      service: ExecutorService = Executors.newSingleThreadExecutor()) : FlybitsResult(service){

    /**
     * Sets the [FlybitsException] that was thrown when requesting more items to be loaded.
     *
     * @param e The [FlybitsException] that caused the network to fail.
     * @return true if the error has been set, false otherwise
     */
    fun setFailed(e: FlybitsException): Boolean {
        return if (failed()) {
            if (callback != null) {
                handler?.post { callback.onException(e) } ?: callback.onException(e)
            }
            true
        } else false
    }

    /**
     * Sets the result of the network request as successful and returns the expected object. This
     * object can be null.
     *
     * @return true if the request was successful, false otherwise
     */
    fun setSuccess(result : T): Boolean {
        return if (success()) {
            if (callback != null) {
                handler?.post { callback.onSuccess(result) } ?: callback.onSuccess(result)
            }
            true
        } else false
    }

    /**
     * Set the [Result] of the network request.
     *
     * @param result The [Result] of the network request.
     * @return true if the request was successful, false if the request failed
     */
    fun setResult(result: Result<T>) : Boolean {
        return if (result.status == RequestStatus.COMPLETED) {
            setSuccess(result.result)
        } else setFailed(result.exception)
    }
}