package com.flybits.commons.library

import android.content.Context
import android.content.Context.MODE_PRIVATE
import android.content.SharedPreferences
import com.flybits.commons.library.logging.Logger
import com.flybits.internal.db.UserDAO
import devliving.online.securedpreferencestore.EncryptionManager
import devliving.online.securedpreferencestore.SecuredPreferenceStore
import java.io.IOException
import java.lang.IllegalArgumentException
import java.nio.charset.Charset
import java.security.*
import javax.crypto.BadPaddingException
import javax.crypto.IllegalBlockSizeException
import javax.crypto.NoSuchPaddingException

/**
 * Instances of this class will encrypt all key and values prior to storing them in the [SharedPreferences].
 *
 * @param securedPreferenceStore [SecuredPreferenceStore] that will store the data in a secure manner.
 *
 * @param userDAO [UserDAO] for retrieving and storing user data.
 */
class SecuredSharedElements internal constructor(private val securedPreferenceStore: SecuredPreferenceStore, userDAO: UserDAO)
    : SharedElements(securedPreferenceStore, userDAO) {

    /**
     * Migrate data from the V1 version of [SecuredSharedElements] to V2, and V0(unencrypted) to V2.
     * Only needed if you used the SDK version less than 1.8.0.
     *
     * @param context Context associated with the application.
     *
     * @param args Map which holds the ARG_PROJECT_ID argument.
     *
     * @return true if the migration succeeded, false otherwise.
     */
    @Throws(IllegalArgumentException::class)
    override fun performMigration(context: Context, args: Map<String, String>): Int {
        val projectId = args[ARG_PROJECT_ID] ?: throw IllegalArgumentException("Missing project id from argument map. Include ARG_PROJECT_ID.")

        //perform migration from V0
        var migratedValues = 0
        migratedValues += migrateV1(context.getSharedPreferences(FLYBITS_STORAGE_UNENCRYPTED, Context.MODE_PRIVATE))

        //perform migration from V1
        migratedValues += migrateV2(projectId, context)

        return migratedValues
    }

    /**
     * Migrate to version of [SecuredPreferenceStore] that doesn't use project id for key prefix.
     *
     * @param projectId Project id that was associated with the previous [SecuredPreferenceStore]
     *
     * @param context Context associated with the application.
     *
     * @return number of values migrated.
     */
    internal fun migrateV2(projectId: String, context: Context): Int {
        Logger.setTag(TAG).d("migrate() projectId: $projectId")
        val oldPreferences = context.getSharedPreferences(FLYBITS_STORAGE_ENC_V1, MODE_PRIVATE)
        val encryptionManager = EncryptionManager(context, oldPreferences, projectId
                , null, SecuredPreferenceStore.KeyStoreRecoveryNotifier { e, keyStore, keyAliases -> true })
        return migrateV2(encryptionManager, oldPreferences)
    }

    /**
     * Migrate to version of [SecuredPreferenceStore] that doesn't use project id for key prefix.
     *
     * @param encryptionManager [EncryptionManager] tied to the old project id.
     *
     * @param oldPreferences [SharedPreferences] to be migrated from.
     *
     * @return number of values migrated.
     */
    internal fun migrateV2(encryptionManager: EncryptionManager, oldPreferences: SharedPreferences): Int {
        data class Key(val hashed: String, val unhashed: String)

        SecuredPreferenceStore.getSharedInstance().edit().apply {
            var migratedCount = 0
            setOf(PREF_JWT_TOKEN, PREF_LANGUAGE_CODES, PREF_IDP_CONNECTED, PREF_PROJECT_ID).map {
                Key(EncryptionManager.getHashed(it), it)
            }.forEach {
                oldPreferences.getString(it.hashed, null)?.let { encryptedValue ->
                    val decryptedValue = decrypt(encryptedValue, encryptionManager)
                    putString(it.unhashed, decryptedValue)
                    oldPreferences.edit().remove(it.hashed).apply()
                    migratedCount++
                }
            }
            apply()
            Logger.setTag(TAG).d("migrateV2() migrated $migratedCount values to V2.")
            return migratedCount
        }
    }

    private fun decodeEncryptedText(text: String): EncryptionManager.EncryptedData {
        val parts = text.split("]".toRegex()).dropLastWhile{ it.isEmpty() }.toTypedArray()
        var mac: ByteArray? = null
        if (parts.size > 2) {
            mac = EncryptionManager.base64Decode(parts[2])
        }

        return EncryptionManager.EncryptedData(
                EncryptionManager.base64Decode(parts[0]), EncryptionManager.base64Decode(parts[1]), mac)
    }

    @Throws(IOException::class, NoSuchPaddingException::class, InvalidKeyException::class, NoSuchAlgorithmException::class, IllegalBlockSizeException::class, BadPaddingException::class, EncryptionManager.InvalidMacException::class, NoSuchProviderException::class, InvalidAlgorithmParameterException::class, KeyStoreException::class, UnrecoverableEntryException::class)
    private fun decrypt(text: String?, encryptionManager: EncryptionManager): String? {
        if (text != null && text.length > 0) {
            val encryptedData = decodeEncryptedText(text)
            val decrypted = encryptionManager.tryDecrypt(encryptedData)

            return String(decrypted, 0, decrypted.size, Charset.defaultCharset())
        }

        return null
    }

    /**
     * Migrate any data stored in the unencrypted preferences to the encrypted preferences and
     * clear the unencrypted version.
     *
     * @param oldUnencryptedPreferences Preferences stored the data that is to be migrated.
     *
     * @return number of values migrated successfully.
     */
    internal fun migrateV1(oldUnencryptedPreferences: SharedPreferences): Int {
        Logger.setTag(TAG).d("migrateV1()")
        var migratedValues = 0
        if (oldUnencryptedPreferences.contains(PREF_IDP_CONNECTED)){
            setStringVariable(PREF_IDP_CONNECTED, oldUnencryptedPreferences.getString(PREF_IDP_CONNECTED, ""))
            migratedValues++
        }
        if (oldUnencryptedPreferences.contains(PREF_PROJECT_ID)){
            setStringVariable(PREF_PROJECT_ID, oldUnencryptedPreferences.getString(PREF_PROJECT_ID, ""))
            migratedValues++
        }
        if (oldUnencryptedPreferences.contains(PREF_LANGUAGE_CODES)){
            setStringVariable(PREF_LANGUAGE_CODES, oldUnencryptedPreferences.getString(PREF_LANGUAGE_CODES, ""))
            migratedValues++
        }
        if (oldUnencryptedPreferences.contains(PREF_JWT_TOKEN)){
            setStringVariable(PREF_JWT_TOKEN, oldUnencryptedPreferences.getString(PREF_JWT_TOKEN, ""))
            migratedValues++
        }
        oldUnencryptedPreferences.edit().clear().apply()
        return migratedValues
    }

}