package com.flybits.commons.library

import android.content.Context
import android.content.SharedPreferences

import com.flybits.commons.library.api.FlybitsManager
import com.flybits.commons.library.utils.Utilities
import com.flybits.internal.db.UserDAO

import java.util.*


/**
 * This class is used to get and set all variables that can be shared across the various Flybits'
 * Android SDKs. Currently the items that can be shared between the SDKs are as follows;
 *
 *  * Language Information
 *  * Device Identifier
 *  * User Identifier
 *  * JWT
 *  * Project ID
 *  * Connected IDP
 *
 *  @param sharedPreferences [SharedPreferences] that data will be stored and retrieved from.
 *  @param userDAO [UserDAO] that user data will be retrieved from.
 *
 */
abstract class SharedElements internal constructor(private val sharedPreferences: SharedPreferences, private val userDAO: UserDAO) {

    companion object {
        internal val TAG = SharedElements::class.java.simpleName

        const val PREF_LANGUAGE_CODES = "com.flybits.language.codes"
        const val PREF_JWT_TOKEN = "com.flybits.jwt.token"
        const val PREF_IDP_CONNECTED = "com.flybits.idp.connected"
        const val PREF_PROJECT_ID = "com.flybits.project.id"
        const val PREF_GATEWAY_URL = "com.flybits.project.url"

        const val FLYBITS_STORAGE_UNENCRYPTED = "FLYBITS_PREF"
        const val FLYBITS_STORAGE_ENC_V1 = "flybits_con_storage"
        const val FLYBITS_STORAGE_ENC_V2 = "flybtis_secure_storage_v2"
        const val ARG_PROJECT_ID = "flybits_arg_project_id"
    }

    /**
     * Migrate data from previous versions of [SharedElements] to the latest version.
     *
     * @param context Context associated with the Application.
     * @param args Arguments required for migration.
     *
     * @return how many values were migrated successfully.
     */
    abstract fun migrateData(context:Context, args: Map<String, String>): Int

    /**
     * Get the IDP that the application used to connect to Flybits with. If the application is not
     * connected to Flybits then "" empty string will be returned.
     *
     * @return The saved string representation of the IDP used to connect to Flybits with, or "" if
     * the user is not connected to Flybits.
     */
    fun getConnectedIDP() = getStringVariable(PREF_IDP_CONNECTED, "")

    /**
     * Get the previously saved [com.flybits.commons.library.models.User.deviceID].
     *
     * @return The saved [com.flybits.commons.library.models.User.deviceID] or "" if no
     * [com.flybits.commons.library.models.User.deviceID] is saved.
     */
    fun getDeviceID(): String {
        val user = userDAO.single
        return if (user != null) {
            user.deviceID
        } else ""
    }

    /**
     * Get the ArrayList representation of the language codes set for this application.
     *
     * @return The ArrayList representation of the language codes set for this application.
     */
    fun getEnabledLanguagesAsArray(): ArrayList<String> {
        val languageCodes = getStringVariable(PREF_LANGUAGE_CODES, "")
        return Utilities.convertLocalizationStringToList(languageCodes)
    }

    /**
     * Get the String representation of the language codes set for this application.
     *
     * @return The String representation of the language codes set for this application or "" if no
     * language is set.
     */
    fun getEnabledLanguagesAsString() = getStringVariable(PREF_LANGUAGE_CODES, "")

    /**
     * Get the unique Flybits Project Identifier associated to your project. This project is set
     * through the [com.flybits.commons.library.api.idps.IDP.getProjectID].
     *
     * @return The saved unique Project identifier which can be retrieved from the
     * [Developer Portal](https://developer.flybits.com)  or "" if no project id.
     */
    fun getProjectID() = getStringVariable(PREF_PROJECT_ID, "")

    /**
     * Get the previously saved `JWT` which is obtained when calling
     * [FlybitsManager.connect] the first time.
     *
     * @return The saved `JWT` which is obtained once the application logs into Flybits  or ""
     * if the application has not successfully received a `JWT` token.
     */
    fun getSavedJWTToken() = getStringVariable(PREF_JWT_TOKEN, "")

    /**
     * Get the gateway URL to be used for communicating with the Flybits servers.
     */
    fun getGatewayURL() = getStringVariable(PREF_GATEWAY_URL, "")

    /**
     * Get the previously saved [com.flybits.commons.library.models.User.id].
     *
     * @return The saved [com.flybits.commons.library.models.User.id] or "" if no
     * [com.flybits.commons.library.models.User.id] is saved.
     */
    fun getUserID(): String {
        val user = userDAO.single
        return if (user != null) {
            user.id
        } else ""
    }

    /**
     * Clear all data present in the SharedElements.
     */
    fun clear() {
        sharedPreferences.edit().clear().apply()
    }

    /**
     * Sets the gateway URL to be used for communicating with the Flybits servers.
     *
     * @param url Gateway URL to be set.
     */
    fun setGatewayURL(url: String) {
        setStringVariable(PREF_GATEWAY_URL, url)
    }

    /**
     * Sets the IDP that was used to connect to Flybits with such as Flybits, Anonymous, OAuth, etc.
     *
     * @param idp The string representation of the IDP used to connect to Flybits with.
     */
    fun setConnectedIDP(idp: String) {
        setStringVariable(PREF_IDP_CONNECTED, idp)
    }

    /**
     * Sets the unique JWT Token obtained from the Flybits Core for the user/device combination.
     *
     * @param jwtToken The unique JWT Token that is used by Flybits to identify a user/device
     * combination.
     */
    fun setJWTToken(jwtToken: String) {
        setStringVariable(PREF_JWT_TOKEN, jwtToken)
    }

    /**
     * Sets the localization values of the device.
     *
     * @param listOfLanguages The array of languages that should be used for this device.
     */
    fun setLocalization(listOfLanguages: ArrayList<String>) {
        setStringVariable(PREF_LANGUAGE_CODES, Utilities.convertLocalizationCodeToString(listOfLanguages))
    }

    /**
     * Sets the Flybits Project Identifier which can be used by other components/sdks within the
     * Flybits ecosystem.
     *
     * @param projectID The unique Flybits Project Identifier that represents this project's
     * application.
     */
    @Deprecated("Use [FlybitsManager.Builder.setProjectId(String)] instead as it is a more\n" +
            "      optimized.", ReplaceWith("[FlybitsManager.Builder.setProjectId(String)]"))
    fun setProjectID(projectID: String) {
        setStringVariable(PREF_PROJECT_ID, projectID)
    }

    protected fun setStringVariable(key: String, value: String) {
        val editor = sharedPreferences.edit()
        editor.putString(key, value)
        editor.apply()
    }

    private fun getStringVariable(key: String, default: String)
            =  sharedPreferences.getString(key,default)

}
