package com.flybits.commons.library

import android.content.Context
import android.content.Context.MODE_PRIVATE
import android.content.SharedPreferences

import com.flybits.commons.library.api.FlybitsManager
import com.flybits.commons.library.logging.Displayer
import com.flybits.commons.library.logging.Logger
import com.flybits.commons.library.utils.Utilities
import com.flybits.internal.db.CommonsDatabase
import com.flybits.internal.db.UserDAO
import devliving.online.securedpreferencestore.DefaultRecoveryHandler

import devliving.online.securedpreferencestore.SecuredPreferenceStore
import java.lang.Exception
import java.util.*


/**
 * This class is used to get and set all variables that can be shared across the various Flybits'
 * Android SDKs. Currently the items that can be shared between the SDKs are as follows;
 *
 *  * Language Information
 *  * Device Identifier
 *  * User Identifier
 *  * JWT
 *  * Project ID
 *  * Connected IDP
 *
 *  @param sharedPreferences [SharedPreferences] that data will be stored and retrieved from.
 *  @param userDAO [UserDAO] that user data will be retrieved from.
 *
 */
abstract class SharedElements internal constructor(private val sharedPreferences: SharedPreferences, private val userDAO: UserDAO) {

    companion object {
        internal val TAG = SharedElements::class.java.simpleName

        const val PREF_LANGUAGE_CODES = "com.flybits.language.codes"
        const val PREF_JWT_TOKEN = "com.flybits.jwt.token"
        const val PREF_IDP_CONNECTED = "com.flybits.idp.connected"
        const val PREF_PROJECT_ID = "com.flybits.project.id"

        const val FLYBITS_STORAGE_UNENCRYPTED = "FLYBITS_PREF"
        const val FLYBITS_STORAGE_ENC_V1 = "flybits_con_storage"
        const val FLYBITS_STORAGE_ENC_V2 = "flybtis_secure_storage_v2"

        @Volatile
        internal var INSTANCE: SharedElements? = null

        /**
         * Get singleton instance of [SharedElements].
         *
         * @param context Context of the application.
         *
         * @return [SecuredSharedElements] if encryption is supported. [UnsecuredSharedElements] otherwise.
         */
        @JvmStatic
        fun get(context: Context): SharedElements {
            return INSTANCE ?: synchronized(this) {
                val userDAO = CommonsDatabase.getDatabase(context).userDao()
                INSTANCE ?: try {
                    Logger.setTag(TAG).d("get(): Creating first instance of SharedElements")
                    SecuredPreferenceStore.init(context, FLYBITS_STORAGE_ENC_V2, null
                            , null, DefaultRecoveryHandler())
                    SecuredSharedElements(SecuredPreferenceStore.getSharedInstance(), userDAO)
                            .also { INSTANCE = it }
                } catch (e: Exception) {
                    Logger.exception("SharedElements.get()", e)
                    UnsecuredSharedElements(context.getSharedPreferences(
                            FLYBITS_STORAGE_UNENCRYPTED, MODE_PRIVATE), userDAO).also { INSTANCE = it }
                }
            }
        }

        /**
         * Migrate data from the V1 version of [SharedElements] to V2, and V0(unencrypted) to V2.
         * Only needed if you used the SDK version less than 1.8.0.
         *
         * @param context Context associated with the application.
         *
         * @param projectId Project id associated with the SDK prior to V1.8.0 of this SDK
         * must be provided to complete the migration.
         *
         * @return true if the migration succeeded, false otherwise.
         */
        @JvmStatic
        fun migrate(context: Context, projectId: String): Boolean {
            Logger.setTag(TAG).d("migrate() projectId: $projectId")
            synchronized (this) {
                return try {
                    SecuredPreferenceStore.init(context, FLYBITS_STORAGE_ENC_V2, null
                            , null, DefaultRecoveryHandler())
                    val sharedElements = SharedElements.get(context)
                    if (sharedElements is SecuredSharedElements) {
                        sharedElements.migrateV1(context.getSharedPreferences(FLYBITS_STORAGE_UNENCRYPTED, Context.MODE_PRIVATE))
                        sharedElements.migrateV2(projectId, context)
                        true
                    } else {
                        false
                    }
                } catch (e: Exception) {
                    Logger.exception("SharedElements.migrate", e)
                    false
                }
            }
        }
    }

    /**
     * Get the IDP that the application used to connect to Flybits with. If the application is not
     * connected to Flybits then "" empty string will be returned.
     *
     * @return The saved string representation of the IDP used to connect to Flybits with, or "" if
     * the user is not connected to Flybits.
     */
    fun getConnectedIDP() = getStringVariable(PREF_IDP_CONNECTED, "")

    /**
     * Get the previously saved [com.flybits.commons.library.models.User.deviceID].
     *
     * @return The saved [com.flybits.commons.library.models.User.deviceID] or "" if no
     * [com.flybits.commons.library.models.User.deviceID] is saved.
     */
    fun getDeviceID(): String {
        val user = userDAO.single
        return if (user != null) {
            user.deviceID
        } else ""
    }

    /**
     * Get the ArrayList representation of the language codes set for this application.
     *
     * @return The ArrayList representation of the language codes set for this application.
     */
    fun getEnabledLanguagesAsArray(): ArrayList<String> {
        val languageCodes = getStringVariable(PREF_LANGUAGE_CODES, "")
        return Utilities.convertLocalizationStringToList(languageCodes)
    }

    /**
     * Get the String representation of the language codes set for this application.
     *
     * @return The String representation of the language codes set for this application or "" if no
     * language is set.
     */
    fun getEnabledLanguagesAsString() = getStringVariable(PREF_LANGUAGE_CODES, "")

    /**
     * Get the unique Flybits Project Identifier associated to your project. This project is set
     * through the [com.flybits.commons.library.api.idps.IDP.getProjectID].
     *
     * @return The saved unique Project identifier which can be retrieved from the
     * [Developer Portal](https://developer.flybits.com)  or "" if no project id.
     */
    fun getProjectID() = getStringVariable(PREF_PROJECT_ID, "")

    /**
     * Get the previously saved `JWT` which is obtained when calling
     * [FlybitsManager.connect] the first time.
     *
     * @return The saved `JWT` which is obtained once the application logs into Flybits  or ""
     * if the application has not successfully received a `JWT` token.
     */
    fun getSavedJWTToken() = getStringVariable(PREF_JWT_TOKEN, "")

    /**
     * Get the previously saved [com.flybits.commons.library.models.User.id].
     *
     * @return The saved [com.flybits.commons.library.models.User.id] or "" if no
     * [com.flybits.commons.library.models.User.id] is saved.
     */
    fun getUserID(): String {
        val user = userDAO.single
        return if (user != null) {
            user.id
        } else ""
    }

    /**
     * Clear all data present in the SharedElements.
     */
    fun clear() {
        sharedPreferences.edit().clear().apply()
    }

    /**
     * Sets the IDP that was used to connect to Flybits with such as Flybits, Anonymous, OAuth, etc.
     *
     * @param idp The string representation of the IDP used to connect to Flybits with.
     */
    fun setConnectedIDP(idp: String) {
        setStringVariable(PREF_IDP_CONNECTED, idp)
    }

    /**
     * Sets the unique JWT Token obtained from the Flybits Core for the user/device combination.
     *
     * @param jwtToken The unique JWT Token that is used by Flybits to identify a user/device
     * combination.
     */
    fun setJWTToken(jwtToken: String) {
        setStringVariable(PREF_JWT_TOKEN, jwtToken)
    }

    /**
     * Sets the localization values of the device.
     *
     * @param listOfLanguages The array of languages that should be used for this device.
     */
    fun setLocalization(listOfLanguages: ArrayList<String>) {
        setStringVariable(PREF_LANGUAGE_CODES, Utilities.convertLocalizationCodeToString(listOfLanguages))
    }

    /**
     * Sets the Flybits Project Identifier which can be used by other components/sdks within the
     * Flybits ecosystem.
     *
     * @param projectID The unique Flybits Project Identifier that represents this project's
     * application.
     */
    @Deprecated("Use {@link FlybitsManager.Builder#setProjectId(String)} instead as it is a more\n" +
            "      optimized.", ReplaceWith("FlybitsManager.Builder(String)"))
    fun setProjectID(projectID: String) {
        setStringVariable(PREF_PROJECT_ID, projectID)
    }

    protected fun setStringVariable(key: String, value: String) {
        val editor = sharedPreferences.edit()
        editor.putString(key, value)
        editor.apply()
    }

    private fun getStringVariable(key: String, default: String)
            =  sharedPreferences.getString(key,default)

}
