package com.flybits.commons.library.api.results

import android.os.Handler
import com.flybits.commons.library.api.results.callbacks.ConnectionResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.http.RequestStatus
import com.flybits.commons.library.models.internal.Result
import java.util.concurrent.ExecutorService
import java.util.concurrent.Executors

/**
 * The {@code ConnectionResult} class is used to query the server when trying to determine whether
 * or not a user is currently connected to Flybits. The response will be one of the options found
 * within the {@link ConnectionResultCallback}.
 *
 * @param callback The {@link ObjectResultCallback} that indicates whether or not the network
 *                 request was successful or not, if successful an object will be returned.*
 * @param handler The handler that allows methods to be run in the UI.
 * @param service The {@link ExecutorService} that is used to run the network request thread.
 */
open class ConnectionResult(val callback : ConnectionResultCallback? = null,
                            val handler : Handler? = null,
                            service: ExecutorService = Executors.newSingleThreadExecutor()) : FlybitsResultV2(service){

    /**
     * Sets the {@link FlybitsException} that was thrown when requesting more items to be loaded.
     *
     * @param e The {@link FlybitsException} that caused the network to fail.
     * @return true if the error has been set, false otherwise
     */
    internal fun setFailed(e: FlybitsException): Boolean {
        return if (failed()) {
            if (callback != null) {
                handler?.post { callback.onException(e) } ?: callback.onException(e)
            }
            true
        } else false
    }

    /**
     * Sets the result of the network request as successful and the application is connected to
     * Flybits.
     *
     * @return true if the request was successful, false otherwise
     */
    internal fun setConnected() : Boolean {
        return if (success()) {
            if (callback != null) {
                handler?.post { callback.onConnected() } ?: callback.onConnected()
            }
            true
        } else false
    }

    /**
     * Sets the result of the network request as successful however the application is not currently
     * connected to Flybits.
     *
     * @return true if the request was successful, false otherwise
     */
    internal fun setNotConnected() : Boolean {
        return if (success()) {
            if (callback != null) {
                handler?.post { callback.notConnected() } ?: callback.notConnected()
            }
            true
        } else false
    }

    /**
     * Set the {@link Result} of the network request.
     *
     * @param result The {@link Result} of the network request.
     */
    fun setResult(result: Result<*>) {

        if (result.status == RequestStatus.COMPLETED) {
            setConnected()
        } else if (result.status == RequestStatus.NOT_CONNECTED){
            setNotConnected()
        } else setFailed(result.exception)
    }
}